
/*************************************************************************************************************** 
 *   Senet is the leading IoT network services provider for low cost, long range IoT applications.             *
 *   Senet provides network services for application developers who are searching for an affordable IoT        *
 *   network to launch and scale their solutions.                                                              *
 *                                                                                                             *
 *   This program demonstrates how to connect to Senet's public network with an mDot and sends some data       *
 *                                                                                                             *
 *   On the Senet Developer Portal website you can view device transactions and even configure forwarding      *
 *   to IoT data collection and analysis services.                                                             * 
 *                                                                                                             *
 *   Get started by registering for a Senet Developer Portal account at http://portal.senetco.com              *
 ****************************************************************************************************************/

#include "mbed.h"
#include "mDot.h"
#include "MTSLog.h"
#include <string>
#include <vector>
#include <algorithm>

// Senet Developer Portal Application EUI
static uint8_t senetDevPortalAppEUI[8]  = {0x00,0x25,0x0C,0x00,0x00,0x01,0x00,0x01};

// Your Device's application key provided by Senet on the Device setup page
static uint8_t senetDevPortalAppKey[16] =  {0xCA,0x40,0x36,0xB8,0xB2,0xB3,0x9F,0x4F,0xAB,0x84,0xD7,0xB3,0x65,0x8C,0x80,0xD3};

// Helper macro to initialize std::vector from array
#define INIT_FROM_ARRAY(ar) ar, ar + sizeof(ar) / sizeof(ar[0])

// Uncomment this line if using a full sized UDK2.0 instead of a Micro UDK
#define UDK2 1
#ifdef UDK2
DigitalOut led(LED1);
DigitalOut tx_led(PA_1);
#else
DigitalOut led(XBEE_RSSI);
#endif

// Led blink ticker
Ticker tick;

// LED states
#define OFF 1
#define ON  0

// Ticker callback function to change LED state
void blink() {
    led = !led;
}
 

int main() {
    int32_t ret;
    mDot* dot;
    std::vector<uint8_t> data;
    std::string data_str = "hello!";
    std::vector<uint8_t> appEUI(INIT_FROM_ARRAY(senetDevPortalAppEUI));
    std::vector<uint8_t> appKey(INIT_FROM_ARRAY(senetDevPortalAppKey));
    
    // configure the Ticker to blink the LED on 500ms interval while joining network
    led    = OFF;
    tx_led = OFF;
    tick.attach(&blink, 0.5);
    
    // get a mDot handle
    dot = mDot::getInstance();
    
    //*******************************************
    // configuration
    //*******************************************
    // reset to default config so we know what state we're in
    dot->resetConfig();
    
    // Turn on mDot logging - output is on the debug UART at 9600 baud
    dot->setLogLevel(mts::MTSLog::INFO_LEVEL);
    
    // set up the mDot with our network information: frequency sub band, application, application encryption key
    // these can all be saved in NVM so they don't need to be set every time - see mDot::saveConfig()
    
    // Senet is a Public Network
    if ((ret = dot->setPublicNetwork(true))!= mDot::MDOT_OK) {
        logError("failed to set public network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
       
    // frequency sub band is only applicable in the 915 (US) frequency band
    // if using a gateway that only supports the first 8 channels, use sub band 1 (channels 1 - 8)
    // if using a gateway that supports all 64 channels, use sub band 0 - the mDot will use all 64 channels
    static uint8_t config_frequency_sub_band = 1;
    logInfo("setting frequency sub band");
    if ((ret = dot->setFrequencySubBand(config_frequency_sub_band)) != mDot::MDOT_OK) {
        logError("failed to set frequency sub band %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // Set Application Id to Senet Developer Portal 
    logInfo("setting application EUI");
    if ((ret = dot->setNetworkId(appEUI)) != mDot::MDOT_OK) {
        logError("failed to set application EUI %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // Set AES-128 Application secret key
    logInfo("setting application key");
    if ((ret = dot->setNetworkKey(appKey)) != mDot::MDOT_OK) {
        logError("failed to set network key %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // a higher spreading factor allows for longer range but lower throughput
    // in the 915 (US) frequency band, spreading factors 7 - 10 are available
    logInfo("setting TX spreading factor");
    if ((ret = dot->setTxDataRate(mDot::SF_10)) != mDot::MDOT_OK) {
        logError("failed to set TX datarate %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // Enable Adaptive Data Rate Control so that the Network server can optimize our data rate and
    // transmit power for the network
    if((ret = dot->setAdr(true)) != mDot::MDOT_OK){
        logError("failed to set ADR %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
     }
    // Set Join byte order to LSB   
    if((ret = dot->setJoinByteOrder(mDot::LSB)) != mDot::MDOT_OK){
        logError("failed to set join LSB byte order %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
        
    // Set Over The Air Join mode   
    if((ret = dot->setJoinMode(mDot::OTA)) != mDot::MDOT_OK){
        logError("failed to set join OTA mode %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
    }
    
    // save this configuration to the mDot's NVM
    logInfo("saving config");
    if (! dot->saveConfig()) {
        logError("failed to save configuration");
    }
    //*******************************************
    // end of configuration
    //*******************************************

    // attempt to join the network
    logInfo("joining network");
    while ((ret = dot->joinNetwork()) != mDot::MDOT_OK) {
        logError("failed to join network %d:%s", ret, mDot::getReturnCodeString(ret).c_str());
        osDelay(5000);
    }
    
    // Detach ticker led callback
    tick.detach();
    
    // Leave led ON
    led = ON;

    // format data to send
    for (std::string::iterator it = data_str.begin(); it != data_str.end(); it++)
        data.push_back((uint8_t) *it);

    while (true) {
        // send data
        tx_led = ON; 
        if ((ret = dot->send(data)) != mDot::MDOT_OK) {
            logError("failed to send", ret, mDot::getReturnCodeString(ret).c_str());
        } else {
            logInfo("successfully sent data on Senet network");
        }
        tx_led = OFF;

        // Wait before transmitting again
        osDelay(60000);
    }
}
