/***
 *       _____                         _   
 *      / ____|                       | |  
 *     | (___     ___   _ __     ___  | |_ 
 *      \___ \   / _ \ | '_ \   / _ \ | __|
 *      ____) | |  __/ | | | | |  __/ | |_ 
 *     |_____/   \___| |_| |_|  \___|  \__|
 *         (C) 2016 Senet, Inc                                
 *                                         
 */

#ifdef MTDOT_EVB
#include "board_evb.h"
#include "MMA845x.h"
#include "MPL3115A2.h"
#include "ISL29011.h"
#include "NCP5623B.h"
#include "DOGS102.h"
#include "font_6x8.h"
#include "MultiTech_Logo.h"

Serial debugUART(PA_9, PA_10); // mDot debug UART

static InterruptIn mDot08(PA_12);  //  GPIO/USB       PB S1 on EVB
static InterruptIn mDot09(PA_11);  //  GPIO/USB       PB S2 on EVB
static InterruptIn mDot12(PA_0);    //  GPIO/UART_CTS  PRESSURE_INT2 on EVB
static DigitalOut  mDot13(PC_13,1); //  GPIO           LCD_C/D
static InterruptIn mDot15(PC_1);    //  GPIO           LIGHT_PROX_INT on EVB
static InterruptIn mDot16(PA_1);    //  GPIO/UART_RTS  ACCEL_INT2 on EVB
static DigitalOut mDot17(PA_4,1);   //  GPIO/SPI_NCS   LCD_CS on EVB
static AnalogIn mDot20(PB_1);         //  GPIO          Current Sense Analog in on EVB
static I2C mDoti2c(PC_9,PA_8);	       // mDot External I2C mDot6 and mDot7
static SPI mDotspi(PA_7,PA_6,PA_5);   // mDot external SPI mDot11, mDot4, and mDot18
static MMA845x_DATA   accel_data   = 0;
static MPL3115A2_DATA baro_data    = 0;
static MMA845x*       evbAccel     = 0;
static MPL3115A2*     evbBaro      = 0;
static ISL29011*      evbAmbLight  = 0;
static NCP5623B*      evbBackLight = 0;
static DOGS102*       evbLCD       = 0;
static Thread         thread_1;
static Thread         thread_2;
static char           txtstr[17];
// flags for pushbutton debounce code
static bool pb1_low     = false;
static bool pb2_low     = false;
static bool evbLedState = false;

CBoardEVB::CBoardEVB()
{
	boardPtr = this;
}


EBoardStatus CBoardEVB::init()
{
	CBoard::init();

	// Setting up LED1 as activity LED
	mDotPtr->setActivityLedPin(PB_0);
	mDotPtr->setActivityLedEnable(true);

    // threads for de-bouncing pushbutton switches
	thread_1.start(callback(this, &CBoardEVB::pb1_debounce));
    thread_2.start(callback(this, &CBoardEVB::pb2_debounce));

    evbAccel     = new MMA845x(mDoti2c,MMA845x::SA0_VSS); // setup Accelerometer
    evbBaro      = new MPL3115A2(mDoti2c); // setup Barometric sensor
    evbAmbLight  = new ISL29011(mDoti2c); // Setup Ambient Light Sensor
    evbBackLight = new NCP5623B(mDoti2c); // setup backlight and LED 2 driver chip
    evbLCD       = new DOGS102(mDotspi, mDot17, mDot13); // setup LCD

    /*
     *  Setup SW1 as program stop function
     */
    mDot08.disable_irq();
    mDot08.fall(callback(this, &CBoardEVB::pb1ISR));

    /*
     *  need to call this function after rise or fall because rise/fall sets
     *  mode to PullNone
     */
    mDot08.mode(PullUp);

    mDot08.enable_irq();

    /*
     *  Setup SW2 as packet time change
     */
    mDot09.disable_irq();
    mDot09.fall(callback(this, &CBoardEVB::pb2ISR));

    /*
     *  need to call this function after rise or fall because rise/fall sets
     *  mode to PullNone
     */
    mDot09.mode(PullUp);

    mDot09.enable_irq();

    /*
    * Setting other InterruptIn pins with Pull Ups
    */
    mDot12.mode(PullUp);
    mDot15.mode(PullUp);
    mDot16.mode(PullUp);

    printf("font table address %p\n\r",&font_6x8);
    printf("bitmap address %p\n\r",&MultiTech_Logo);

    // Setup and display logo on LCD
    evbLCD->startUpdate();

    evbLCD->writeBitmap(0,0,MultiTech_Logo);

    sprintf(txtstr,"MTDOT");
    evbLCD->writeText(24,3,font_6x8,txtstr,strlen(txtstr));
    sprintf(txtstr,"Evaluation");
    evbLCD->writeText(24,4,font_6x8,txtstr,strlen(txtstr));
    sprintf(txtstr,"Board");
    evbLCD->writeText(24,5,font_6x8,txtstr,strlen(txtstr));

    evbLCD->endUpdate();

    return Board_Ok;
}

EBoardStatus CBoardEVB::start()
{
	osDelay(200);
	evbBackLight->setPWM(NCP5623B::LED_3,16); // enable LED2 on EVB and set to 50% PWM

	// sets LED2 to 50% max current
	evbBackLight->setLEDCurrent(16);

	printf("Start of Test\n\r");

	osDelay (500); // allows other threads to process
	printf("shutdown LED:\n\r");
	evbBackLight->shutdown();

	osDelay (500); // allows other threads to process
	printf("Turn on LED2\n\r");
	evbBackLight->setLEDCurrent(16);

	char data = evbAccel->getWhoAmI();
	printf("Accelerometer who_am_i value = %x \n\r", data);

	uint8_t result = evbAccel->getStatus();
	printf("status byte = %x \n\r", result);

	printf("Barometer who_am_i check = %s \n\r", evbBaro->testWhoAmI() ? "TRUE" : "FALSE");

	result = evbBaro->getStatus();
	printf("status byte = %x \n\r", result);

	/*
	*  Setup the Accelerometer for 8g range, 14 bit resolution, Noise reduction off, sample rate 1.56 Hz
	*  normal oversample mode, High pass filter off
	*/
	evbAccel->setCommonParameters(MMA845x::RANGE_8g,MMA845x::RES_MAX,MMA845x::LN_OFF,
			MMA845x::DR_1_56,MMA845x::OS_NORMAL,MMA845x::HPF_OFF );

	/*
	 * Setup the Barometric sensor for post processed Ambient pressure, 4 samples per data acquisition.
	 * and a sample taken every second when in active mode
	 */
	evbBaro->setParameters(MPL3115A2::DATA_NORMAL, MPL3115A2::DM_BAROMETER, MPL3115A2::OR_16, MPL3115A2::AT_1);

	/*
	 * Setup the Ambient Light Sensor for continuous Ambient Light Sensing, 16 bit resolution,
	 * and 16000 lux range
	 */
	evbAmbLight->setMode(ISL29011::ALS_CONT);
	evbAmbLight->setResolution(ISL29011::ADC_16BIT);
	evbAmbLight->setRange(ISL29011::RNG_16000);

	/*
	 * Set the accelerometer for active mode
	 */
	evbAccel->activeMode();

	/*
	 * Clear the min-max registers in the Barometric Sensor
	 */
	evbBaro->clearMinMaxRegs();

	evbBackLight->setLEDCurrent(0);

	return Board_Ok;
}

EBoardStatus CBoardEVB::readSensors ( BoardSensorData &sensorData )
{
    MMA845x_DATA accel_data;
    uint8_t      result;
	int32_t      num_whole;
	uint32_t     pressure;
	int16_t      num_frac;

	sensorData.init();

    // Update accelerometer state
    evbLCD->startUpdate();
    evbLCD->clearBuffer();

    /*
     * Retrieve and print out accelerometer data
     */
    for(uint32_t i = 0; i < 10; i++)
    {
        osDelay(100); // allows other threads to process

        result = evbAccel->getStatus();

        if( result & MMA845x::XYZDR )
        {
			accel_data = evbAccel->getXYZ();

			sprintf(txtstr,"Accelerometer");
			evbLCD->writeText(0,0,font_6x8,txtstr,strlen(txtstr));
			sprintf(txtstr, "x = %d", accel_data._x);
			evbLCD->writeText(20,1,font_6x8,txtstr,strlen(txtstr));
			sprintf(txtstr, "y = %d", accel_data._y);
			evbLCD->writeText(20,2,font_6x8,txtstr,strlen(txtstr));
			sprintf(txtstr, "z = %d", accel_data._z );
			evbLCD->writeText(20,3,font_6x8,txtstr,strlen(txtstr));

			sensorData.accel_x = accel_data._x;
			sensorData.accel_y = accel_data._y;
			sensorData.accel_z = accel_data._z;

			sensorData.orientation.vertical =  abs(accel_data._x > 600);

			break;
        }
    }

    /*
     * Trigger a Pressure reading
     */
    evbBaro->setParameters(MPL3115A2::DATA_NORMAL, MPL3115A2::DM_BAROMETER, MPL3115A2::OR_16,
                           MPL3115A2::AT_1);

    evbBaro->triggerOneShot();

    /*
     * Retrieve and print out barometric pressure
     */
    for(uint32_t i = 0; i < 10; i++)
    {
    	osDelay(100);			// allows other threads to process
        result = evbBaro->getStatus();
        if( result & MPL3115A2::PTDR)
        {
			pressure = evbBaro->getBaroData() >> 12; // convert 32 bit signed to 20 bit unsigned value
			num_whole = pressure >> 2;			// 18 bit integer significant
			num_frac = (pressure & 0x3) * 25;		// 2 bit fractional  0.25 per bit
			sensorData.pressure = pressure + (.25 * num_frac);

			sprintf(txtstr,"Press=%ld.%02d Pa", num_whole, num_frac);
			evbLCD->writeText(0,4,font_6x8,txtstr,strlen(txtstr));

			break;
        }
    }


    /*
     * Trigger a Altitude reading
     */
    evbBaro->setParameters(MPL3115A2::DATA_NORMAL, MPL3115A2::DM_ALTIMETER, MPL3115A2::OR_16,
                           MPL3115A2::AT_1);
    evbBaro->triggerOneShot();


    /*
     * Retrieve and print out altitude and temperature
     */
    for(uint32_t i = 0; i < 10; i++)
    {
        osDelay(100);			// allows other threads to process
        result = evbBaro->getStatus();
        if( result & MPL3115A2::PTDR)
        {
			baro_data = evbBaro->getAllData(false);
			baro_data._baro /= 4096;				// convert 32 bit signed to 20 bit signed value
			num_whole = baro_data._baro / 16;		//	18 bit signed significant integer
			num_frac = (baro_data._baro & 0xF) * 625 / 100;		// 4 bit fractional .0625 per bit
			sprintf(txtstr,"Alti=%ld.%03d m", num_whole, num_frac);
			evbLCD->writeText(0,5,font_6x8,txtstr,strlen(txtstr));
			num_whole = baro_data._temp / 16;		// 8 bit signed significant integer
			num_frac = (baro_data._temp & 0x0F) * 625 / 100;		// 4 bit fractional .0625 per bit
			sensorData.temperature = num_whole  + ((float)num_frac / 100);
			sprintf(txtstr,"Temp=%ld.%03d C", num_whole, num_frac);
			evbLCD->writeText(0,6,font_6x8,txtstr,strlen(txtstr));

			break;
        }
    }


    /*
     * retrieve and print out Ambient Light level
     */
    uint16_t lux_data = evbAmbLight->getData();
    num_whole = lux_data * 24 / 100;		// 16000 lux full scale .24 lux per bit
    num_frac = lux_data * 24 % 100;
    sprintf(txtstr, "Light=%ld.%02d lux", num_whole, num_frac );
    evbLCD->writeText(0,7,font_6x8,txtstr,strlen(txtstr));

    evbLCD->endUpdate();

    return Board_Ok;
}

EBoardStatus CBoardEVB::setLED(uint8_t ledNum, bool on)
{
	if(ledNum != 1)
		return Board_Invalid;

	evbBackLight->setLEDCurrent(on ? 16 : 0);
	evbLedState = on;
	return Board_Ok;
}

EBoardStatus CBoardEVB::toggleLED(uint8_t ledNum)
{
	if(ledNum != 1)
		return Board_Invalid;

	evbLedState = !evbLedState;
	evbBackLight->setLEDCurrent(evbLedState ? 16 : 0);
	return Board_Ok;
}

void CBoardEVB::pb1ISR(void)
{
    if (!pb1_low)
        pb1_low = true;
}

void CBoardEVB::pb1_debounce()
{
    static uint8_t count = 0;

    while (true) {
        if (pb1_low && (mDot08 == 0))
            count++;
        else {
            count = 0;
            pb1_low = false;
        }

        if (count == 5){
        	if(buttonCallback != 0)
        		buttonCallback(1);
        }

        Thread::wait(5);
    }
}

void CBoardEVB::pb2ISR(void)
{
    if (!pb2_low)
        pb2_low = true;
}

void CBoardEVB::pb2_debounce()
{
    static uint8_t count = 0;

    while (true) {

        if (pb2_low && (mDot09 == 0))
            count++;
        else {
            count = 0;
            pb2_low = false;
        }

        if (count == 5){

        	if(buttonCallback != 0)
        		buttonCallback(2);

        }
        Thread::wait(5);
    }
 }


#endif
