/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    ( C )2014 Semtech

Description: Tx Continuous Wave implementation

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainers: Miguel Luis, Gregory Cristian and Nicolas Huguenin
*/
#include "mbed.h"
#include "sx1276-hal.h"
#include "debug.h"

/* Set this flag to '1' to test the HF max output power or '0' to the the LF max output power */
#define TEST_HF_OUTPUT                              1
#define TEST_LF_OUTPUT = !TEST_HF_OUTPUT

#define LORA_BANDWIDTH                              0         // [0: 125 kHz,
                                                              //  1: 250 kHz,
                                                              //  2: 500 kHz,
                                                              //  3: Reserved]
#define LORA_SPREADING_FACTOR                       9         // [SF7..SF12]
#define LORA_CODINGRATE                             1         // [1: 4/5,
                                                              //  2: 4/6,
                                                              //  3: 4/7,
                                                              //  4: 4/8]
#define LORA_PREAMBLE_LENGTH                        8         // Same for Tx and Rx
#define LORA_SYMBOL_TIMEOUT                         5         // Symbols
#define LORA_FIX_LENGTH_PAYLOAD_ON                  false
#define LORA_FHSS_ENABLED                           false  
#define LORA_NB_SYMB_HOP                            4         // Symbols
#define LORA_IQ_INVERSION_ON                        false
#define LORA_CRC_ENABLED                            true

/*!
 * Radio events function pointer
 */
static RadioEvents_t RadioEvents;

/*!
 * Tx continuous wave parameters
 */
struct
{
    uint32_t RfFrequency;
    uint32_t TxOutputPower;
}TxCwParams;

/*!
 * Radio object
 */
SX1276MB1xAS Radio( NULL );

/*!
 * \brief Function executed on Radio Tx Timeout event
 */
void OnRadioTxTimeout( void )
{
    // Restarts continuous wave transmission when timeout expires after 65535 seconds
    Radio.SetTxContinuousWave( TxCwParams.RfFrequency, TxCwParams.TxOutputPower, 65535 );
}

/**
 * Main application entry point.
 */
int main( void )
{
    debug( "\n\r\n\r     SX1276 Continuous Wave at full power Demo Application \n\r" );
        
#if defined TARGET_NUCLEO_L152RE
    debug( "         > Nucleo-L152RE Platform <\r\n" );
#elif defined TARGET_KL25Z
    debug( "         > KL25Z Platform <\r\n" );
#elif defined TARGET_LPC11U6X
    debug( "         > LPC11U6X Platform <\r\n" );
#else
    debug( "         > Untested Platform <\r\n" );
#endif

#if( TEST_HF_OUTPUT == 1 )

    if( Radio.DetectBoardType( ) == SX1276MB1LAS ) // 
    {
        debug("\r\n     TEST_HF_OUTPUT on SX1276MB1LAS: 20 dBm at 915 MHz \r\n" );

        TxCwParams.RfFrequency = 915e6;
        TxCwParams.TxOutputPower = 20;
    }
    else
    {   // SX1276MB1MAS
        debug("\r\n     TEST_HF_OUTPUT on SX1276MB1MAS: 14 dBm at 868 MHz \r\n" );

        TxCwParams.RfFrequency = 868e6;
        TxCwParams.TxOutputPower = 14;
    }
    
#else //if( TEST_LF_OUTPUT == 1 )

    debug("\r\n     TEST_LF_OUTPUT on SX1276MB1xAS: 14 dBm at 434 MHz \r\n" );
    TxCwParams.RfFrequency = 433e6;
    TxCwParams.TxOutputPower = 14;

#endif

    // Radio initialization
    RadioEvents.TxTimeout = OnRadioTxTimeout;
    Radio.Init( &RadioEvents );

    // Start continuous wave transmission fucntion expires after 65535 seconds
    Radio.SetTxContinuousWave( TxCwParams.RfFrequency, TxCwParams.TxOutputPower, 65535 );

    debug( "Start main loop: \r\n" );
    // Blink LEDs just to show some activity
    while( 1 )
    {
        debug( "Continuous Wave activated... \r\n" );
        wait_ms( 200 );
    }
}
