/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: LoRaMac classA device implementation

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/
#include "mbed.h"
#include "board.h"
#include "radio.h"

#include "LoRaMac.h"
#include "Comissioning.h"
#include "SerialDisplay.h"
#include "ComplianceTest.h"
#include "LoRaDeviceStateProc.h"
#include "LoRaEventProc.h"
#include "LoRaApp.h"

/*!
 * Defines a random delay for application data transmission duty cycle. 1s,
 * value in [us].
 */


#if( OVER_THE_AIR_ACTIVATION != 0 )

uint8_t DevEui[] = LORAWAN_DEVICE_EUI;
uint8_t AppEui[] = LORAWAN_APPLICATION_EUI;
uint8_t AppKey[] = LORAWAN_APPLICATION_KEY;

#else

uint8_t NwkSKey[] = LORAWAN_NWKSKEY;
uint8_t AppSKey[] = LORAWAN_APPSKEY;

/*!
 * Device address
 */
uint32_t DevAddr = LORAWAN_DEVICE_ADDRESS;

#endif

/*!
 * Application port
 */
uint8_t AppPort = LORAWAN_APP_PORT;

/*!
 * User application data size
 */
uint8_t AppDataSize = LORAWAN_APP_DATA_SIZE;

/*!
 * User application data
 */
uint8_t AppData[LORAWAN_APP_DATA_MAX_SIZE];

/*!
 * Application to handle functions
 */
Application LoRaApp( AppData );

/*!
 * Indicates if the node is sending confirmed or unconfirmed messages
 */
uint8_t IsTxConfirmed = LORAWAN_CONFIRMED_MSG_ON;

/*!
 * Timer to handle the application data transmission duty cycle
 */
TimerEvent_t TxNextPacketTimer;

/*!
 * Indicates if a new transmit interrupt can be set
 */
bool IsTxIntUpdate = false;

/*!
 * Indicates if a new packet can be sent
 */
bool NextTx = true;

/*!
 * LoRaWAN compliance tests support data
 */
ComplianceTest_s ComplianceTest;

/*!
 * Indicates if the MAC layer network join status has changed.
 */
bool IsNetworkJoinedStatusUpdate = false;

/*!
 * Indicates if the message sent.
 */
bool IsTxUpdate = false;

/*!
 * Indicates if the message received in the RX window.
 */
bool IsRxUpdate = false;


/**
 * Main application entry point.
 */
int main( void )
{

    // Initialize board peripherals
    BoardInit( );

    // Initialize Device state
    DeviceState = DEVICE_STATE_INIT;

    while( 1 )
    {               
        if( IsNetworkJoinedStatusUpdate == true )
        {
            IsNetworkJoinedStatusUpdate = false;

            DeviceJoinUpdate( );
        }

        if( IsTxUpdate == true )
        {
            // If downlink received then update Serial Terminal and execute Rx event
            IsTxUpdate = false;
            
            // Update serial terminal
            SerialDisplayTxUpdate( );           
        }

        if( IsTxIntUpdate == true )
        {
            IsTxIntUpdate = false;
            
            // Initialize next Tx Interrupt
            InitNextTxInterrupt( AppPort );
        }

        if( IsRxUpdate == true )
        {
            // If downlink received then update Serial Terminal and execute Rx event
            IsRxUpdate = false;
            RxEvent( );
            SerialDisplayRxUpdate( );            
        }
        
        switch( DeviceState )
        {
            case DEVICE_STATE_INIT:
            {
                // Initialize MAC, MAC services, Primitives
                DeviceInit( );

                // Change Device state
                DeviceState = DEVICE_STATE_JOIN;
                break;
            }
            case DEVICE_STATE_JOIN:
            {
#if( OVER_THE_AIR_ACTIVATION != 0 )                // OTA 

                // Generate DevEUI if not defined by User
                BoardGetDevEUI( DevEui );
                
                Led = 1;

                // Join N/w server
                DeviceJoin( );                

                // Show on serial terminal
                SerialDisplayJoinUpdate( );

                // Execute Join event
                JoinEvent( );

                DeviceState = DEVICE_STATE_SLEEP;

#else                                           // ABP
                DeviceJoin( );

                DeviceState = DEVICE_STATE_SEND;
#endif
                IsNetworkJoinedStatusUpdate = true;
                break;
            }
            case DEVICE_STATE_SEND:
            {   
                Led = 0;
                            
                if( NextTx == true )
                {
                    // Prepare payload frame based on application port
                    PrepareTxFrame( AppPort );

                    // Send payload over the air
                    NextTx = SendFrame( );

                    // Execute transmit event
                    TxEvent( );                 
                }

                if( NextTx == false )
                {
                    IsTxUpdate = true;
                }
                
                DeviceState = DEVICE_STATE_SLEEP;
                break;
            }            
            case DEVICE_STATE_SLEEP:
            {
                // Wake up through events
                break;
            }
            default:
            {
                DeviceState = DEVICE_STATE_INIT;
                break;
            }
        }
    }
}
