/*
 / _____)             _              | |
( (____  _____ ____ _| |_ _____  ____| |__
 \____ \| ___ |    (_   _) ___ |/ ___)  _ \
 _____) ) ____| | | || |_| ____( (___| | | |
(______/|_____)_|_|_| \__)_____)\____)_| |_|
    (C)2015 Semtech

Description: VT100 serial display management

License: Revised BSD License, see LICENSE.TXT file include in the project

Maintainer: Miguel Luis and Gregory Cristian
*/
#include "SerialDisplay.h"

VT100 vt( USBTX, USBRX );

void SerialDisplayJoinUpdate( void )
{
    MibRequestConfirm_t mibGet;

    printf( "###### ===== JOINING ==== ######\r\n" );
    DisplayNetworkParam( );
    if( LoRaMacJoinStatus.Status == LORAMAC_STATUS_OK )
    {
        mibGet.Type = MIB_CHANNELS_DATARATE;
        LoRaMacMibGetRequestConfirm( &mibGet );
        printf( "DATA RATE: DR%d\r\n", mibGet.Param.ChannelsDatarate );
    }
    else
    {
        printf( "JOIN TRANSMIT ERROR: %d\r\n", LoRaMacJoinStatus.Status );
    }
    printf( "\r\n" );
}

void SerialDisplayJoinDataRateUpdate( uint8_t Datarate )
{
#if defined( USE_BAND_915 ) || defined( USE_BAND_915_HYBRID )
    // Debug to check for reported issue where join datarate does not
    // alternate after compliance mode exit on command 6
    if( Datarate == LoRaMacJoinStatus.LastDatarate )
    {
        printf( "###### ===== JOIN WARNING ==== ######\r\n" );
        printf( "DATA RATE IS NOT ALTERNATING: CURRENT=DR%d, LAST=DR%d\r\n",
                LoRaMacJoinStatus.LastDatarate, Datarate);

        MibRequestConfirm_t mibGet;
        mibGet.Type = MIB_CHANNELS_MASK;
        if( LoRaMacMibGetRequestConfirm( &mibGet ) == LORAMAC_STATUS_OK)
        {
            printf("CHANNEL MASK: ");
            for( uint8_t i = 0; i < 5; i++ )
            {
                printf("%04x ", mibGet.Param.ChannelsMask[i] );
            }
            printf("\r\n");
        }
        printf("\r\n");
    }
#endif
}

void SerialDisplayTxUpdate( void )
{
    printf( "###### ===== UPLINK FRAME %d ==== ######\r\n", LoRaMacUplinkStatus.UplinkCounter );

    DisplayNetworkParam( );

    printf( "TX PORT: %d\r\n", LoRaMacUplinkStatus.Port );

    if( LoRaMacUplinkStatus.BufferSize != 0 )
    {
        printf( "TX DATA: " );
        if( LoRaMacUplinkStatus.Type == MCPS_CONFIRMED )
        {
            printf( "CONFIRMED\r\n" );
        }
        else
        {
            printf( "UNCONFIRMED\r\n" );
        }
        SerialDisplayHex( LoRaMacUplinkStatus.Buffer, LoRaMacUplinkStatus.BufferSize );
    }

    printf( "DATA RATE: DR%d\r\n", LoRaMacUplinkStatus.Datarate );

    printf( "U/L FREQ: %d\r\n", LoRaMacUplinkStatus.UpLinkFrequency );

    printf( "TX POWER: %d dBm\r\n", 30 - ( LoRaMacUplinkStatus.TxPower << 1 ) );

#if defined( USE_BAND_915 ) || defined( USE_BAND_915_HYBRID )
    MibRequestConfirm_t mibGet;
    mibGet.Type  = MIB_CHANNELS_MASK;
    if( LoRaMacMibGetRequestConfirm( &mibGet ) == LORAMAC_STATUS_OK)
    {
        printf("CHANNEL MASK: ");
        for( uint8_t i = 0; i < 5; i++)
        {
            printf("%04x ", mibGet.Param.ChannelsMask[i] );
        }
        printf("\r\n");
    }
#endif

    printf( "BATTERY: %2.2fV\r\n", BoardGetBatteryVoltage( ) / 1000.0 );
    printf( "\r\n" );
}

void SerialDisplayRxUpdate( void )
{
    printf( "###### ===== DOWNLINK FRAME %d ==== ######\r\n", LoRaMacDownlinkStatus.DownlinkCounter );

    printf( "RX WINDOW: %d\r\n", LoRaMacDownlinkStatus.RxSlot + 1 );
    
    printf( "RX PORT: %d\r\n", LoRaMacDownlinkStatus.Port );

    if( LoRaMacDownlinkStatus.BufferSize != 0 )
    {
        printf( "RX DATA: \r\n" );
        SerialDisplayHex( LoRaMacDownlinkStatus.Buffer, LoRaMacDownlinkStatus.BufferSize );
    }

    printf( "RX RSSI: %d\r\n", LoRaMacDownlinkStatus.Rssi );

    printf( "RX SNR: %d\r\n", LoRaMacDownlinkStatus.Snr );

    printf( "\r\n" );
}

void SerialDisplayHex( uint8_t *pData, uint8_t len )
{
    int i;
    bool newline = 0;

    for( i = 0; i < len; i++ )
    {
        if( newline != 0 )
        {
            printf( "\r\n" );
            newline = 0;
        }

        printf( "%02X ", pData[i] );

        if( ( ( i + 1 ) % 16 ) == 0 )
        {
            newline = 1;
        }
    }
    printf( "\r\n" );
}

void SerialAcclMetrDisplay( uint8_t statusReg )
{
    printf( "===== DEVICE ORIENTATION ====\r\n" );
    if( ( statusReg & 0x40 ) != 0 )
    {
        printf( "HORIZONTAL + " );
        if( ( statusReg & 0x01 ) != 0 )
        {
            printf( "FACE DOWN" );
        }
        else
        {
            printf( "FACE UP" );
        }
    }
    else
    {
        printf( "VERTICAL" ); 
    }
    printf( "\r\n\r\n" );
}

void DisplayNetworkParam( void )
{
    if( Otaa == true )
    {
        printf( "DEVEUI: " );
        SerialDisplayHex( DevEui, 8 );

        printf( "APPEUI: " );
        SerialDisplayHex( AppEui, 8 );

        printf( "APPKEY: " );
        SerialDisplayHex( AppKey, 16 );
    }
#if ( OVER_THE_AIR_ACTIVATION == 0 )
    else
    {
        printf( "DEVADDR: " );

        uint8_t *pData = ( uint8_t* )&DevAddr;
        for( int32_t i = 3; i >= 0; i-- )
        {
            printf( "%02X ", pData[i] );
        }
        printf( "\r\n" );

        printf( "NWKSKEY: " );
        SerialDisplayHex( NwkSKey, 16 );

        printf( "APPSKEY: " );
        SerialDisplayHex( AppSKey, 16 );
    }
#endif
}

void DisplaySwVersion( void )
{
    printf( "###### ===== NAMote-72 Application Demo v1.0.0 ==== ######\r\n" );
    printf( "###### ===== Based on GitHub master branch commit e506c24 ==== ######\r\n\r\n" );
}
