/*
  SerialLCD.h - Serial LCD driver Library

  2010-2013 Copyright (c) Seeed Technology Inc (www.seeedstudio.com)
  Authors: Jimbo.We, Visweswara R and Frankie.Chu  (Orignially written for Seeeduino)

  This library can be used under Apache License 2.0 or MIT License.
 */

#include "SerialLCD.h"

//Initialization Commands or Responses

#define SLCD_INIT   0xA3
#define SLCD_INIT_ACK   0xA5
#define SLCD_INIT_DONE  0xAA

//WorkingMode Commands or Responses
#define SLCD_CONTROL_HEADER 0x9F
#define SLCD_CHAR_HEADER    0xFE
#define SLCD_CURSOR_HEADER  0xFF
#define SLCD_CURSOR_ACK     0x5A

#define SLCD_RETURN_HOME    0x61
#define SLCD_DISPLAY_OFF    0x63
#define SLCD_DISPLAY_ON     0x64
#define SLCD_CLEAR_DISPLAY  0x65
#define SLCD_CURSOR_OFF     0x66
#define SLCD_CURSOR_ON      0x67
#define SLCD_BLINK_OFF      0x68
#define SLCD_BLINK_ON       0x69
#define SLCD_SCROLL_LEFT    0x6C
#define SLCD_SCROLL_RIGHT   0x72
#define SLCD_NO_AUTO_SCROLL 0x6A
#define SLCD_AUTO_SCROLL    0x6D
#define SLCD_LEFT_TO_RIGHT  0x70
#define SLCD_RIGHT_TO_LEFT  0x71
#define SLCD_POWER_ON       0x83
#define SLCD_POWER_OFF      0x82
#define SLCD_INVALIDCOMMAND 0x46
#define SLCD_BACKLIGHT_ON   0x81
#define SLCD_BACKLIGHT_OFF  0x80

SerialLCD::SerialLCD(PinName rx, PinName tx) : Serial(rx,tx)
{
    Serial::baud(9600);
}

/********** High level commands, for the user! **********/

/// Initialize the Serial LCD Driver.
void SerialLCD::begin()
{
    wait_ms(2);
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_POWER_OFF);
    wait_ms(1);
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_POWER_ON);
    wait_ms(1);
    Serial::putc(SLCD_INIT_ACK);
    while(1) {
        if (Serial::readable() > 0 &&Serial::getc()==SLCD_INIT_DONE)
            break;
    }
    wait_ms(2);
}
/// Clear the display
void SerialLCD::clear()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_CLEAR_DISPLAY);
}
/// Return to home(top-left corner of LCD)
void SerialLCD::home()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_RETURN_HOME);
    wait_ms(2); //this command needs more time;
}
/// Set Cursor to (Column,Row) Position
void SerialLCD::setCursor(uint8_t column, uint8_t row)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_CURSOR_HEADER); //cursor header command
    Serial::putc(column);
    Serial::putc(row);
}
/// Switch the display off without clearing RAM
void SerialLCD::noDisplay()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_DISPLAY_OFF);
}
/// Switch the display on
void SerialLCD::display()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_DISPLAY_ON);
}
/// Switch the underline cursor off
void SerialLCD::noCursor()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_CURSOR_OFF);
}
/// Switch the underline cursor on
void SerialLCD::cursor()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_CURSOR_ON);
}

/// Switch off the blinking cursor
void SerialLCD::noBlink()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_BLINK_OFF);
}
/// Switch on the blinking cursor
void SerialLCD::blink()
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_BLINK_ON);
}
/// Scroll the display left without changing the RAM
void SerialLCD::scrollDisplayLeft(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_SCROLL_LEFT);
}
/// Scroll the display right without changing the RAM
void SerialLCD::scrollDisplayRight(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_SCROLL_RIGHT);
}
/// Set the text flow "Left to Right"
void SerialLCD::leftToRight(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_LEFT_TO_RIGHT);
}
/// Set the text flow "Right to Left"
void SerialLCD::rightToLeft(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_RIGHT_TO_LEFT);
}
/// This will 'right justify' text from the cursor
void SerialLCD::autoscroll(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_AUTO_SCROLL);
}
/// This will 'left justify' text from the cursor
void SerialLCD::noAutoscroll(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_NO_AUTO_SCROLL);
}
/// Switch on the LCD power
void SerialLCD::Power(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_POWER_ON);
}
/// Switch off the LCD  power
void SerialLCD::noPower(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_POWER_OFF);
}
/// Switch off the back light
void SerialLCD::noBacklight(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_BACKLIGHT_OFF);
}
/// Switch on the back light
void SerialLCD::backlight(void)
{
    Serial::putc(SLCD_CONTROL_HEADER);
    Serial::putc(SLCD_BACKLIGHT_ON);
}
/// Print char
void SerialLCD::print(uint8_t b)
{
    Serial::putc(SLCD_CHAR_HEADER);
    Serial::putc(b);
}
/// Print char 
void SerialLCD::print(const char b[])
{
    Serial::putc(SLCD_CHAR_HEADER);
    Serial::puts(b);
}
