/* mbed Microcontroller Library
 * Copyright (c) 2018 ARM Limited
 * SPDX-License-Identifier: Apache-2.0
 */
#include <stdio.h>
#include <LoRaRadio.h>
#include "dot_util.h"
#include "RadioEvent.h"

// Application helper
#include "mbed.h"
#include <AnalogIn.h>
#include <DigitalOut.h>

static std::string network_name = "LAleakers";
static std::string network_passphrase = "stephisbroke";
static uint8_t network_id[] = { 0x6C, 0x4E, 0xEF, 0x66, 0xF4, 0x79, 0x86, 0xA6 };
static uint8_t network_key[] = { 0x1F, 0x33, 0xA1, 0x70, 0xA5, 0xF1, 0xFD, 0xA0, 0xAB, 0x69, 0x7A, 0xAE, 0x2B, 0x95, 0x91, 0x6B };
static uint8_t frequency_sub_band = 7;
static lora::NetworkType network_type = lora::PUBLIC_LORAWAN;
static uint8_t join_delay = 5;
static uint8_t ack = 0;
static bool adr = true;

mDot* dot = NULL;
lora::ChannelPlan* plan = new lora::ChannelPlan_US915();

I2C i2c(I2C_SDA, I2C_SCL);

// PIN DEFINITIONS
DigitalOut vcc(GPIO0);
AnalogIn mic(PB_0);

// To sleep, 'wait' should be replaced by 'ThisThread::sleep_for' (C++) or 'thread_sleep_for' (C). If you wish to wait (without sleeping), call 'wait_us'. 'wait_us' is safe to call from ISR context. [since mbed-os-5.14] [-Wdeprecated-declarations] in "main.cpp", Line: 59, Col: 9

int main() {
    i2c.frequency(400000);
    assert(plan);
    RadioEvent events;
    dot = mDot::getInstance(plan);
    assert(dot);
    
    // attach the custom events handler
    dot->setEvents(&events);

    if (!dot->getStandbyFlag()) {
        logInfo("mbed-os library version: %d.%d.%d", MBED_MAJOR_VERSION, MBED_MINOR_VERSION, MBED_PATCH_VERSION);

        // start from a well-known state
        logInfo("defaulting Dot configuration");
        dot->resetConfig();
        dot->resetNetworkSession();

        // make sure library logging is turned on
        dot->setLogLevel(mts::MTSLog::INFO_LEVEL);

        // update configuration if necessary
        if (dot->getJoinMode() != mDot::OTA) {
            logInfo("changing network join mode to OTA");
            if (dot->setJoinMode(mDot::OTA) != mDot::MDOT_OK) {
                logError("failed to set network join mode to OTA");
            }
        }
        // in OTA and AUTO_OTA join modes, the credentials can be passed to the library as a name and passphrase or an ID and KEY
        // only one method or the other should be used!
        // network ID = crc64(network name)
        // network KEY = cmac(network passphrase)
        update_ota_config_name_phrase(network_name, network_passphrase, frequency_sub_band, network_type, ack);
        //update_ota_config_id_key(network_id, network_key, frequency_sub_band, network_type, ack);

        // configure network link checks
        // network link checks are a good alternative to requiring the gateway to ACK every packet and should allow a single gateway to handle more Dots
        // check the link every count packets
        // declare the Dot disconnected after threshold failed link checks
        // for count = 3 and threshold = 5, the Dot will ask for a link check response every 5 packets and will consider the connection lost if it fails to receive 3 responses in a row
        update_network_link_check_config(3, 5);

        // enable or disable Adaptive Data Rate
        dot->setAdr(adr);

        // Configure the join delay
        dot->setJoinDelay(join_delay);

        // save changes to configuration
        logInfo("saving configuration");
        if (!dot->saveConfig()) {
            logError("failed to save configuration");
        }

        // display configuration
        display_config();
    } else {
        // restore the saved session if the dot woke from deepsleep mode
        // useful to use with deepsleep because session info is otherwise lost when the dot enters deepsleep
        logInfo("restoring network session from NVM");
        dot->restoreNetworkSession();
    }
    // join network if not joined
    if (!dot->getNetworkJoinStatus()) {
        join_network();
    }
    
    /* 
        tests: 
            1. Check received transmissions/second from this loop
            2. make the packet size a lot bigger
            3. 
    */
    // get some data in send vector
    std::vector<uint8_t> tx_data;
    tx_data.clear();
    snprintf(arr, 5, "%u", mic.read_u16());
    for(int i = 0; i < 5; i++) {
        tx_data.push_back(arr[i]);
    }
    
    while(1) {
        send_data(tx_data);
    }
}
