#ifndef __CONTROLLER_H__
#define __CONTROLLER_H__

#include <string>

#include "rtos.h"

/** The base class for everything which interfaces with sensors or actuators.
 * Extending classes must implement \a std::string \a getName() and \a void \a update()
 */
class Controller {
    public:
    
        /**
         * Initializes the Controller.
         * @param threaded whether or not the controller should be threaded.
         * @param interval_ms delay between each iteration, only if threaded is set to \a true.
        */
        Controller(bool threaded, int interval_ms);
        
        /**
         * Returns the name of this Controller. Must be implemented by extending classes.
         * @return The name of this Controller.
        */        
        virtual std::string getName() = 0;
        
        /**
         * Returns whether or not this Controller is being threaded.
         * @return \a true if this Controller is being threaded, \a false otherwise.
        */   
        bool isThreaded();

        /**
         * If this Controller is not threaded, \a update() is called. Otherwise, the first time a thread is spawned which repeatedly calls \a update().
         * When the thread has already been spawned no action is taken.
        */           
        void run();

        /**
         * Returns the interval between each iteration if threaded.
         * @return \a interval_ms as specified in the constructor.
        */                 
        int getIntervalMs();

        /**
         * Sets the underlying RTOS thread priority.
         * @see cmis_os.h
        */           
        void setPriority(osPriority priority);

        /**
         * Returns the thread priority.
         * @return the thread priority.
        */           
        osPriority getPriority();
        
    protected:
        /**
         * Updates the Controller. This is called once every \a interval_ms if \a threaded is set to \a true.
        */    
        virtual void update() = 0;
        
        /**
         * The number of times \a run() is called.
        */
        unsigned long num_iters;
                
    private:
        /**
         * Whether or not this Controller is threaded.
        */    
        bool threaded;
        
        /**
         * Whether or not a thread has been spawned for this Controller.
        */        
        bool has_spawned;
        
        /**
         * Pointer to the spawned thread if \a has_spawned is \a true, undefined otherwise.
        */
        Thread *ctrl_thread;
        
        /**
         * Interval in ms between each iteration, only if \a threaded is set to \a true.
        */
        int interval_ms;
        
        /**
         * Priority of the underlying thread.
        */        
        osPriority prio;
        
        /**
         * This function is called when the thread is created, it repeatedly calls \a run() and waits for \a interval_ms milliseconds.
        */        
        static void threadStub(void const *args);
};

#endif