/**
  ******************************************************************************
  * @file       main.cpp
  * @date       22/01/2016
  * @brief      Test the async comunication api
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"
#include "XNucleoNFC01A1.h"
#include "ReadUriCallbacks.h"
#include "WriteUriCallbacks.h"
#include "NDefNfcTagM24SR.h"
#include "NDefLib/RecordType/RecordURI.h"

Thread thread;

/** flag to signal an interrupt from the nfc component*/
#define NFC_INTERRUPT_FLAG    0x1
 
/** flag to signal user button pressure*/
#define BUTTON_PRESSED_FLAG   0x2


NDefLib::NDefNfcTag *tag;

XNucleoNFC01A1 *nfcNucleo;
 
/** Nfc ISR called when the nfc component has a message ready*/
static void nfc_interrupt_callback() {
    thread.signal_set(NFC_INTERRUPT_FLAG);
}//nfcInterruptCallback
 
/** ISR handling button pressure **/ 
static void set_button_press() {
    thread.signal_set(BUTTON_PRESSED_FLAG);    
}//buttonPressed event


void handleEvents() {
    //create the callback to read a tag
    ReadUriCallbacks NDefReadCallback(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
    while (true) {
        osEvent event = thread.signal_wait(0); // wait for any event
        if (event.value.v == NFC_INTERRUPT_FLAG) {
            nfcNucleo->get_M24SR().manage_event();
        } else if (event.value.v == BUTTON_PRESSED_FLAG) {
            printf("Button pressed! Reading the Tag.\n\r");
            tag->set_callback(&NDefReadCallback);
            tag->open_session();
        }
    }
}
 
int main() {
    
#if defined(TARGET_STM)
    InterruptIn userButton(USER_BUTTON);    
    userButton.fall(set_button_press);
#endif
 
    //create the nfc component
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);

    nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,&nfc_interrupt_callback,
        XNucleoNFC01A1::DEFAULT_GPO_PIN,XNucleoNFC01A1::DEFAULT_RF_DISABLE_PIN,
        XNucleoNFC01A1::DEFAULT_LED1_PIN,XNucleoNFC01A1::DEFAULT_LED2_PIN,
        XNucleoNFC01A1::DEFAULT_LED3_PIN);
 
    //No call back needed since default behavior is sync
    nfcNucleo->get_M24SR().get_session();
    nfcNucleo->get_M24SR().manage_I2C_GPO(M24SR::I2C_ANSWER_READY); //switch to async mode
 
    tag = &(nfcNucleo->get_M24SR().get_NDef_tag());
    printf("System Init done!\n\r");
    
     //Start the event handling thread
    thread.start(handleEvents);
    
    //create the callback to write a tag
    WriteUriCallbacks NDefWriteCallback(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
    // write the tag
    tag->set_callback(&NDefWriteCallback); //set the callback
    printf("Writing Tag\n\r");
    tag->open_session();
      
    while (true) {
        __WFE();
    }  

}
