/**
 ******************************************************************************
 * @file    CLT01_38S.h
 * @author  AST/CL
 * @version V1.0.0
 * @date    Feb 4th, 2016
 * @brief   This file contains the class of an CLT01_38SQ7 PLC component.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __CLT01_38S_CLASS_H
#define __CLT01_38S_CLASS_H


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/        
#include "mbed.h"

/* ACTION 2 ------------------------------------------------------------------*
 * Include here component specific header files.                              *
 *----------------------------------------------------------------------------*/
 #include "../Common/plc.h"

/* ACTION 3 ------------------------------------------------------------------*
 * Include here interface specific header files.                              *
 *                                                                            *
 * Example:                                                                   *
 *   #include "../Interfaces/PLCInput.h"                                      *
 *----------------------------------------------------------------------------*/
#include "../Interfaces/PLCInput.h"


/* Classes -------------------------------------------------------------------*/

/**
 * @brief Class representing an CLT01_38S component.
 */
class CLT01_38S : public PLCInput
{
public:

    /*** Constructor and Destructor Methods ***/

    /**
     * @brief Constructor.
     * @param ssel          pin name of the SSEL pin of the SPI device to be used for communication.
     * @param spi           SPI device to be used for communication.
     */
    CLT01_38S(PinName input_ssel, SPI &spi) : PLCInput(), ssel(input_ssel), dev_spi(spi)
    {
        /* Checking stackability. */
        if (!(number_of_plc_input_components < MAX_NUMBER_OF_PLC_INPUT_COMPONENTS)) {
            error("Instantiation of the CLT01_38S component failed: it can be stacked up to %d times.\r\n", MAX_NUMBER_OF_PLC_INPUT_COMPONENTS);
        }

        plc_input_component_instance = number_of_plc_input_components++;
        memset(spi_tx_buff, 0, NB_BYTES * sizeof(uint8_t));
        memset(spi_rx_buff, 0, NB_BYTES * sizeof(uint8_t));
    }
    
    /**
     * @brief Destructor.
     */
    virtual ~CLT01_38S(void) {}
    

    /*** Public Component Related Methods ***/

    /* ACTION 5 --------------------------------------------------------------*
     * Implement here the component's public methods, as wrappers of the C    *
     * component's functions.                                                 *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2).                                     *
     *                                                                        *
     * Example:                                                               *
     *   virtual int get_value(float *pData) //(1)                            *
     *   {                                                                    *
     *     return COMPONENT_get_value(float *pfData);                         *
     *   }                                                                    *
     *                                                                        *
     *   virtual int enable_feature(void) //(2)                               *
     *   {                                                                    *
     *     return COMPONENT_enable_feature();                                 *
     *   }                                                                    *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Initializing the component in 1/16 Microstepping mode.
     * @param  init Pointer to device specific initalization structure.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int init(void *init = NULL)
    {
        return (int) CLT01_38S_Init((void *) init);
    }

    /**
     * @brief  Getting the ID of the component.
     * @param  id Pointer to an allocated variable to store the ID into.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int read_id(uint8_t *id = NULL)
    {
        return (int) CLT01_38S_ReadID((uint8_t *) id);
    }

    /**
     * @brief  Set Read Status
     * @param  None
     * @retval Status
     */
    virtual uint8_t get_read_status(void)
    {
        return (uint8_t) CLT01_38S_GetReadStatus();
    }

    /**
     * @brief   Set Read Status
     * @param   Status
     * @retval  None
     */
    virtual void set_read_status(uint8_t status)
    {
        CLT01_38S_SetReadStatus(status);
    }

    /**
     * @brief   Get Input Status
     * @param   None
     * @retval  Channels status corresponding to 8 inputs
     */
    virtual uint8_t get_input_data(void)
    {
        return (uint8_t) CLT01_38S_GetInputData();
    }

    /**
     * @brief   Over Temperature Alarm bit
     * @param   None
     * @retval  Overtemperature bit, 1 in case of alarm
     */
    virtual uint8_t over_temp_alarm(void)
    {
        return (uint8_t) CLT01_38S_OverTempAlarm();
    }

    /**
     * @brief   Parity Check bits
     * @param   None
     * @retval  Parity bits for diagnosing inconsistency in data transmission
     */
    virtual uint8_t check_parity(void)
    {
        return (uint8_t) CLT01_38S_CheckParity();
    }

    /**
     * @brief   Under Voltage Alarm bit
     * @param   None
     * @retval  Under voltage alarm bit, 1 in case of alarm
     */
    virtual uint8_t under_volt_alarm(void)
    {
        return (uint8_t) CLT01_38S_UnderVoltAlarm();
    }

    /**
     * @brief   Get input information from output channels component
     * @param   TX buffer
     * @param   RX buffer
     * @retval  None
     */
    virtual void dig_inp_array_get_input(uint8_t *inputArray)
    {
        CLT01_38S_DigInpArray_GetInput(inputArray);
    }

    /* Auxiliary method to enable or disable SPI CS pin */
    virtual void set_input_spi(uint8_t l)
    {
        ssel = l;
    }

protected:

    /*** Protected Component Related Methods ***/

    /* ACTION 7 --------------------------------------------------------------*
     * Declare here the component's specific methods.                         *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2);                                     *
     *   + Helper methods, if any, like functions declared in the component's *
     *     source files but not pointed by the component's virtual table (3). *
     *                                                                        *
     * Example:                                                               *
     *   status_t COMPONENT_Init(void *init);                                 *
     *------------------------------------------------------------------------*/
    status_t CLT01_38S_Init(void *init);
    status_t CLT01_38S_ReadID(uint8_t *id);
    uint8_t CLT01_38S_GetReadStatus(void);
    void CLT01_38S_SetReadStatus(uint8_t status);
    uint8_t CLT01_38S_GetInputData(void);
    status_t CLT01_38S_OverTempAlarm(void);
    uint8_t CLT01_38S_CheckParity(void);
    status_t CLT01_38S_UnderVoltAlarm(void);
    void CLT01_38S_DigInpArray_GetInput(uint8_t *inputArray);
    
    /*** Component's I/O Methods ***/

    /**
     * @brief      Utility function to read and write data from/to CLT01_38S at the same time.
     * @param[out] pBufferToRead pointer to the buffer to read data into.
     * @param[in]  pBufferToWrite pointer to the buffer of data to send.
     * @param[in]  NumValues number of values to read and write.
     * @retval     COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
     */
    status_t ReadWrite(uint8_t* pBufferToRead, uint8_t* pBufferToWrite, uint16_t NumValues)
    {
        (void) NumValues;
        uint16_t dataToRead;
        uint16_t dataToWrite;

        // Converts two uint8_t words into one of uint16_t
        dataToWrite = convertFrom8To16(pBufferToWrite);
        
        // Select the chip.
        ssel = 0;

        dataToRead = dev_spi.write(dataToWrite);

        // Unselect the chip.
        ssel = 1;

        // Converts one uint16_t word into two uint8_t
        convertFrom16To8(dataToRead, pBufferToRead);

        return COMPONENT_OK;
    }

    /* ACTION 8 --------------------------------------------------------------*
     * Implement here other I/O methods beyond those already implemented      *
     * above, which are declared extern within the component's header file.   *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Making the CPU wait.
     * @param  None.
     * @retval None.
     */
    void CLT01_38S_Delay(uint32_t delay)
    {
        ThisThread::sleep_for(chrono::milliseconds(delay));
    }

    /**
     * @brief      Writing and reading bytes to/from the component through the SPI at the same time.
     * @param[in]  pByteToTransmit pointer to the buffer of data to send.
     * @param[out] pReceivedByte pointer to the buffer to read data into.
     * @retval     "0" in case of success, "1" otherwise.
     */
    uint8_t CLT01_38S_SpiReadBytes(uint8_t *pByteToTransmit, uint8_t *pReceivedByte)
    {
        return (uint8_t) (ReadWrite(pReceivedByte, pByteToTransmit, BUFFERSIZE) == COMPONENT_OK ? 0 : 1);
    }


    /*** Component's Instance Variables ***/

    /* ACTION 9 --------------------------------------------------------------*
     * Declare here interrupt related variables, if needed.                   *
     * Note that interrupt handling is platform dependent, see                *
     * "Interrupt Related Methods" above.                                     *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq;                                           *
     *------------------------------------------------------------------------*/

    /* ACTION 10 -------------------------------------------------------------*
     * Declare here other pin related variables, if needed.                   *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     PwmOut pwm;                                                        *
     *------------------------------------------------------------------------*/

    /* ACTION 11 -------------------------------------------------------------*
     * Declare here communication related variables, if needed.               *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut ssel;                                                   *
     *     SPI &dev_spi;                                                      *
     *------------------------------------------------------------------------*/
    /* Configuration. */
    DigitalOut ssel;

    /* IO Device. */
    SPI &dev_spi;

    /* ACTION 12 -------------------------------------------------------------*
     * Declare here identity related variables, if needed.                    *
     * Note that there should be only a unique identifier for each component, *
     * which should be the "who_am_i" parameter.                              *
     *------------------------------------------------------------------------*/
    /* Identity */
    uint8_t who_am_i;

    /* ACTION 13 -------------------------------------------------------------*
     * Declare here the component's static and non-static data, one variable  *
     * per line.                                                              *
     *                                                                        *
     * Example:                                                               *
     *   float measure;                                                       *
     *   int instance_id;                                                     *
     *   static int number_of_instances;                                      *
     *------------------------------------------------------------------------*/
    /* Data. */
    uint8_t plc_input_component_instance;

    uint8_t OTA_Buffer;
    uint8_t CLT_PARITY_Buffer;
    uint8_t UVA_Buffer;
    uint8_t CLT_INPUT_STATUS_Buffer;
    uint8_t FLAG_CLT_READ;

    /* Static data. */
    static uint8_t number_of_plc_input_components;

    /* Digital Input Array buffers for SPI communication */
    uint8_t spi_tx_buff[NB_BYTES];
    uint8_t spi_rx_buff[NB_BYTES];


public:

    /* Static data. */
};

#endif // __CLT01_38S_CLASS_H

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/