/**
  ******************************************************************************
  * @file    VNI8200XP.cpp
  * @author  System Lab Noida
  * @version V1.0.0
  * @date    08-July-2015
  * @brief   PLC_CLT01-38SQ7
  * This file provides firmware functions for how  to manage I/O from VNI8200XP
  ==============================================================================    
 
           
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************  
  */
  
  /* Includes ------------------------------------------------------------------*/
#include "plc.h"

#include "VNI8200XP.h"
    
/** @addtogroup Drivers     Drivers
  * @{
  * @brief Demo Driver Layer
  */    
  
/** @addtogroup BSP     BSP
  * @{
  */    
    
/** @addtogroup Components     Components
  * @{
  */

/** @defgroup VNI8200XP    VNI8200XP
  * @{ 
  * @brief Digital Output Driver Layer
  */

/** @defgroup VNI8200XP_Private_variables     VNI8200XP Private variables
  * @{
  * @brief Digital Output Private variables
  */

/* Number of components */
uint8_t VNI8200XP::number_of_plc_output_components = 0;


/** @defgroup VNI8200XP_Exported_Functions     VNI8200XP Exported Functions
  * @{
  * @brief Digital Input exported Function 
  */

/**********************************************************
 * @brief Starts the VNI8200XP library
 * @param  init Initialization structure.
 * @retval COMPONENT_OK in case of success.
 **********************************************************/
status_t VNI8200XP::VNI8200XP_Init(void *init)
{
    VNI_PARITY_Buffer = 0x00;
    VNI_FAULT_Buffer = 0x00;
    VNI_FB_OK = 0x00;
    VNI_TEMP_WARNING = 0x00;
    VNI_PARITY_CHECK = 0x00;
    VNI_POWER_GOOD = 0x00;
    VNI_PARITY_FALL_Buffer = 0x00;

    /* Unselect the Ssrelay chip at startup to avoid FAULT */
    ssel = 1;

    /* Enable Ssrelay Outputs */
    out_en = 1;

    return COMPONENT_OK;
}

/**********************************************************
 * @brief Read id
 * @param id pointer to the identifier to be read.
 * @retval COMPONENT_OK in case of success.
 **********************************************************/
status_t VNI8200XP::VNI8200XP_ReadID(uint8_t *id)
{
  *id = plc_output_component_instance;

  return COMPONENT_OK;
}

/**********************************************************
  * @brief      Set output channels state
  * @param      Out_array, output channel data 
  * @retval     None
 **********************************************************/
void VNI8200XP::VNI8200XP_SetChannels(uint8_t Out_array)
{
  spi_tx_buff[1] = Out_array;
}

/**********************************************************
  * @brief      Get output fault status
  * @param      None
  * @retval     Output channel fault data
 **********************************************************/
uint8_t VNI8200XP::VNI8200XP_ManageFault(void)
{
  VNI_FAULT_Buffer = spi_rx_buff[1];
  return VNI_FAULT_Buffer;
}

/**********************************************************
  * @brief      Get VNI8200XP DC-DC status
  * @param      None
  * @retval     Feedback status, COMPONENT_OK if OK
 **********************************************************/
status_t VNI8200XP::VNI8200XP_CheckDCDCStatus(void)
{
  VNI_FB_OK = spi_rx_buff[0];
  VNI_FB_OK = VNI_FB_OK & 0x80;
  if(VNI_FB_OK == 0x80) {
    return COMPONENT_ERROR;
  } else {
    return COMPONENT_OK;
  }
}

/**********************************************************
  * @brief      Get temperature warning status
  * @param      None
  * @retval     Temperature warning status, COMPONENT_ERROR if over temperature
 **********************************************************/
status_t VNI8200XP::VNI8200XP_TemperatureWarning(void)
{
  VNI_TEMP_WARNING = spi_rx_buff[0];
  VNI_TEMP_WARNING = VNI_TEMP_WARNING & 0x40;
  if(VNI_TEMP_WARNING == 0x40) {
    return COMPONENT_ERROR;
  } else {
    return COMPONENT_OK;
  }
}

/**********************************************************
  * @brief      Get parity check status
  * @param      None
  * @retval     Parity check flag, COMPONENT_ERROR if 0x20
 **********************************************************/
status_t VNI8200XP::VNI8200XP_CheckParity(void)
{
  VNI_PARITY_CHECK = spi_rx_buff[0];
  VNI_PARITY_CHECK = VNI_PARITY_CHECK & 0x20;
  if(VNI_PARITY_CHECK == 0x20) {
    return COMPONENT_ERROR;
  } else {
    return COMPONENT_OK;
  }
}

/**********************************************************
  * @brief      Get power supply status
  * @param      None
  * @retval     Power good bit, COMPONENT_OK in case of power good
 **********************************************************/
status_t VNI8200XP::VNI8200XP_CheckPowerGood(void)
{
  VNI_POWER_GOOD = spi_rx_buff[0];
  VNI_POWER_GOOD = VNI_POWER_GOOD & 0x10;
  if(VNI_POWER_GOOD == 0x10) {
    return COMPONENT_OK;
  } else {
    return COMPONENT_ERROR;
  }
}

/**********************************************************
  * @brief      Get parity bits for input data from VNI
  * @param      None
  * @retval     Parity bits from VNI
 **********************************************************/
uint8_t VNI8200XP::VNI8200XP_CheckCommError(void)
{
    VNI_PARITY_FALL_Buffer = spi_rx_buff[0];
    VNI_PARITY_FALL_Buffer = VNI_PARITY_FALL_Buffer & 0x0F;
    return VNI_PARITY_FALL_Buffer;
}

/**********************************************************
 * @brief  Set output channels component 
 * @param  txBuff
 * @retval None
 **********************************************************/
void VNI8200XP::VNI8200XP_Ssrelay_SetOutput(uint8_t *outputArray)
{
  spi_tx_buff[0] = outputArray[0];
  spi_tx_buff[1] = outputArray[1];

  if(VNI8200XP_SpiWriteBytes(spi_tx_buff, spi_rx_buff) != 0) {
      /* Aborting the program */
      exit(EXIT_FAILURE);
  }
}

/**
  * @} //end VNI8200XP Exported Functions
  */

/**
  * @}  //end VNI8200XP Device Driver
  */
/**
  * @}  //end Components
   */
/**
  * @}  //close group  BSP
  */

/**
  * @}  //close group  Drivers
  */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/  
