/**
 ******************************************************************************
 * @file    XNucleoPLC01A1.h
 * @author  AST/CL
 * @version V1.1.0
 * @date    February 23rd, 2016
 * @brief   Class header file for the X_NUCLEO_PLC01A1 expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __X_NUCLEO_PLC01A1_CLASS_H
#define __X_NUCLEO_PLC01A1_CLASS_H


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/
#include "mbed.h"
/* ACTION 2 ------------------------------------------------------------------*
 * Include here expansion board configuration's header files.                 *
 *----------------------------------------------------------------------------*/
#include "x_nucleo_plc01a1_config.h"
/* ACTION 3 ------------------------------------------------------------------*
 * Include here expansion board's components' header files.                   *
 *                                                                            *
 * Example:                                                                   *
 *   #include "COMPONENT_1.h"                                                 *
 *   #include "COMPONENT_2.h"                                                 *
 *----------------------------------------------------------------------------*/
#include "CLT01_38S.h"
#include "VNI8200XP.h"


/* Classes  ------------------------------------------------------------------*/

class XNucleoPLC01A1
{
public:

    /*** Constructor and Destructor Methods ***/

    /**
     * @brief Constructor.
     * @param input_ssel    pin name of the SSEL pin of the SPI device to be used for communication with Input Channels component
     * @param output_ssel   pin name of the SSEL pin of the SPI device to be used for communication with Output Channels component
     * @param spi           SPI device to be used for communication.
     */
    XNucleoPLC01A1(PinName input_ssel, PinName output_ssel, PinName out_en, SPI &spi) : plcIn(input_ssel, spi), plcOut(out_en, output_ssel, spi), freezeTo(false), attached(false)
    {
        if (plcIn.init() != COMPONENT_OK)
            exit(EXIT_FAILURE);

        if (plcOut.init() != COMPONENT_OK)
            exit(EXIT_FAILURE);

        outBuff[0] = 0;
        outBuff[1] = 0;        
    }
    
    /**
     * @brief Destructor.
     */
    virtual ~XNucleoPLC01A1(void) {}

    /*
     * Accessors to PLC input channels component. Please refer to "PLCInput.h".
     * All PLCInput related unctionality requires going through this accessor.
     */
    const PLCInput& plc_input() const {
        return plcIn;
    }
    PLCInput& plc_input() {
        return plcIn;
    }

    /*
     * Accessors to PLC output channels component. Please refer to "PLCOutput.h".
     * All PLCOutput related functionality requires going through this accessor.
     */
    const PLCOutput& plc_output() const {
        return plcOut;
    }
    PLCOutput& plc_output() {
        return plcOut;
    }

    /*
     * @brief      Mirrors input data
     * @param      Input channel state buffer
     * @retval     Input buffer state
     */
    uint8_t signal_mirror(uint8_t In_Array);

    /*
     * @brief      Freeze selected output for a given time
     * @param      Output channels to be freezed
     * @param      Duration of freeze
     * @retval     Output value
     */
    uint8_t output_freeze(uint8_t N_Channel, uint16_t msec);
    
    /*
     * @brief      Regroup output buffer according to out_Array
     * @param      Regroup array
     * @retval     Value
     */
    uint8_t output_regroup(uint8_t Out_Array);
    
    /*
     * @brief      Sum all the inputs at high state
     * @param      Input channel state data
     * @retval     Value corresponding to the sum of inputs at high
     */
    uint8_t input_sum(uint8_t* Buffer, uint8_t In_Array);

    /*
     * @brief      Set the output channels on/off
     * @param      Output to set
     * @retval     Output value
     */
    uint8_t set_output(uint8_t Out_Array);

    /*
     * @brief      AND Inputs for selected output channels
     * @param      Input channels state
     * @param      Outputs to be AND with inputs
     * @retval     Result of AND operation
     */
    uint8_t inputs_and(uint8_t In_Array, uint8_t Out_Channel);

    /*
     * @brief      OR Inputs for selected output channels
     * @param      Input channels state
     * @param      Outputs to be OR with inputs
     * @retval     Result of OR operation
     */
    uint8_t inputs_or(uint8_t In_Array, uint8_t Out_Channel);
    
    /*
     * @brief      NOT Inputs
     * @param      Input channels state
     * @retval     Result of OR operation
     */
    uint8_t inputs_not(uint8_t In_Array);

    /*
     * @brief      XOR Inputs for selected output channels
     * @param      Input channels state
     * @param      Outputs to be XOR with inputs
     * @retval     Result of XOR operation
     */
    uint8_t inputs_xor(uint8_t In_Array, uint8_t Out_Channel);

    /*
     * @brief      Calculate and set parity bits for Ssrelay transmission
     * @param      Output value buffer
     * @retval     None
     */
    void output_parity_bits(uint8_t* Buffer);

    /*
     * @brief      Toggle selected output for given frequency
     * @param      Output channels to be toggled
     * @retval     None
     */
    void output_cycling(void);

protected:
    void handle_freeze_to(void);
    void toggle_output(void);

    CLT01_38S plcIn;
    VNI8200XP plcOut;

    Timeout timeout;
    bool freezeTo;
    bool attached;
    
    Ticker ticker;
    uint8_t outBuff[2];
};

#endif /* __X_NUCLEO_PLC01A1_CLASS_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
