/**
 ******************************************************************************
 * @file    VNI8200XP.h
 * @author  AST/CL
 * @version V1.0.0
 * @date    Feb 5th, 2016
 * @brief   This file contains the class of an VNI8200XP PLC component.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __VNI8200XP_CLASS_H
#define __VNI8200XP_CLASS_H


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/        
#include "mbed.h"

/* ACTION 2 ------------------------------------------------------------------*
 * Include here component specific header files.                              *
 *----------------------------------------------------------------------------*/
 #include "../Common/plc.h"

/* ACTION 3 ------------------------------------------------------------------*
 * Include here interface specific header files.                              *
 *                                                                            *
 * Example:                                                                   *
 *   #include "../Interfaces/PLCOutput_class.h"                               *
 *----------------------------------------------------------------------------*/
#include "../Interfaces/PLCOutput.h"


/* Classes -------------------------------------------------------------------*/

/**
 * @brief Class representing an VNI8200XP component.
 */
class VNI8200XP : public PLCOutput
{
public:

    /*** Constructor and Destructor Methods ***/

    /**
     * @brief Constructor.
     * @param out_en        pin name of the OUTPUT ENABLE pin used for communication.
     * @param ssel          pin name of the SSEL pin of the SPI device to be used for communication.
     * @param spi           SPI device to be used for communication.
     */
    VNI8200XP(PinName output_en, PinName output_ssel, SPI &spi) : PLCOutput(), out_en(output_en), ssel(output_ssel), dev_spi(spi)
    {
        /* Checking stackability. */
        if (!(number_of_plc_output_components < MAX_NUMBER_OF_PLC_OUTPUT_COMPONENTS)) {
            error("Instantiation of the VNI8200XP component failed: it can be stacked up to %d times.\r\n", MAX_NUMBER_OF_PLC_OUTPUT_COMPONENTS);
        }

        plc_output_component_instance = number_of_plc_output_components++;
        memset(spi_tx_buff, 0, NB_BYTES * sizeof(uint8_t));
        memset(spi_rx_buff, 0, NB_BYTES * sizeof(uint8_t));
    }
    
    /**
     * @brief Destructor.
     */
    virtual ~VNI8200XP(void) {}
    

    /*** Public Component Related Methods ***/

    /* ACTION 5 --------------------------------------------------------------*
     * Implement here the component's public methods, as wrappers of the C    *
     * component's functions.                                                 *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2).                                     *
     *                                                                        *
     * Example:                                                               *
     *   virtual int get_value(float *pData) //(1)                            *
     *   {                                                                    *
     *     return COMPONENT_get_value(float *pfData);                         *
     *   }                                                                    *
     *                                                                        *
     *   virtual int enable_feature(void) //(2)                               *
     *   {                                                                    *
     *     return COMPONENT_enable_feature();                                 *
     *   }                                                                    *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Initializing the component in 1/16 Microstepping mode.
     * @param  init Pointer to device specific initalization structure.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int init(void *init = NULL)
    {
        return (int) VNI8200XP_Init((void *) init);
    }

    /**
     * @brief  Getting the ID of the component.
     * @param  id Pointer to an allocated variable to store the ID into.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int read_id(uint8_t *id = NULL)
    {
        return (int) VNI8200XP_ReadID((uint8_t *) id);
    }

    /**
     * @brief   Set output channels state
     * @param   Output channel data
     * @retval  None
     */
    virtual void set_channels(uint8_t Out_array)
    {
        VNI8200XP_SetChannels(Out_array);
    }

    /**
     * @brief   Get output fault status
     * @param   None
     * @retval  Output channel fault data
     */
    virtual uint8_t manage_fault(void)
    {
        return (uint8_t) VNI8200XP_ManageFault();
    }

    /**
     * @brief   Get DC-DC status of the output channels component
     * @param   None
     * @retval  Feedback status, 1 if OK else 0
     */
    virtual uint8_t check_dcdc_status(void)
    {
         return (uint8_t) VNI8200XP_CheckDCDCStatus();
    }   

    /**
     * @brief   Get temperature warning status
     * @param   None
     * @retval  Temperature warning status, 1 if over temperature
     */
    virtual uint8_t temperature_warning(void)
    {
        return (uint8_t) VNI8200XP_TemperatureWarning();
    }

    /**
     * @brief   Get parity check status
     * @param   None
     * @retval  Parity check flag
     */
    virtual uint8_t check_parity(void)
    {
        return (uint8_t) VNI8200XP_CheckParity();
    }

    /**
     * @brief   Get power supply status
     * @param   None
     * @retval  Power good bit, 1 in case of power good
     */
    virtual uint8_t check_power_good(void)
    {
        return (uint8_t) VNI8200XP_CheckPowerGood();
    }

    /**
     * @brief   Get parity bits for input data
     * @param   None
     * @retval  Parity bits
     */
    virtual uint8_t check_comm_error(void)
    {
        return (uint8_t) VNI8200XP_CheckCommError();
    }

    /**
     * @brief   Set output for output channels component
     * @param   TX buffer
     * @param   RX buffer
     * @retval  None
     */
    virtual void ssrelay_set_output(uint8_t *outputArray)
    {
        VNI8200XP_Ssrelay_SetOutput(outputArray);
    }

    /* Auxiliary method to enable or disable SPI CS pin */
    virtual void set_output_spi(uint8_t l)
    {
        ssel = l;
    }

    /* Auxiliary method to enable or disable Out_En pin */
    virtual void set_output_en(uint8_t l)
    {
        out_en = l;
    }

protected:

    /*** Protected Component Related Methods ***/

    /* ACTION 7 --------------------------------------------------------------*
     * Declare here the component's specific methods.                         *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2);                                     *
     *   + Helper methods, if any, like functions declared in the component's *
     *     source files but not pointed by the component's virtual table (3). *
     *                                                                        *
     * Example:                                                               *
     *   status_t COMPONENT_Init(void *init);                                 *
     *------------------------------------------------------------------------*/
    status_t VNI8200XP_Init(void *init);
    status_t VNI8200XP_ReadID(uint8_t *id);
    void VNI8200XP_SetChannels(uint8_t Out_array);
    uint8_t VNI8200XP_ManageFault(void);
    status_t VNI8200XP_CheckDCDCStatus(void);
    status_t VNI8200XP_TemperatureWarning(void);
    status_t VNI8200XP_CheckParity(void);
    status_t VNI8200XP_CheckPowerGood(void);
    uint8_t VNI8200XP_CheckCommError(void);
    void VNI8200XP_Ssrelay_SetOutput(uint8_t *outputArray);
    
    /*** Component's I/O Methods ***/

    /**
     * @brief      Utility function to read and write data from/to VNI8200XP at the same time.
     * @param[out] pBufferToRead pointer to the buffer to read data into.
     * @param[in]  pBufferToWrite pointer to the buffer of data to send.
     * @param[in]  NumValues number of values to read and write.
     * @retval     COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
     */
    status_t ReadWrite(uint8_t* pBufferToRead, uint8_t* pBufferToWrite, uint16_t NumValues)
    {
        (void) NumValues;
        uint16_t dataToRead;
        uint16_t dataToWrite;

        // Converts two uint8_t words into one of uint16_t
        dataToWrite = convertFrom8To16(pBufferToWrite);
        
        // Select the chip.
        ssel = 0;

        dataToRead = dev_spi.write(dataToWrite);

        // Unselect the chip.
        ssel = 1;

        // Converts one uint16_t word into two uint8_t
        convertFrom16To8(dataToRead, pBufferToRead);

        return COMPONENT_OK;
    }

    /* ACTION 8 --------------------------------------------------------------*
     * Implement here other I/O methods beyond those already implemented      *
     * above, which are declared extern within the component's header file.   *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Making the CPU wait.
     * @param  None.
     * @retval None.
     */
    void VNI8200XP_Delay(uint32_t delay)
    {
        ThisThread::sleep_for(chrono::milliseconds(delay));
    }

    /**
     * @brief      Writing and reading bytes to/from the component through the SPI at the same time.
     * @param[in]  pByteToTransmit pointer to the buffer of data to send.
     * @param[out] pReceivedByte pointer to the buffer to read data into.
     * @retval     "0" in case of success, "1" otherwise.
     */
    uint8_t VNI8200XP_SpiWriteBytes(uint8_t *pByteToTransmit, uint8_t *pReceivedByte)
    {
        return (uint8_t) (ReadWrite(pReceivedByte, pByteToTransmit, BUFFERSIZE) == COMPONENT_OK ? 0 : 1);
    }


    /*** Component's Instance Variables ***/

    /* ACTION 9 --------------------------------------------------------------*
     * Declare here interrupt related variables, if needed.                   *
     * Note that interrupt handling is platform dependent, see                *
     * "Interrupt Related Methods" above.                                     *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq;                                           *
     *------------------------------------------------------------------------*/

    /* ACTION 10 -------------------------------------------------------------*
     * Declare here other pin related variables, if needed.                   *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     PwmOut pwm;                                                        *
     *------------------------------------------------------------------------*/

    /* ACTION 11 -------------------------------------------------------------*
     * Declare here communication related variables, if needed.               *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut ssel;                                                   *
     *     SPI &dev_spi;                                                      *
     *------------------------------------------------------------------------*/
    /* Configuration. */
    DigitalOut out_en;
    DigitalOut ssel;

    /* IO Device. */
    SPI &dev_spi;

    /* ACTION 12 -------------------------------------------------------------*
     * Declare here identity related variables, if needed.                    *
     * Note that there should be only a unique identifier for each component, *
     * which should be the "who_am_i" parameter.                              *
     *------------------------------------------------------------------------*/
    /* Identity */
    uint8_t who_am_i;

    /* ACTION 13 -------------------------------------------------------------*
     * Declare here the component's static and non-static data, one variable  *
     * per line.                                                              *
     *                                                                        *
     * Example:                                                               *
     *   float measure;                                                       *
     *   int instance_id;                                                     *
     *   static int number_of_instances;                                      *
     *------------------------------------------------------------------------*/
    /* Data. */
    uint8_t plc_output_component_instance;

    uint8_t VNI_PARITY_Buffer;
    uint8_t VNI_FAULT_Buffer;
    uint8_t VNI_FB_OK;
    uint8_t VNI_TEMP_WARNING;
    uint8_t VNI_PARITY_CHECK;
    uint8_t VNI_POWER_GOOD;
    uint8_t VNI_PARITY_FALL_Buffer;

    /* Static data. */
    static uint8_t number_of_plc_output_components;

    /* Ssrelay Array buffers for SPI communication */
    uint8_t spi_tx_buff[NB_BYTES];
    uint8_t spi_rx_buff[NB_BYTES];


public:

    /* Static data. */
};

#endif // __VNI8200XP_CLASS_H

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/