/**
 ******************************************************************************
 * @file    XNucleoOUT01A1.cpp
 * @author  ST CLab
 * @version V1.0.0
 * @date    1 February 2018
 * @brief   Implementation of XNucleoOUT01A1 class
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */
 
  /* Includes ----------------------------------------------------------------*/

#include "XNucleoOUT01A1.h"

/* Macros --------------------------------------------------------------------*/

#define SET_PIN(pin)    ((pin != NC) ? new DigitalOut(pin) : NULL)

/* Class Implementation ------------------------------------------------------*/

    /** Constructor
     * @param ctl_mode Control mode, direct or sync. 
     * @param CTL_MODE_PIN Control mode pin, if connected then jumper J4 overrides ctl_mode parameter
     * @param OUT_EN Output enable pin.
     * @param N_SYNC Input-to-output synchronization signal. Active low.
     * @param N_LOAD Load input data signal. Active low.
     * @param INx  Channel x input pin, NC if not connected
     */

    XNucleoOUT01A1::XNucleoOUT01A1(control_mode_t ctl_mode, PinName CTL_MODE_PIN, 
            PinName OUT_EN, PinName N_SYNC, PinName N_LOAD, PinName IN1, 
            PinName IN2, PinName IN3, PinName IN4, PinName IN5, PinName IN6, 
            PinName IN7, PinName IN8) 
            : iso8200bq(OUT_EN, N_SYNC, N_LOAD)
    {

        if (CTL_MODE_PIN != NC) {   // ctl_mode selected by jumper J4
            _ctl_mode_pin = new DigitalIn(CTL_MODE_PIN);
            if (*_ctl_mode_pin == 1) {
                _mode = mode_sync;
            } else {
                _mode = mode_direct;
            }
        } else {                   // ctl_mode provided by the user
            _mode = ctl_mode;
            
        }
        
        if (_mode == mode_direct) { // set the component in direct mode
            iso8200bq.direct_mode();
        }
        
        _in_array[0] = SET_PIN(IN1); 
        _in_array[1] = SET_PIN(IN2);
        _in_array[2] = SET_PIN(IN3);
        _in_array[3] = SET_PIN(IN4);
        _in_array[4] = SET_PIN(IN5);
        _in_array[5] = SET_PIN(IN6);
        _in_array[6] = SET_PIN(IN7);
        _in_array[7] = SET_PIN(IN8);
    
    }


     /**
     *
     * @brief Enable outputs 
     *
     * @param enable Enable outputs when true, disable otherwise.   
    */
    
    void XNucleoOUT01A1::enable_outputs(bool enable) {
        iso8200bq.enable_outputs(enable);    
    }
    
    /**
     *
     * @brief Set input status, also loaded into input buffer when in sync mode 
     *
     * @param input_values Bitmask of input values, bit x represents INx pin's value  
    */
    void XNucleoOUT01A1::set_inputs(uint8_t input_values) {
    
        if (_mode == mode_sync) {
            iso8200bq.sync_mode_load_inputs();
        }                   
                
        for (int i=0; i<8; i++) {
            int value = (input_values >> i) & 0x01;
            if (_in_array[i] != NULL) {
                *_in_array[i] = value;
            }
        }
    }
    
    /**
     *
     * @brief Update output status according to input buffer when in sync mode 
     *
     * @param input_values Bitmask of input values, bit x represents INx pin's value  
    */
    void XNucleoOUT01A1::update_outputs(void) {
        
        if (_mode == mode_sync) {
            iso8200bq.sync_mode_update_outputs();
        }
    }