/**
 ******************************************************************************
 * @file    ISO8200BQ.cpp
 * @author  ST CLab
 * @version V1.0.0
 * @date    1 February 2018
 * @brief   Implementation of a ISO8200BQ class
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */
 
 /* Includes ------------------------------------------------------------------*/

#include "ISO8200BQ.h"


/* Class Implementation ------------------------------------------------------*/

    /** Constructor
     * @param OUT_EN Output enable pin.
     * @param N_SYNC Input-to-output synchronization signal. Active low.
     * @param N_LOAD Load input data signal. Active low.
     */

    ISO8200BQ::ISO8200BQ(PinName OUT_EN, PinName N_SYNC, 
            PinName N_LOAD) : _out_en(OUT_EN), _n_sync(N_SYNC), _n_load(N_LOAD)
    {
        /* start with the component disabled*/
        _out_en = 0;
    };

   /**
    * @brief  Getting the ID of the component.
    * @param  id Pointer to an allocated variable to store the ID into.
    * @retval "0" in case of success, an error code otherwise.
    */
    int ISO8200BQ::read_id(uint8_t *id = NULL)
    {
        return 0;
    }    

    /**
     *
     * @brief One time device initialization
     *
     * @param void
     * @retval "0" in case of success, an error code otherwise.
    */
    int ISO8200BQ::init(void *init)
    {
        /* initalize with SYNC and LOAD signals disabled*/
        _n_sync = _n_load = 1;
        return 0;
    }   
    
    /**
     *
     * @brief Enable/Disable all outputs 
     *
     * @param enable Enable the outputs when true, disable otherwise
    */
    void ISO8200BQ::enable_outputs(bool enable) {
        _out_en = enable ? 1 : 0;
    }
    
    /**
     *
     * @brief Get outputs enabling status 
     *
     * @retval Returns true if outputs are enabled, false otherwise
    */
    bool ISO8200BQ::are_outputs_enabled (void) {
        return (_out_en == 0) ? false : true;
    }
    
    /**
     *
     * @brief Load input pin status into input buffer (synch mode) 
     *
    */
    void ISO8200BQ::sync_mode_load_inputs (void) {
        _n_load = 1; //disabled
        _n_sync = 1; //disabled
        wait_us(20); //tdis(sync)
        _n_load = 0; //enabled
    }
    
    /**
     *
     * @brief Update outputs accordingly to input buffer (synch mode) 
     *
    */
    void ISO8200BQ::sync_mode_update_outputs(void) {
        wait_us(100); // tw(load)
        _n_load = 1; //disabled
        wait_us(1); //tsu(load)
        _n_sync = 0; //enabled
        wait_us(100); // tw(sync)
        _n_sync = 1; // trigger output updates (on rising edge)
    }
    
    /**
     *
     * @brief Update outputs accordingly to input pins (direct mode) 
     *
    */
    void ISO8200BQ::direct_mode (void) {
        _n_load = 0; //enabled
        _n_sync = 0; //enabled
        wait_us(1); //tinld
    }
    
    