/**
  ******************************************************************************
  * @file       Type4NfcTagM24SR.h
  * @author  	ST Central Labs
  * @version 	V1.0.0
  * @date       05 Nov 2015
  * @brief      M24SR specific NDefLib derived class
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#ifndef NDEFNFCTAGSTM24_H_
#define NDEFNFCTAGSTM24_H_
#include <stdint.h>

#include "NDefLib/NDefNfcTag.h"

class M24SR;

/**
 * Implement all NDefLib abstract methods. */
class NDefNfcTagM24SR: public NDefLib::NDefNfcTag {

public:
	/**
	 * Create the object.
	 * @param Device device to write the Ndef tags into.
	 */
	NDefNfcTagM24SR(M24SR &device) :
			mDevice(device), mMaxReadBytes(0xFF), mMaxWriteBytes(
					0xFF) {
	}

	/**
	 * Open the communication session with the nfc tag.
	 * @par This method should be called at the end of an overriding implementation, just before returning.
	 * @param Force force to open a communication.
	 * @return true if success
	 */
	virtual bool openSession(bool force = false);
	
	/**
	 * Close the communication with the nfc tag.
	 * @par This method should be called at the end of an overriding implementation, just before returning.
	 * @return true if success
	 */
	virtual bool closeSession();

	/**
	 * Close the open session.
	 */
	virtual ~NDefNfcTagM24SR() {
		if(isSessionOpen())
			closeSession();
	}//~Type4NfcTagM24SR

	

protected:
	
	/**
	 * Write a sequence of bytes to the NDEF file.
	 * @param buffer Buffer to write.
	 * @param length Number of bytes to write.
	 * @param offset Write offset in bytes.
	 * @return true if success
	 */
	virtual bool writeByte(const uint8_t *buffer, const uint16_t length, uint16_t offset);
	
	/**
	 * Read a sequence of bytes from the NDEF file.
	 * @param byteOffset Read offsetin bytes.
	 * @param byteLength Number of bytes to read.
	 * @param[out] buffer Buffer to store the read data into.
	 * @return true if success
	 */
	virtual bool readByte(const uint16_t byteOffset, const uint16_t byteLength,
			uint8_t *buffer);

private:

	M24SR &mDevice;

	/**
	 * Max length for a read operation
	 */
	uint16_t mMaxReadBytes;

	/**
	 * Max length for a write operation
	 */
	uint16_t mMaxWriteBytes;
};

#endif /* NDEFNFCTAGSTM24_H_ */
