/**
  ******************************************************************************
  * @file       NdefNfcTagSTM24SR.cpp
  * @author  	ST Central Labs
  * @version 	V1.0.0 
  * @date       05 Nov 2015
  * @brief      wrapper class for use the NDefLib library for write/read ndef message
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "NDefNfcTagM24SR.h"
#include "m24sr_class.h"
/* wait 1sec, driver is configured to let 200ms for command to complete */
/* which is enough for all commands except GetSession if RF session is already opened */
/* Smartphone generally releases the session within the second, anyway the user can modify this value */
#define OPENSESSION_NTRIALS 5

#define CC_FILE_LENGTH_BYTE 15

bool NDefNfcTagM24SR::openSession(bool force) {

	if (isSessionOpen())
		return true;
	//else
	if (force)
		mDevice.KillSession();

	int8_t nTrials = OPENSESSION_NTRIALS;
	NFC_StatusTypeDef status = NFC_ERROR;
	while (status != NFC_SUCCESS && nTrials) {
		status = mDevice.GetSession();
		nTrials--;
	}

	if (status != NFC_SUCCESS) {
		return false;
	} //else

	//Select the NFC type 4 application
	if (mDevice.SelectApplication() != NFC_SUCCESS)
		return false;

	if (mDevice.SelectCCfile() != NFC_SUCCESS)
		return false;

	uint8_t CCFile[CC_FILE_LENGTH_BYTE];
	/* read the first 15 bytes of the CC file */
	if (mDevice.ReadBinary(0x0000, CC_FILE_LENGTH_BYTE, CCFile) != NFC_SUCCESS)
		return false;

	//read the ndefFileId
	uint16_t ndefFileId = (uint16_t) ((CCFile[0x09] << 8) | CCFile[0x0A]);
	mMaxReadBytes = (uint16_t) ((CCFile[0x03] << 8) | CCFile[0x04]);
	mMaxWriteBytes = (uint16_t) ((CCFile[0x05] << 8) | CCFile[0x06]);

	//open it
	if (mDevice.SelectNDEFfile(ndefFileId) != NFC_SUCCESS)
		return false;

	NDefNfcTag::openSession();
	return true;
}

bool NDefNfcTagM24SR::closeSession() {

	//no open session
	if (!isSessionOpen())
		return true;

	//close the CC file
	if (mDevice.Deselect() != NFC_SUCCESS)
		return false;

	NDefNfcTag::closeSession();
	return true;
}

bool NDefNfcTagM24SR::writeByte(const uint8_t *buffer, uint16_t length, uint16_t offset) {

	//no open session
	if (!isSessionOpen())
		return false;

	NFC_StatusTypeDef status = NFC_SUCCESS;
	bool prevSessionStatus = isSessionOpen();
	if (!prevSessionStatus) {
		//try to acquire a session or fail
		if (openSession(false))
			return false;
	} //if

	if (length > mMaxWriteBytes) {
		do {
			status = mDevice.UpdateBinary(offset, mMaxWriteBytes,(uint8_t*) buffer);
			offset += mMaxWriteBytes;
			buffer += mMaxWriteBytes;
			length -= mMaxWriteBytes;
		} while (length > mMaxWriteBytes && status == NFC_SUCCESS);
	} //if

	//finish to write the buffer
	if (status == NFC_SUCCESS && mMaxWriteBytes)
		status = mDevice.UpdateBinary(offset, length,(uint8_t*) buffer);

	return status==NFC_SUCCESS;
}

bool NDefNfcTagM24SR::readByte(uint16_t byteOffset, uint16_t byteLength,
		uint8_t *buffer) {
			
	//no open session
	if (!isSessionOpen())
		return false;
		
	NFC_StatusTypeDef status = NFC_SUCCESS;
	uint16_t offset = byteOffset;
	if (byteLength > mMaxReadBytes) {
		do {
			status = mDevice.ReadBinary(offset, mMaxReadBytes, buffer);
			offset += mMaxReadBytes;
			buffer += mMaxReadBytes;
			byteLength -= mMaxReadBytes;
		} while (byteLength > mMaxWriteBytes && status == NFC_SUCCESS);
	} //if
	  //finish to write the buffer
	if (status == NFC_SUCCESS && mMaxWriteBytes)
		status = mDevice.ReadBinary(offset, byteLength, buffer);

	return status == NFC_SUCCESS;
}

