/**
 ******************************************************************************
 * @file    Nfc_class.h
 * @author  ST Central Labs
 * @version V1.0.0
 * @date    13-April-2015
 * @brief   This file contains the abstract class describing the interface of a
 *          nfc component.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Generated with Stm32CubeTOO -----------------------------------------------*/

/* Define to prevent from recursive inclusion --------------------------------*/

#ifndef __NFC_CLASS_H
#define __NFC_CLASS_H


/* Includes ------------------------------------------------------------------*/

#include "Component_class.h"


/* Classes  ------------------------------------------------------------------*/

/**
 * An abstract class for Nfc components.
 */
class Nfc : public Component
{
public:

	/**
	 * Possible password to set.
	 */
	typedef enum{
		ReadPwd=READ_PWD,  //!< Password to use before read the tag
		WritePwd=WRITE_PWD,//!< Password to use before write the tag
		I2CPwd=I2C_PWD,    //!< Root password, used only thought nfc
	}PasswordType_t;


	/**
	 * Open a I2C connection with the tag if an RF connection isn't already open.
	 * @return operation status
	 */
	virtual NFC_StatusTypeDef GetSession(void) = 0;
	
	/**
	 * Force to open an I2C connection , abort the RF connection.
	 * @return NFC_SUCCESS if the session is acquired
	 */
	virtual NFC_StatusTypeDef KillSession(void) = 0;

	/**
	 * Close an I2C connection.
	 * @return NFC_SUCCESS if the session is release
	 */
	virtual NFC_StatusTypeDef Deselect(void) = 0;

	/**
	 * Select the application file.
	 * @return NFC_SUCCESS if the application is selected
	 */
	virtual NFC_StatusTypeDef SelectApplication(void) = 0;

	/**
	 * Select the CC file.
	 * @return NFC_SUCCESS if the CC file is selected.
	 */
	virtual NFC_StatusTypeDef SelectCCfile(void) = 0;

	/**
	 * Select the ndef file.
	 * @param NDEFfileId File id to open.
	 * @return NFC_SUCCESS if the file is selected
	 */
	virtual NFC_StatusTypeDef SelectNDEFfile(uint16_t NDEFfileId) = 0;

	/**
	 * Select the system file.
	 * @return NFC_SUCCESS if the system file is selected
	 */
	virtual NFC_StatusTypeDef SelectSystemfile(void) = 0;

	/**
	 * Read data from the tag.
	 * @param Offset Read offset.
	 * @param NbByteToRead Number of bytes to read.
	 * @param[out] pBufferRead Buffer to store the read data into.
	 * @return NFC_SUCCESS if no errors 
	 */
	virtual NFC_StatusTypeDef ReadBinary(uint16_t Offset, uint8_t NbByteToRead, uint8_t *pBufferRead) = 0;

	/**
	 * Write data to the tag.
	 * @param Offset Write offset.
	 * @param NbByteToWrite Number of bytes to write.
	 * @param pDataToWrite Buffer to write.
	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef UpdateBinary(uint16_t Offset, uint8_t NbByteToWrite, uint8_t *pDataToWrite) = 0;

	/**
	 * Check that the password is correct.
	 * @param uPwdId Password type.
	 * @param NbPwdByte Password size ( must be 0x10).
	 * @param pPwd Password buffer.
	 * @return NFC_SUCCESS if no errors and write/read permission granted
	 */
	virtual NFC_StatusTypeDef Verify(PasswordType_t uPwdId, uint8_t NbPwdByte,const uint8_t *pPwd) = 0;

	/**
	 * Replace the read or write password.
	 * @param uPwdId Password to change.
	 * @param pPwd New password.
	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef ChangeReferenceData(PasswordType_t uPwdId,const uint8_t *pPwd) = 0;
	
	/**
	 *	Activate NDEF file password protection.
	 *  When this command is successful, read or write access to the NDEF file is protected by a 128-bit password.
	 *  @param uReadOrWrite Read or write password.
	 *  @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef EnableVerificationRequirement(PasswordType_t uReadOrWrite) = 0;
	
	/**
     * Deactivate NDEF file password protection.
     * When this command is successful, read or write access to the NDEF file is granted.
     * with no security enforcement.
	 * @param uReadOrWrite Read or write password.
	 * @return NFC_SUCCESS if no errors
	 */     
	virtual NFC_StatusTypeDef DisableVerificationRequirement(PasswordType_t uReadOrWrite) = 0;
	
	/**
	 * Same as {@link NFC#ReadBinary}, however permitting to read more bytes than available.
	 * @param Offset read offset.
	 * @param NbByteToRead Number of bytes to read.
	 * @param[out] pBufferRead Buffer to store the read data into.
	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef STReadBinary(uint16_t Offset, uint8_t NbByteToRead, uint8_t *pBufferRead) = 0;
	
	/**
	 * Set the tag as read/write only, stored in the eeprom.
	 * @param  uReadOrWrite ReadPwd if write only, WritePwd if read only.
	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef EnablePermanentState(PasswordType_t uReadOrWrite) = 0;
	
	/**
	 * Unset the tag as read/write only, stored in the eeprom.
	 * @param  uReadOrWrite ReadPwd if write only, WritePwd if read only.
	 * @return NFC_SUCCESS if no errors
	 * @par Caller must have I2Csuper user permissions to call this function.
	 */
	virtual NFC_StatusTypeDef DisablePermanentState(PasswordType_t uReadOrWrite) = 0;
	
	/**
	 * Set the gpo output pin.
	 * @param uSetOrReset New pin status.
 	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef StateControl(uint8_t uSetOrReset) = 0;
	
	/**
	 * @brief  This function configures GPO for I2C session.
	 * @param  GPO_I2Cconfig GPO configuration to set.
 	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef ManageI2CGPO(uint8_t GPO_I2Cconfig) = 0;
	
	/**
 	* @brief  This function configures GPO for RF session.
 	* @param  GPO_RFconfig GPO configuration to set.
  	* @return NFC_SUCCESS if no errors
 	*/
	virtual NFC_StatusTypeDef ManageRFGPO(uint8_t GPO_RFconfig) = 0;
	
	/**
	 * @brief  This function enables or disables the RF communication.
	 * @param  OnOffChoice GPO configuration to set.
  	 * @return NFC_SUCCESS if no errors
	 */
	virtual NFC_StatusTypeDef RFConfig(uint8_t OnOffChoice) = 0;

	virtual ~Nfc(){};
};

#endif /* __NFC_CLASS_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
