/**
 ******************************************************************************
 * @file    m24sr_class.h
 * @author  ST Central Labs
 * @date    05 Nov 2015
 * @brief   This file provides a set of functions to interface with the M24SR
 *          device.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Revision ------------------------------------------------------------------*/
/*
 Based on:         X-CUBE-MEMS1/trunk/Drivers/BSP/Components/m24sr/m24sr.h
 Revision:         M24SR Driver V1.0.0
 */

/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __M24SR_CLASS_H
#define __M24SR_CLASS_H

/* Includes ------------------------------------------------------------------*/

#include "mbed.h"
#include "I2C.h"

#include "m24sr.h"
#include "Interfaces/Nfc_class.h"
#include "NDefNfcTagM24SR.h"

/* Classes -------------------------------------------------------------------*/

/** Class representing a M24SR component.
 */
class M24SR: public Nfc {

public:

	/**
	 * default password uesed for change the write/read permission
	 */
	static const uint8_t DEFAULT_PASSWORD[16];
	
	/*** Constructor and Destructor Methods ***/

	/**
	 * @brief Constructor.
	 * @param address I2C address of the component.
	 * @param i2c     I2C device to be used for communication.
	 * @param GPOPinName pin used as GPIO
	 * @param RFDISPinName pin used for disable the RF function 
	 */
	M24SR(const uint8_t address, I2C &i2c, const PinName& GPOPinName,
			const PinName& RFDISPinName) :
			Nfc(), address(address), dev_i2c(i2c), GPOPin(GPOPinName), RFDisablePin(
					RFDISPinName), answerReadyInterrupt(GPOPinName), interruptIsFired(
					false),NDefTagUtil(*this) {

		memset(uM24SRbuffer, 0, 0xFF * sizeof(int8_t));
		syncMode = M24SR_WAITINGTIME_POLLING;
		uDIDbyte = 0;
		RFDisablePin = 0;
		GPOPin.mode(PullNone);
		answerReadyInterrupt.disable_irq();
		answerReadyInterrupt.fall(this,
				&M24SR::M24SR_AnswerReadyInterruptCallback);
		answerReadyInterrupt.mode(PullUp);

	}

	/**
	 * @brief Destructor.
	 */
	virtual ~M24SR(void) {
	}

	/*** Public Component Related Methods ***/
	
	/**
	 * @brief  This function initialize the M24SR device
	 * @param ptr configure paramiters, not used
     * @return NFC_SUCCESS if no error happen
	 */
	virtual int Init(void *ptr) {
		return (NFC_StatusTypeDef) M24SR_Init((NFC_InitTypeDef*)ptr);
	}

	virtual int ReadID(uint8_t *id) {
		return (NFC_StatusTypeDef) M24SR_ReadID((uint8_t *) id);
	}

	/** lock the tag channel */
	virtual NFC_StatusTypeDef GetSession(void) {
		return (NFC_StatusTypeDef) M24SR_GetSession();
	}

	virtual NFC_StatusTypeDef KillSession(void) {
		return (NFC_StatusTypeDef) M24SR_KillSession();
	}

	virtual NFC_StatusTypeDef Deselect(void) {
		return (NFC_StatusTypeDef) M24SR_Deselect();
	}

	////////// tag4 command ////////////////////
	
	virtual NFC_StatusTypeDef SelectApplication(void) {
		return (NFC_StatusTypeDef) M24SR_SelectApplication();
	}

	virtual NFC_StatusTypeDef SelectCCfile(void) {
		return (NFC_StatusTypeDef) M24SR_SelectCCfile();
	}

	virtual NFC_StatusTypeDef SelectNDEFfile(uint16_t NDEFfileId) {
		return (NFC_StatusTypeDef) M24SR_SelectNDEFfile((uint16_t) NDEFfileId);
	}

	virtual NFC_StatusTypeDef SelectSystemfile(void) {
		return (NFC_StatusTypeDef) M24SR_SelectSystemfile();
	}

	virtual NFC_StatusTypeDef ReadBinary(uint16_t Offset, uint8_t NbByteToRead,
			uint8_t *pBufferRead) {
		return (NFC_StatusTypeDef) M24SR_ReadBinary((uint16_t) Offset,
				(uint8_t) NbByteToRead, (uint8_t *) pBufferRead);
	}

	virtual NFC_StatusTypeDef UpdateBinary(uint16_t Offset,
			uint8_t NbByteToWrite, uint8_t *pDataToWrite) {
		return (NFC_StatusTypeDef) M24SR_UpdateBinary((uint16_t) Offset,
				(uint8_t) NbByteToWrite, (uint8_t *) pDataToWrite);
	}

	/////////////////// iso 7816-4 commands //////////////////////////
	
	virtual NFC_StatusTypeDef Verify(PasswordType_t pwdId, uint8_t NbPwdByte,
			const uint8_t *pPwd) {
		return (NFC_StatusTypeDef) M24SR_Verify((uint16_t) pwdId,
				(uint8_t) NbPwdByte, pPwd);
	}

	virtual NFC_StatusTypeDef ChangeReferenceData(PasswordType_t pwdId,
			const uint8_t *pPwd) {
		return (NFC_StatusTypeDef) M24SR_ChangeReferenceData((uint16_t) pwdId,
				(uint8_t *) pPwd);
	}

	virtual NFC_StatusTypeDef EnableVerificationRequirement(
			PasswordType_t uReadOrWrite) {
		return (NFC_StatusTypeDef) M24SR_EnableVerificationRequirement(
				(uint16_t) uReadOrWrite);
	}

	virtual NFC_StatusTypeDef DisableVerificationRequirement(
			PasswordType_t uReadOrWrite) {
		return (NFC_StatusTypeDef) M24SR_DisableVerificationRequirement(
				(uint16_t) uReadOrWrite);
	}

    ////////////////////// ST proprietary //////////////////////////////////
    /**
     * as 
     */
	virtual NFC_StatusTypeDef STReadBinary(uint16_t Offset,
			uint8_t NbByteToRead, uint8_t *pBufferRead) {
		return (NFC_StatusTypeDef) M24SR_STReadBinary((uint16_t) Offset,
				(uint8_t) NbByteToRead, (uint8_t *) pBufferRead);
	}

	virtual NFC_StatusTypeDef EnablePermanentState(PasswordType_t uReadOrWrite) {
		return (NFC_StatusTypeDef) M24SR_EnablePermanentState(
				(uint16_t) uReadOrWrite);
	}

	virtual NFC_StatusTypeDef DisablePermanentState(PasswordType_t uReadOrWrite) {
		return (NFC_StatusTypeDef) M24SR_DisablePermanentState(
				(uint16_t) uReadOrWrite);
	}

	///////////////////// chip configuration /////////////////////////////////
	virtual NFC_StatusTypeDef StateControl(uint8_t uSetOrReset) {
		return (NFC_StatusTypeDef) M24SR_StateControl((uint8_t) uSetOrReset);
	}

	virtual NFC_StatusTypeDef ManageI2CGPO(uint8_t GPO_I2Cconfig) {
		return (NFC_StatusTypeDef) M24SR_ManageI2CGPO(
				(M24SR_GPO_MGMT) GPO_I2Cconfig);
	}

	virtual NFC_StatusTypeDef ManageRFGPO(uint8_t GPO_RFconfig) {
		return (NFC_StatusTypeDef) M24SR_ManageRFGPO(
				(M24SR_GPO_MGMT) GPO_RFconfig);
	}

	virtual NFC_StatusTypeDef RFConfig(uint8_t OnOffChoice) {
		return (NFC_StatusTypeDef) M24SR_RFConfig((uint8_t) OnOffChoice);
	}

	/**
	 * Generates a negative pulse on the GPO pin.
	 * It starts at the end of the command and ends at the end of the RF response
  	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef SendInterrupt(void) {
		return (NFC_StatusTypeDef) M24SR_SendInterrupt();
	}

	/////////////////// hight level/utility function /////////////////////

	/**
	 * get an implementation of NDefNfcTag for use the library NDefLib
	 * @retrun object of type NdefNfcTag for use this M24SR component
	 */
	NDefLib::NDefNfcTag& getNDefTag(){
		return NDefTagUtil;
	}


	/**
	 * enable the request of a password before read the tag
	 * @param pCurrentWritePassword current password ( use M24SR::DEFAULT_PASSWORD if it is the first time)
	 * @param pNewPassword password to request before read the tag
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef enableReadPassword(const uint8_t* pCurrentWritePassword,
			const uint8_t* pNewPassword);

	/**
	 * disable the request of a password before read the tag
	 * @param pCurrentWritePassword current password ( use M24SR::DEFAULT_PASSWORD if it is the first time)
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef disableReadPassword(const uint8_t* pCurrentWritePassword=M24SR::DEFAULT_PASSWORD);

	/**
	 * enable the request of a password before write the tag
	 * @param pCurrentWritePassword current password ( use M24SR::DEFAULT_PASSWORD if it is the first time)
	 * @param pNewPassword password to request before read the tag
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef enableWritePassword(const uint8_t* pCurrentWritePassword,
			uint8_t* pNewPassword);

	NFC_StatusTypeDef disableWritePassword(const uint8_t* pCurrentWritePassword=M24SR::DEFAULT_PASSWORD);

	/**
	 * @brief  This function deactivate the need of read and write password for next access
	 * @param	pSuperUserPassword I2C super user password to overwrite read and write password
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef disableAllPassword(const uint8_t* pSuperUserPassword=M24SR::DEFAULT_PASSWORD);

	/**
	 * @brief  This function enable read only mode
	 * @param	pCurrentWritePassword Write password is needed to have right to enable read only mode
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef enableReadOnly(const uint8_t* pCurrentWritePassword=M24SR::DEFAULT_PASSWORD);

	/**
	 * @brief  This fonction disable read only mode
	 * @param	pCurrentWritePassword Write password is needed to have right to disable read only mode
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef disableReadOnly();

	/**
	 * @brief  This fonction enable write only mode
	 * @param	pCurrentWritePassword Write password is needed to have right to enable write only mode
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef enableWriteOnly(const uint8_t* pCurrentWritePassword=M24SR::DEFAULT_PASSWORD);

	/**
	 * @brief  This fonction disable write only mode
	 * @param	pCurrentWritePassword Write password is needed to have right to disable write only mode
	 * @return return NFC_SUCCESS if no error happen
	 * @par the password must have a length of 16 char
	 */
	NFC_StatusTypeDef disableWriteOnly();

protected:

	/*** Protected Component Related Methods ***/

	NFC_StatusTypeDef M24SR_Init(NFC_InitTypeDef *);
	NFC_StatusTypeDef M24SR_ReadID(uint8_t *nfc_id);
	NFC_StatusTypeDef M24SR_GetSession(void);
	NFC_StatusTypeDef M24SR_KillSession(void);
	NFC_StatusTypeDef M24SR_Deselect(void);
	NFC_StatusTypeDef M24SR_SelectApplication(void);
	NFC_StatusTypeDef M24SR_SelectCCfile(void);
	NFC_StatusTypeDef M24SR_SelectNDEFfile(uint16_t NDEFfileId);
	NFC_StatusTypeDef M24SR_SelectSystemfile(void);
	NFC_StatusTypeDef M24SR_ReadBinary(uint16_t Offset, uint8_t NbByteToRead,
			uint8_t *pBufferRead);
	NFC_StatusTypeDef M24SR_STReadBinary(uint16_t Offset, uint8_t NbByteToRead,
			uint8_t *pBufferRead);
	NFC_StatusTypeDef M24SR_UpdateBinary(uint16_t Offset, uint8_t NbByteToWrite,
			uint8_t *pDataToWrite);
	NFC_StatusTypeDef M24SR_Verify(uint16_t uPwdId, uint8_t NbPwdByte,
			const uint8_t *pPwd);
	NFC_StatusTypeDef M24SR_ChangeReferenceData(uint16_t uPwdId, uint8_t *pPwd);
	NFC_StatusTypeDef M24SR_EnableVerificationRequirement(
			uint16_t uReadOrWrite);
	NFC_StatusTypeDef M24SR_DisableVerificationRequirement(
			uint16_t uReadOrWrite);
	NFC_StatusTypeDef M24SR_EnablePermanentState(uint16_t uReadOrWrite);
	NFC_StatusTypeDef M24SR_DisablePermanentState(uint16_t uReadOrWrite);
	NFC_StatusTypeDef M24SR_SendInterrupt(void);
	NFC_StatusTypeDef M24SR_StateControl(uint8_t uSetOrReset);
	NFC_StatusTypeDef M24SR_ManageI2CGPO(M24SR_GPO_MGMT GPO_I2Cconfig);
	NFC_StatusTypeDef M24SR_ManageRFGPO(M24SR_GPO_MGMT GPO_RFconfig);
	NFC_StatusTypeDef M24SR_RFConfig(uint8_t OnOffChoice);
	NFC_StatusTypeDef M24SR_FWTExtension(uint8_t FWTbyte);

	/**
	 * wait until the answer is not ready
	 */
	NFC_StatusTypeDef M24SR_IO_IsAnswerReady(void);
	
	/**
	 * send a command to the component
	 * @param NbByte lenght of the command
	 * @param pBuffer buffer containing the command
	 * @return NFC_SUCCESS if no error happen
	 */
	NFC_StatusTypeDef M24SR_IO_SendI2Ccommand(uint8_t NbByte, uint8_t *pBuffer);
	
	/**
	 * read a command renspose
	 * @param NbByte number of byte to read
	 * @param pBuffer buffer where store the rensponse
	 * @return NFC_SUCCESS if no error happen
	 */
	NFC_StatusTypeDef M24SR_IO_ReceiveI2Cresponse(uint8_t NbByte,
			uint8_t *pBuffer);

	/**
	 * do and active polling on the i2c bus until the answer is not ready
	 * @return NFC_SUCCESS if no error happen
	 */
	NFC_StatusTypeDef M24SR_IO_PollI2C(void);

	/**
	 * read the gpo pin
	 * @param[out] pPinState variable where store the pin state
	 */
	void M24SR_IO_GPO_ReadPin(GPIO_PinState *pPinState) {
		if (GPOPin == 0)
			(*pPinState) = GPIO_PIN_RESET;
		else
			(*pPinState) = GPIO_PIN_SET;
	}

	/**
	 * write the gpo pin
	 * @param pPinState state to write
	 */
	void M24SR_IO_RFDIS_WritePin(GPIO_PinState PinState) {
		if (PinState == GPIO_PIN_RESET)
			RFDisablePin = 0;
		else
			RFDisablePin = 1;
	}

	/**
	 * set how wait the command answer
	 * @param mode how to wait the i2c rensponse
	 */
	void M24SR_IO_SetI2CSynchroMode(M24SR_WAITINGTIME_MGMT mode) {
		syncMode = mode;
	}

	/**
	 * callback trigger when the chip finish to do a command
	 */
	void M24SR_AnswerReadyInterruptCallback() {
		interruptIsFired = true;
		answerReadyInterrupt.disable_irq();
	}

	/*** Component's Instance Variables ***/

	/* Identity */
	uint8_t who_am_i;

	/* Type. */
	uint8_t type;

	/* I2C address */
	uint8_t address;

	/* IO Device. */
	I2C &dev_i2c;

	/* GPIO */
	DigitalIn GPOPin;

	/**
	 * pin used for disable the rf chip functionality
	 */
	DigitalOut RFDisablePin;

	/**
	 * pin used as interrupt
	 */
	InterruptIn answerReadyInterrupt;

	/**
	 * method used for wait the chip response
	 */
	M24SR_WAITINGTIME_MGMT syncMode;

	/**
	 * buffer used for build the command to send to the chip
	 */
	uint8_t uM24SRbuffer[0xFF];

	/**
	 * ???
	 */
	uint8_t uDIDbyte;

	/**
	 * state variable change when the interrupt is fired
	 */
	volatile bool interruptIsFired;
	
	/**
	 * Object implementing the interface for use the NDefLib
	 */
	NDefNfcTagM24SR NDefTagUtil;
};

#endif // __M24SR_CLASS_H

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
