/**
 ******************************************************************************
 * @file    Nfc_class.h
 * @author  ST Central Labs
 * @version V0.0.1
 * @date    13-April-2015
 * @brief   This file contains the abstract class describing the interface of a
 *          nfc component.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Generated with Stm32CubeTOO -----------------------------------------------*/

/* Define to prevent from recursive inclusion --------------------------------*/

#ifndef __NFC_CLASS_H
#define __NFC_CLASS_H


/* Includes ------------------------------------------------------------------*/

#include "Component_class.h"


/* Classes  ------------------------------------------------------------------*/

/** An abstract class for Nfc components.
 */
class Nfc : public Component
{
public:

	/**
	 * possible password to set
	 */
	typedef enum{
		ReadPwd=READ_PWD,  //!< password to use before read the tag
		WritePwd=WRITE_PWD,//!< password to use before write the tag
		I2CPwd=I2C_PWD,    //!< root password, used only thought nfc
	}PasswordType_t;


	/**
	 * open a i2c connection with the tag if an RF connection isn't already open
	 * @return operation status
	 */
	virtual NFC_StatusTypeDef GetSession(void) = 0;
	/**
	 * force to open an i2c connection , abort the RF connection
	 * @return NFC_SUCCESS if the session is acquired
	 */
	virtual NFC_StatusTypeDef KillSession(void) = 0;

	/**
	 * close an i2c connection
	 * @return NFC_SUCCESS if the session is release
	 */
	virtual NFC_StatusTypeDef Deselect(void) = 0;

	/**
	 * Select the application file
	 * @return NFC_SUCCESS if the application is selected
	 */
	virtual NFC_StatusTypeDef SelectApplication(void) = 0;

	/**
	 * select the CC file
	 * @return NFC_SUCCESS if the cc file is selected
	 */
	virtual NFC_StatusTypeDef SelectCCfile(void) = 0;

	/**
	 * select the ndef file
	 * @param NDEFfileId file id to open
	 * @return NFC_SUCCESS if the file is selected
	 */
	virtual NFC_StatusTypeDef SelectNDEFfile(uint16_t NDEFfileId) = 0;

	/**
	 * select the system file
	 * @return NFC_SUCCESS if the system file is selected
	 */
	virtual NFC_StatusTypeDef SelectSystemfile(void) = 0;

	/**
	 * read data from the tag
	 * @param Offset offset where start to read
	 * @param NbByteToRead number of byte to read
	 * @param[out] pBufferRead buffer where store the read data
	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef ReadBinary(uint16_t Offset, uint8_t NbByteToRead, uint8_t *pBufferRead) = 0;

	/**
	 * write data to the tag
	 * @param Offset offset where start to write
	 * @param NbByteToWrite number of byte to write
	 * @param pDataToWrite buffer to write
	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef UpdateBinary(uint16_t Offset, uint8_t NbByteToWrite, uint8_t *pDataToWrite) = 0;

	/**
	 * check that the password is correct
	 * @param uPwdId password type
	 * @param NbPwdByte password size ( must be 0x10)
	 * @param pPwd password buffer
	 * @return NFC_SUCCESS if no error happen, and you have the permission to write/read
	 */
	virtual NFC_StatusTypeDef Verify(PasswordType_t uPwdId, uint8_t NbPwdByte,const uint8_t *pPwd) = 0;

	/**
	 * replaces the read or write password related to the NDEF files previously selected.
	 * @param uPwdId password to change
	 * @param pPwd new password
	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef ChangeReferenceData(PasswordType_t uPwdId,const uint8_t *pPwd) = 0;
	
	/**
	 *	activates the protection by password of the NDEF file.
	 *  When this command is successful, the read or write access to the NDEF file is protected by a 128-bit password.
	 *  @param uReadOrWrite password to requeire
	 *  @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef EnableVerificationRequirement(PasswordType_t uReadOrWrite) = 0;
	
	/**
     * deactivates the protection by password of the NDEF file.
     * When this command is successful, the read or write access to the NDEF file is granted
     * without security requirements.
	 * @param uReadOrWrite password to requeire
	 * @return NFC_SUCCESS if no error happen
	 */     
	virtual NFC_StatusTypeDef DisableVerificationRequirement(PasswordType_t uReadOrWrite) = 0;
	
	/**
	 * as {@link NFC#ReadBinary} but it doesn't give error if you read more byte than the ones 
	 * that are available
	 * @param Offset offset where start to read
	 * @param NbByteToRead number of byte to read
	 * @param[out] pBufferRead buffer where store the read data
	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef STReadBinary(uint16_t Offset, uint8_t NbByteToRead, uint8_t *pBufferRead) = 0;
	
	/**
	 * set the tag as read/write only and write it in the eeprom
	 * @param  uReadOrWrite ReadPwd if you need a write only, WritePwd if you need a read only
	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef EnablePermanentState(PasswordType_t uReadOrWrite) = 0;
	/**
	 * unset the tag as read/write only and write it in the eeprom
	 * @param  uReadOrWrite ReadPwd if you need a write only, WritePwd if you need a read only
	 * @return NFC_SUCCESS if no error happen
	 * @par you must be in i2csuper user state for call this function
	 */
	virtual NFC_StatusTypeDef DisablePermanentState(PasswordType_t uReadOrWrite) = 0;
	
	/**
	 * set the gpo output pin
	 * @param uSetOrReset new pin status
 	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef StateControl(uint8_t uSetOrReset) = 0;
	
	/**
	 * @brief  This function configure GPO purpose for I2C session
	 * @param	GPO_I2Cconfig: GPO configuration to set
 	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef ManageI2CGPO(uint8_t GPO_I2Cconfig) = 0;
	
	/**
 	* @brief  This function configure GPO purpose for RF session
 	* @param	GPO_RFconfig: GPO configuration to set
  	* @return NFC_SUCCESS if no error happen
 	*/
	virtual NFC_StatusTypeDef ManageRFGPO(uint8_t GPO_RFconfig) = 0;
	
	/**
	 * @brief  This function enable or disable RF communication
	 * @param	OnOffChoice: GPO configuration to set
  	 * @return NFC_SUCCESS if no error happen
	 */
	virtual NFC_StatusTypeDef RFConfig(uint8_t OnOffChoice) = 0;

	virtual ~Nfc(){};
};

#endif /* __NFC_CLASS_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
