/**
 ******************************************************************************
 * @file    led6001_class.h
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    December 9th, 2015
 * @brief   This file contains the class of an LED6001 component.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Generated with STM32CubeTOO -----------------------------------------------*/


/* Revision ------------------------------------------------------------------*/
/*
    Repository:       http://svn.x-nucleodev.codex.cro.st.com/svnroot/X-NucleoDev
    Branch/Trunk/Tag: trunk
    Based on:         X-CUBE-LED1/trunk/Drivers/BSP/Components/led6001/led6001.h
    Revision:         0
*/


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __LED6001_CLASS_H
#define __LED6001_CLASS_H


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/        
#include "mbed.h"
/* ACTION 2 ------------------------------------------------------------------*
 * Include here component specific header files.                              *
 *----------------------------------------------------------------------------*/        
#include "component.h"
/* ACTION 3 ------------------------------------------------------------------*
 * Include here interface specific header files.                              *
 *                                                                            *
 * Example:                                                                   *
 *   #include "../Interfaces/Humidity_class.h"                                *
 *   #include "../Interfaces/Temperature_class.h"                             *
 *----------------------------------------------------------------------------*/
#include "../Interfaces/Led_class.h"


/* Definitions ---------------------------------------------------------------*/

/* Period of PWM and Analog Dimming modes's PWM signal. */
#define PWM_PERIOD_ms          (10)

/* Duty Cycle of PWM and Analog Dimming modes's PWM signal to power the LED on. */
#define PWM_DIMMING_ON         (1.0f)

/* Duty Cycle of PWM and Analog Dimming modes's PWM signal to power the LED off. */
#define PWM_DIMMING_OFF        (0.0f)

/* Duty Cycle Range of Analog Dimming. */
#define MAX_PWM_ANALOG_DIMMING (1.2f/3.3f)
#define MIN_PWM_ANALOG_DIMMING (0.3f/3.3f)


/* Classes -------------------------------------------------------------------*/

/**
 * @brief Class representing a LED6001 component.
 */
class LED6001 : public Led
{
public:

    /*** Constructor and Destructor Methods ***/

    /**
     * @brief Constructor.
     * @param xfault_irq pin name of the XFAULT pin of the component.
     * @param current    pin name of the ADC pin of the component responsible
     *                   for sensing the current flowing through the LED.
     * @param pwm        pin name of the PWM pin of the component.
     * @param analog     pin name of the Analog pin of the component.
     */
    LED6001(PinName xfault_irq, PinName current, PinName pwm, PinName analog) : Led(), xfault_irq(xfault_irq), current(current), pwm(pwm), analog(analog)
    {
        /* ACTION 4 ----------------------------------------------------------*
         * Initialize here the component's member variables, one variable per *
         * line.                                                              *
         *                                                                    *
         * Example:                                                           *
         *   measure = 0;                                                     *
         *   instance_id = number_of_instances++;                             *
         *--------------------------------------------------------------------*/
        this->xfault_irq.mode(PullUp);
    }
    
    /**
     * @brief Destructor.
     */
    virtual ~LED6001(void) {}
    

    /*** Public Component Related Methods ***/

    /* ACTION 5 --------------------------------------------------------------*
     * Implement here the component's public methods, as wrappers of the C    *
     * component's functions.                                                 *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2).                                     *
     *                                                                        *
     * Example:                                                               *
     *   virtual int GetValue(float *f)  //(1)                                *
     *   {                                                                    *
     *     return COMPONENT_GetValue(float *f);                               *
     *   }                                                                    *
     *                                                                        *
     *   virtual int EnableFeature(void) //(2)                                *
     *   {                                                                    *
     *     return COMPONENT_EnableFeature();                                  *
     *   }                                                                    *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Initializing the component.
     * @param  init Pointer to device specific initalization structure.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int Init(void *init = NULL)
    {
        pwm.period_ms(PWM_PERIOD_ms);
        analog.period_ms(PWM_PERIOD_ms);
        return (int) COMPONENT_OK;
    }

    /**
     * @brief  Getting the ID of the component.
     * @param  id Pointer to an allocated variable to store the ID into.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int ReadID(uint8_t *id = NULL)
    {
        return (int) COMPONENT_OK;
    }

    /**
     * @brief  Getting the current flowing through the LED.
     * @param  None.
     * @retval The current flowing through the LED, represented as a floating
     *         point number in the range [0.0, 1.0].
     */
    virtual float GetCurrent(void)
    {
        return current.read();
    }

    /**
     * @brief  Setting PWM dimming value.
     * @param  dimming PWM dimming value, represented as a floating
     *         point number in the range [0.0, 1.0].
     * @retval None.
     */
    virtual void SetPWMDimming(float dimming)
    {
        if (dimming >= 0.0f && dimming <= 1.0f)
            pwm.write(dimming);
    }

    /**
     * @brief  Setting Analog dimming value.
     * @param  dimming Analog dimming value, represented as a floating
     *         point number in the range [0.0, 1.0].
     * @retval None.
     */
    virtual void SetAnalogDimming(float dimming)
    {
        if (dimming >= 0.0f && dimming <= 1.0f)
            analog.write(dimming * (MAX_PWM_ANALOG_DIMMING - MIN_PWM_ANALOG_DIMMING) + MIN_PWM_ANALOG_DIMMING);
    }

    /**
     * @brief  Powering ON at maximum light intensity.
     * @param  None.
     * @retval None.
     */
    virtual void PowerON(void)
    {
        pwm.write(PWM_DIMMING_ON);
        analog.write(PWM_DIMMING_ON);
    }

    /**
     * @brief  Powering OFF.
     * @param  None.
     * @retval None.
     */
    virtual void PowerOFF(void)
    {
        pwm.write(PWM_DIMMING_OFF);
        analog.write(PWM_DIMMING_OFF);
    }


    /*** Public Interrupt Related Methods ***/

    /* ACTION 6 --------------------------------------------------------------*
     * Implement here interrupt related methods, if any.                      *
     * Note that interrupt handling is platform dependent, e.g.:              *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq(pin);           //Interrupt object.        *
     *     feature_irq.fall(callback);             //Attach a callback.       *
     *     feature_irq.mode(PullNone);             //Set interrupt mode.      *
     *     feature_irq.enable_irq();               //Enable interrupt.        *
     *     feature_irq.disable_irq();              //Disable interrupt.       *
     *   + Arduino:                                                           *
     *     attachInterrupt(pin, callback, RISING); //Attach a callback.       *
     *     detachInterrupt(pin);                   //Detach a callback.       *
     *                                                                        *
     * Example (mbed):                                                        *
     *   void AttachFeatureIRQ(void (*fptr) (void))                           *
     *   {                                                                    *
     *     feature_irq.fall(fptr);                                            *
     *   }                                                                    *
     *                                                                        *
     *   void EnableFeatureIRQ(void)                                          *
     *   {                                                                    *
     *     feature_irq.enable_irq();                                          *
     *   }                                                                    *
     *                                                                        *
     *   void DisableFeatureIRQ(void)                                         *
     *   {                                                                    *
     *     feature_irq.disable_irq();                                         *
     *   }                                                                    *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Attaching an interrupt handler to the XFAULT interrupt.
     * @param  fptr An interrupt handler.
     * @retval None.
     */
    void AttachXFaultIRQ(void (*fptr)(void))
    {
        xfault_irq.fall(fptr);
    }
    
    /**
     * @brief  Enabling the XFAULT interrupt handling.
     * @param  None.
     * @retval None.
     */
    void EnableXFaultIRQ(void)
    {
        xfault_irq.enable_irq();
    }
    
    /**
     * @brief  Disabling the XFAULT interrupt handling.
     * @param  None.
     * @retval None.
     */
    void DisableXFaultIRQ(void)
    {
        xfault_irq.disable_irq();
    }


protected:

    /*** Protected Component Related Methods ***/

    /* ACTION 7 --------------------------------------------------------------*
     * Declare here the component's specific methods.                         *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2);                                     *
     *   + Helper methods, if any, like functions declared in the component's *
     *     source files but not pointed by the component's virtual table (3). *
     *                                                                        *
     * Example:                                                               *
     *   Status_t COMPONENT_GetValue(float *f);   //(1)                       *
     *   Status_t COMPONENT_EnableFeature(void);  //(2)                       *
     *   Status_t COMPONENT_ComputeAverage(void); //(3)                       *
     *------------------------------------------------------------------------*/


    /*** Component's I/O Methods ***/

    /* ACTION 8 --------------------------------------------------------------*
     * Implement here other I/O methods beyond those already implemented      *
     * above, which are declared extern within the component's header file.   *
     *------------------------------------------------------------------------*/


    /*** Component's Instance Variables ***/

    /* ACTION 9 --------------------------------------------------------------*
     * Declare here interrupt related variables, if needed.                   *
     * Note that interrupt handling is platform dependent, see                *
     * "Interrupt Related Methods" above.                                     *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq;                                           *
     *------------------------------------------------------------------------*/
    /* XFault Interrupt. */
    InterruptIn xfault_irq;

    /* ACTION 10 -------------------------------------------------------------*
     * Declare here other pin related variables, if needed.                   *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut standby_reset;                                          *
     *------------------------------------------------------------------------*/
    /* Pin to sense the current flowing through the LED. */
    AnalogIn current;

    /* Pulse Width Modulation dimming pin. */
    PwmOut pwm;

    /* Analog dimming pin. */
    PwmOut analog;

    /* ACTION 11 -------------------------------------------------------------*
     * Declare here communication related variables, if needed.               *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut address;                                                *
     *     DevI2C &dev_i2c;                                                   *
     *------------------------------------------------------------------------*/

    /* ACTION 12 -------------------------------------------------------------*
     * Declare here identity related variables, if needed.                    *
     * Note that there should be only a unique identifier for each component, *
     * which should be the "who_am_i" parameter.                              *
     *------------------------------------------------------------------------*/

    /* ACTION 13 -------------------------------------------------------------*
     * Declare here the component's static and non-static data, one variable  *
     * per line.                                                              *
     *                                                                        *
     * Example:                                                               *
     *   float measure;                                                       *
     *   int instance_id;                                                     *
     *   static int number_of_instances;                                      *
     *------------------------------------------------------------------------*/
};

#endif // __LED6001_CLASS_H

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
