/******************************************************//**
  * @file    STSpin220.h 
  * @author  IPC Rennes
  * @version V1.1.0
  * @date    May 26th, 2016
  * @brief   Header for STSpin220 driver (fully integrated microstepping motor driver)
  * @note    (C) COPYRIGHT 2016 STMicroelectronics
  ******************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __STSPIN220_H
#define __STSPIN220_H

#ifdef __cplusplus
 extern "C" {
#endif 

/* Includes ------------------------------------------------------------------*/
#include "STSpin220_config.h"
#include "motor_def.h"

/* Definitions ---------------------------------------------------------------*/

/** @addtogroup Components
 * @{
 */
    
/** @defgroup STSpin220
  * @{
  */

/** @defgroup STSpin220_Exported_Constants STSpin220 Exported Constants
  * @{
  */
/// Current FW major version
#define STSPIN220_FW_MAJOR_VERSION (uint8_t)(1)
/// Current FW minor version
#define STSPIN220_FW_MINOR_VERSION (uint8_t)(1)
/// Current FW patch version
#define STSPIN220_FW_PATCH_VERSION (uint8_t)(0)
/// Current FW version
#define STSPIN220_FW_VERSION       (uint32_t)((STSPIN220_FW_MAJOR_VERSION<<16)|\
                                              (STSPIN220_FW_MINOR_VERSION<<8)|\
                                              (STSPIN220_FW_PATCH_VERSION))

/// Max position
#define STSPIN220_MAX_POSITION           (0x7FFFFFFF)

/// Min position
#define STSPIN220_MIN_POSITION           (0x80000000)

/// Position range
#define STSPIN220_POSITION_RANGE         ((uint32_t)(STSPIN220_MAX_POSITION -\
                                                        STSPIN220_MIN_POSITION))
/// STSpin220 error base number
#define STSPIN220_ERROR_BASE             (0xA000)

/// run bit mask
#define STSPIN220_RUN_BIT_MASK           (0x01)

/// move bit mask
#define STSPIN220_MOVE_BIT_MASK          (0x02)

/// soft stop bit mask
#define STSPIN220_SOFT_STOP_BIT_MASK     (0x04)
   
/// direction change bit mask
#define STSPIN220_DIR_CHANGE_BIT_MASK    (0x08)

/// Maximum frequency of the step clock frequency in Hz
#define STSPIN220_MAX_STCK_FREQ          (10000)

/// Minimum frequency of the step clock frequency in Hz
#define STSPIN220_MIN_STCK_FREQ          (8)

/// Minimum duration of standby 
#define STANDBY_MIN_DURATION             (1)
    
/// Dead time after standby exit
#define AFTER_STANDBY_EXIT_DEAD_TIME     (1)

/// Reset delay to select step mode
#define SELECT_STEP_MODE_DELAY           (1)

/// PWM REF and bridges disable delay
#define DISABLE_DELAY                    (1)

/// Microstepping sequencer maximum value
#define SEQUENCER_MAX_VALUE              (uint16_t)(0x3FF)

/// MCU wait time after power bridges are enabled
#define BRIDGE_TURN_ON_DELAY                                     (10)

/// RC Filtering delay on the PWM
#define PWM_FILTER_TIME_CONSTANT                                 (5)
/**
  * @}
  */

/* Types ---------------------------------------------------------------------*/

/** @defgroup STSpin220_Exported_Types STSpin220 Exported Types
  * @{
  */

/** @defgroup Error_Types Error Types
  * @{
  */
/// Errors
typedef enum {
  STSPIN220_ERROR_SET_HOME         = STSPIN220_ERROR_BASE,      /// Error while setting home position
  STSPIN220_ERROR_SET_MAX_SPEED    = STSPIN220_ERROR_BASE + 1,  /// Error while setting max speed
  STSPIN220_ERROR_SET_MIN_SPEED    = STSPIN220_ERROR_BASE + 2,  /// Error while setting min speed
  STSPIN220_ERROR_SET_ACCELERATION = STSPIN220_ERROR_BASE + 3,  /// Error while setting acceleration
  STSPIN220_ERROR_SET_DECELERATION = STSPIN220_ERROR_BASE + 4,  /// Error while setting decelaration
  STSPIN220_ERROR_MCU_OSC_CONFIG   = STSPIN220_ERROR_BASE + 5,  /// Error while configuring mcu oscillator
  STSPIN220_ERROR_MCU_CLOCK_CONFIG = STSPIN220_ERROR_BASE + 6,  /// Error while configuring mcu clock
  STSPIN220_ERROR_POSITION         = STSPIN220_ERROR_BASE + 7,  /// Unexpected current position (wrong number of steps)
  STSPIN220_ERROR_SPEED            = STSPIN220_ERROR_BASE + 8,  /// Unexpected current speed
  STSPIN220_ERROR_INIT             = STSPIN220_ERROR_BASE + 9,  /// Unexpected number of devices or unexpected value for predefined parameter
  STSPIN220_ERROR_SET_DIRECTION    = STSPIN220_ERROR_BASE + 10, /// Error while setting direction
  STSPIN220_ERROR_SET_STEP_MODE    = STSPIN220_ERROR_BASE + 11, /// Attempt to set an unsupported step mode
  STSPIN220_ERROR_APPLY_SPEED      = STSPIN220_ERROR_BASE + 12, /// Error while applying speed
  STSPIN220_ERROR_SET_TORQUE       = STSPIN220_ERROR_BASE + 13, /// Error while setting torque
  STSPIN220_ERROR_STEP_CLOCK       = STSPIN220_ERROR_BASE + 14  /// Error related to step clock
}errorTypes_t;
/**
  * @}
  */

/** @defgroup Device_Commands Device Commands
  * @{
  */
/// Device commands 
typedef enum {
  NO_CMD              = 0x00, 
  RUN_CMD             = (STSPIN220_RUN_BIT_MASK),
  MOVE_CMD            = (STSPIN220_MOVE_BIT_MASK),
} deviceCommand_t;
/**
  * @}
  */

/** @defgroup Device_Parameters Device Parameters
  * @{
  */
/// Device Parameters Structure Type
typedef struct {
    /// accumulator used to store speed increase smaller than 1 pps
    volatile uint32_t accu;           
    /// Position in microstep according to current step mode
    volatile int32_t currentPosition;
    /// Position of sequencer
    volatile int16_t sequencerPosition;
    /// mark position in microstep (motor position control mode)
    volatile int32_t markPosition;
    /// position in microstep at the end of the accelerating phase
    volatile uint32_t endAccPos;      
    /// nb of in microstep performed from the beggining of the goto or the move command 
    volatile uint32_t relativePos;    
    /// position in microstep step at the start of the decelerating phase
    volatile uint32_t startDecPos;    
    /// nb of microstep steps to perform for the goto or move commands
    uint32_t stepsToTake;
    
    /// constant speed phase torque value (%)
    volatile uint8_t runTorque;
    /// acceleration phase torque value (%)
    volatile uint8_t accelTorque;
    /// deceleration phase torque value (%)
    volatile uint8_t decelTorque;
    /// holding phase torque value (%)
    volatile uint8_t holdTorque;
    /// current selected torque value
    volatile uint8_t currentTorque;
    /// torque update
    volatile bool updateTorque;
    /// PWM frequency used to generate REF voltage
    volatile uint32_t refPwmFreq;
    /// torque boost enable
    volatile bool torqueBoostEnable;
    /// torque boost speed threshold
    volatile uint16_t torqueBoostSpeedThreshold;
    
    /// acceleration in pps^2 
    volatile uint16_t acceleration;
    /// deceleration in pps^2
    volatile uint16_t deceleration;
    /// max speed in pps (speed use for goto or move command)
    volatile uint16_t maxSpeed;
    /// min speed in pps
    volatile uint16_t minSpeed;
    /// current speed in pps
    volatile uint16_t speed;
    
    /// command under execution
    volatile deviceCommand_t commandExecuted; 
    /// FORWARD or BACKWARD direction
    volatile motorDir_t direction;                 
    /// current state of the device
    volatile motorState_t motionState;
    /// current step mode
    volatile motorStepMode_t stepMode;
    /// latched step mode
    motorStepMode_t stepModeLatched;
    /// current stop mode
    motorStopMode_t stopMode;
    
}deviceParams_t; 
/**
  * @}
  */

/// Motor driver initialization structure definition  
typedef struct
{
  /// acceleration in pps^2
  uint16_t acceleration;
  /// deceleration in pps^2
  uint16_t deceleration;
  /// max speed in pps (speed use for goto or move command)
  uint16_t maxSpeed;
  /// min speed in pps
  uint16_t minSpeed;
  /// acceleration phase torque value (%)
  uint8_t accelTorque;
  /// deceleration phase torque value (%)
  uint8_t decelTorque;
  /// constant speed phase torque value (%)
  uint8_t runTorque;
  /// holding phase torque value (%)
  uint8_t holdTorque;
  /// torque boost enable
  bool torqueBoostEnable;
  /// torque boost speed threshold
  uint16_t torqueBoostSpeedThreshold;
  /// step mode
  motorStepMode_t stepMode;
  /// stop mode
  motorStopMode_t stopMode;
  /// PWM frequency used to generate REF voltage
  uint32_t vrefPwmFreq;
} STSpin220_init_t;
/**
  * @}
  */

/**
  * @}
  */

/* Functions --------------------------------------------------------*/

/** @defgroup MotorControl_Board_Linked_Functions MotorControl Board Linked Functions
  * @{
  */
///Delay of the requested number of milliseconds
extern void STSpin220_Board_Delay(uint32_t delay);
///Enable Irq
extern void STSpin220_Board_EnableIrq(void);
///Disable Irq
extern void STSpin220_Board_DisableIrq(void);
///Setting the Stck Timeout delay and attaching a callback function to it
extern void STSpin220_Board_TimStckSetFreq(uint16_t newFreq);
///Initialises the step clock pin level
extern void STSpin220_Board_TimStckInit(void);
///Stopping the Timeout
extern uint8_t STSpin220_Board_TimStckStop(volatile uint8_t *pToggleOdd);
///Set the duty cycle of the PwmOut used for the REF
extern void STSpin220_Board_PwmRefSetDutyCycle(uint8_t dutyCycle);
///Set the frequency of the PwmOut used for the REF
extern void STSpin220_Board_PwmRefSetFreq(uint32_t newFreq);
///Start the reference voltage pwm
extern void STSpin220_Board_PwmRefStart(void);
///Reset the STSpin220 reset pin
extern void STSpin220_Board_Releasereset(void);
///Set the STSpin220 reset pin 
extern void STSpin220_Board_reset(void);
///Set direction GPIO
extern void STSpin220_Board_SetDirectionGpio(uint8_t gpioState);
///Reset the STCK\MODE3 pin
extern void STSpin220_Board_StckMode3_reset(void);
///Set the STCK\MODE3 pin
extern void STSpin220_Board_StckMode3_Set(void);
///Enable the power bridges (leave the output bridges HiZ)
extern void STSpin220_Board_enable(void); 
///Disable the power bridges (leave the output bridges HiZ)    
extern void STSpin220_Board_disable(void); 
///Select the STSpin220 mode1, mode2, mode3 and mode4 pins levels
extern bool STSpin220_Board_SetModePins(uint8_t modePin1Level,\
  uint8_t modePin2Level,\
  uint8_t modePin3Level,\
  uint8_t modePin4Level);
///Select Full Step mode
extern void STSpin220_Board_SetFullStep(void);
///Unselect Full Step mode
extern void STSpin220_Board_UnsetFullStep(void);
/**
  * @}
  */

  /**
  * @}
  */

/**
  * @}
  */
  
#ifdef __cplusplus
  }
#endif

#endif /* #ifndef __STSPIN220_H */

/******************* (C) COPYRIGHT 2016 STMicroelectronics *****END OF FILE****/
