/**
 ******************************************************************************
 * @file    stspin220_class.h
 * @author  IPC Rennes
 * @version V1.0.0
 * @date    May 26th, 2016
 * @brief   This file contains the class of a STSPIN220 Motor Control component.
 * @note    (C) COPYRIGHT 2016 STMicroelectronics
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __STSPIN220_CLASS_H
#define __STSPIN220_CLASS_H


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/        
#include "mbed.h"

/* ACTION 2 ------------------------------------------------------------------*
 * Include here component specific header files.                              *
 *----------------------------------------------------------------------------*/        
#include "stspin220.h"
/* ACTION 3 ------------------------------------------------------------------*
 * Include here interface specific header files.                              *
 *                                                                            *
 * Example:                                                                   *
 *   #include "../Interfaces/Humidity_class.h"                                *
 *   #include "../Interfaces/Temperature_class.h"                             *
 *----------------------------------------------------------------------------*/
#include "../Interfaces/StepperMotor_class.h"


/* Classes -------------------------------------------------------------------*/

/**
 * @brief Class representing a STSPIN220 component.
 */
class STSPIN220 : public StepperMotor
{
public:
    /*** Constructor and Destructor Methods ***/
    /**
     * @brief Constructor.
     * @param fault_and_enable_pin  pin name of the EN\FAULT pin of the component.
     * @param stby_reset_pin        pin name of the STBY\RESET pin of the component.
     * @param direction_mode4_pin   pin name of the MODE4\DIR pin of the component.
     * @param mode1_pin             pin name of the MODE1 pin of the component.
     * @param mode2_pin             pin name of the MODE2 pin of the component.
     * @param stck_mode3_pin        pin name of the MODE3\STCK pin of the component.
     * @param pwm_ref_pin           pin name of the PWM connected to the REF pin of the component.
     * @param Monitor_pin           pin name for the step clock handler duration Monitoring.
     */
    STSPIN220(PinName fault_and_enable_pin,
              PinName stby_reset_pin,
              PinName direction_mode4_pin,
              PinName mode1_pin,
              PinName mode2_pin,
              PinName stck_mode3_pin,
              PinName pwm_ref_pin,
              PinName monitor_pin = NC) : StepperMotor(),
                                     fault_and_enable(fault_and_enable_pin),
                                     stby_reset(stby_reset_pin),
                                     direction_mode4(direction_mode4_pin),
                                     mode1(mode1_pin),
                                     mode2(mode2_pin),
                                     stck_mode3(stck_mode3_pin),
                                     pwm_ref(pwm_ref_pin),
                                     monitor(monitor_pin)
    {
        /* Checking stackability. */
        if (numberOfDevices!=0)
            error("Instantiation of the STSPIN220 component failed: it can't be stacked on itself.\r\n");

        /* ACTION 4 ----------------------------------------------------------*
         * Initialize here the component's member variables, one variable per *
         * line.                                                              *
         *                                                                    *
         * Example:                                                           *
         *   measure = 0;                                                     *
         *   instance_id = number_of_instances++;                             *
         *--------------------------------------------------------------------*/
        errorHandlerCallback = 0;
        memset(&devicePrm, 0, sizeof(devicePrm));
        deviceInstance = numberOfDevices++;
        
        fault_and_enable_pinName = fault_and_enable_pin;
    }
    
    /**
     * @brief Destructor.
     */
    virtual ~STSPIN220(void) {}
    

    /*** Public Component Related Methods ***/

    /* ACTION 5 --------------------------------------------------------------*
     * Implement here the component's public methods, as wrappers of the C    *
     * component's functions.                                                 *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2).                                     *
     *                                                                        *
     * Example:                                                               *
     *   virtual int GetValue(float *pData) //(1)                             *
     *   {                                                                    *
     *     return COMPONENT_GetValue(float *pfData);                          *
     *   }                                                                    *
     *                                                                        *
     *   virtual int EnableFeature(void) //(2)                                *
     *   {                                                                    *
     *     return COMPONENT_EnableFeature();                                  *
     *   }                                                                    *
     *------------------------------------------------------------------------*/

    /**
     * @brief Public functions inherited from the Component Class
     */

    /**
     * @brief  Initialize the component.
     * @param  init Pointer to device specific initalization structure.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int Init(void *init = NULL)
    {
        return (int) Stspin220_Init((void *) init);
    }

    /**
     * @brief  Getting the ID of the component.
     * @param  id Pointer to an allocated variable to store the ID into.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int ReadID(uint8_t *id = NULL)
    {
        return (int) Stspin220_ReadID((uint8_t *) id);
    }

    /**
     * @brief Public functions inherited from the StepperMotor Class
     */

    /**
     * @brief  Getting the value of the motor state .
     * @param  None.
     * @retval The motor state accoring to motorState_t in motor.h
     */
    virtual unsigned int GetStatus(void)
    {
        return (unsigned int) Stspin220_GetDeviceState();
    }

    /**
     * @brief  Getting the position.
     * @param  None.
     * @retval The position.
     */
    virtual signed int GetPosition(void)
    {
        return (signed int)Stspin220_GetPosition();
    }

    /**
     * @brief  Getting the marked position.
     * @param  None.
     * @retval The marked position.
     */
    virtual signed int GetMark(void)
    {
        return (signed int)Stspin220_GetMark();
    }

    /**
     * @brief  Getting the current speed in pps.
     * @param  None.
     * @retval The current speed in pps.
     */
    virtual unsigned int GetSpeed(void)
    {
        return (unsigned int)Stspin220_GetCurrentSpeed();
    }

    /**
     * @brief  Getting the maximum speed in pps.
     * @param  None.
     * @retval The maximum speed in pps.
     */
    virtual unsigned int GetMaxSpeed(void)
    {
        return (unsigned int)Stspin220_GetMaxSpeed();
    }

    /**
     * @brief  Getting the minimum speed in pps.
     * @param  None.
     * @retval The minimum speed in pps.
     */
    virtual unsigned int GetMinSpeed(void)
    {
        return (unsigned int)Stspin220_GetMinSpeed();
    }

    /**
     * @brief  Getting the acceleration in pps^2.
     * @param  None.
     * @retval The acceleration in pps^2.
     */
    virtual unsigned int GetAcceleration(void)
    {
        return (unsigned int)Stspin220_GetAcceleration();
    }

    /**
     * @brief  Getting the deceleration in pps^2.
     * @param  None.
     * @retval The deceleration in pps^2.
     */
    virtual unsigned int GetDeceleration(void)
    {
        return (unsigned int)Stspin220_GetDeceleration();
    }
    
    /**
     * @brief  Getting the direction of rotation.
     * @param  None.
     * @retval The direction of rotation.
     */
    virtual direction_t GetDirection(void)
    {
        if (Stspin220_GetDirection()!=BACKWARD)
        {
            return FWD;
        }
        else
        {
            return BWD;
        }
    }
    
    /**
     * @brief  Setting the current position to be the home position.
     * @param  None.
     * @retval None.
     */
    virtual void SetHome(void)
    {
        Stspin220_SetHome();
    }

    /**
     * @brief  Setting the current position to be the marked position.
     * @param  None.
     * @retval None.
     */
    virtual void SetMark(void)
    {
        Stspin220_SetMark();
    }

    /**
     * @brief  Setting the maximum speed in pps.
     * @param  speed The maximum speed in pps.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool SetMaxSpeed(unsigned int speed)
    {
        if (speed <= 0xFFFF)
        {
            return Stspin220_SetMaxSpeed((uint16_t) speed);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the minimum speed in pps.
     * @param  speed The minimum speed in pps.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool SetMinSpeed(unsigned int speed)
    {
        if (speed <= 0xFFFF)
        {
            return Stspin220_SetMinSpeed((uint16_t) speed);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the acceleration in pps^2.
     * @param  acceleration The acceleration in pps/s^2.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool SetAcceleration(unsigned int acceleration)
    {
        if (acceleration <= 0xFFFF)
        {
            return Stspin220_SetAcceleration((uint16_t) acceleration);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the deceleration in pps^2.
     * @param  deceleration The deceleration in pps^2.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool SetDeceleration(unsigned int deceleration)
    {
        if (deceleration <= 0xFFFF)
        {
            return Stspin220_SetDeceleration((uint16_t) deceleration);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the Step Mode.
     * @param  step_mode The Step Mode.
     * @retval "true" in case of success, "false" otherwise.
     * @note   step_mode can be one of the following:
     *           + STEP_MODE_FULL
     *           + STEP_MODE_WAVE
     *           + STEP_MODE_HALF
     *           + STEP_MODE_1_4
     *           + STEP_MODE_1_8
     *           + STEP_MODE_1_16
     */
    virtual bool SetStepMode(step_mode_t step_mode)
    {
        return Stspin220_SetStepMode((motorStepMode_t) step_mode);
    }

    /**
     * @brief  Going to a specified position.
     * @param  position The desired position.
     * @retval None.
     */
    virtual void GoTo(signed int position)
    {
        Stspin220_GoTo((int32_t)position);
    }

    /**
     * @brief  Going to the home position.
     * @param  None.
     * @retval None.
     */
    virtual void GoHome(void)
    {
        Stspin220_GoHome();
    }

    /**
     * @brief  Going to the marked position.
     * @param  None.
     * @retval None.
     */
    virtual void GoMark(void)
    {
        Stspin220_GoMark();
    }

    /**
     * @brief  Running the motor towards a specified direction.
     * @param  direction The direction of rotation.
     * @retval None.
     */
    virtual void Run(direction_t direction)
    {
        Stspin220_Run((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD));
    }

    /**
     * @brief  Moving the motor towards a specified direction for a certain number of steps.
     * @param  direction The direction of rotation.
     * @param  steps The desired number of steps.
     * @retval None.
     */
    virtual void Move(direction_t direction, unsigned int steps)
    {
        Stspin220_Move((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD), (uint32_t)steps);
    }

    /**
     * @brief  Stopping the motor through an immediate deceleration up to zero speed.
     * @param  None.
     * @retval None.
     */
    virtual void SoftStop(void)
    {
        Stspin220_SoftStop();
    }

    /**
     * @brief  Stopping the motor through an immediate infinite deceleration.
     * @param  None.
     * @retval None.
     */
    virtual void HardStop(void)
    {
        Stspin220_HardStop();
    }

    /**
     * @brief  Disabling the power bridge after performing a deceleration to zero.
     * @param  None.
     * @retval None.
     */
    virtual void SoftHiZ(void)
    {
        Stspin220_SetStopMode(HIZ_MODE);
        Stspin220_SoftStop();
    }

    /**
     * @brief  Disabling the power bridge immediately.
     * @param  None.
     * @retval None.
     */
    virtual void HardHiZ(void)
    {
        Stspin220_HardHiZ();
    }

    /**
     * @brief  Waiting while the motor is active.
     * @param  None.
     * @retval None.
     */
    virtual void WaitWhileActive(void)
    {
        Stspin220_WaitWhileActive();
    }  

    /**
     * @brief Public functions NOT inherited
     */
     
    /**
     * @brief  Attaching an error handler.
     * @param  fptr An error handler.
     * @retval None.
     */
    virtual void AttachErrorHandler(void (*fptr)(uint16_t error))
    {
        Stspin220_AttachErrorHandler((void (*)(uint16_t error)) fptr);
    }

    /**
     * @brief  Checks if the device is disabled or/and has an alarm flag set 
     * by reading the EN FAULT pin position.
     * @param  None.
     * @retval One if the EN FAULT pin is low (the device is disabled or/and 
     * has an alarm flag set), otherwise zero.
     */
    virtual unsigned int CheckStatusHw(void)
    {
        if (!fault_and_enable.read()) return 0x01;
        else return 0x00;
    }
    
    /**
     * @brief  Disabling the device.
     * @param  None.
     * @retval None.
     */
    virtual void Disable(void)
    {
        Stspin220_Disable();
    }
    
    /**
     * @brief  Enabling the device.
     * @param  None.
     * @retval None.
     */
    virtual void Enable(void)
    {
        Stspin220_Enable();
    }
    
    /**
     * @brief  Exit STSPIN220 device from standby (low power consumption) by
     * setting STBY\RESET pin to high level and changing the motion state to
     * INACTIVE.
     * @param  None.
     * @retval None.
     */    
    virtual void ExitDeviceFromStandby(void)
    {
        Stspin220_ExitDeviceFromStandby();
    }
    
    /**
     * @brief  Set the frequency of the VREFA and VREFB PWM
     * @param  frequency in Hz
     * @retval None.
     */  
    virtual uint32_t GetFreqVrefPwm(void)
    {
        return Stspin220_VrefPwmGetFreq();
    }    
    
    /**
     * @brief  Getting the version of the firmware.
     * @param  None.
     * @retval The version of the firmware.
     */
    virtual unsigned int GetFwVersion(void)
    {
        return (unsigned int) Stspin220_GetFwVersion();
    }
    
    /**
     * @brief  Getting the motor step mode.
     * @param  None.
     * @retval The motor step mode.
     */
    virtual step_mode_t GetStepMode(void)
    {
      return (step_mode_t) Stspin220_GetStepMode();
    }
    
    /**
     * @brief  Getting the motor stop mode.
     * @param  None.
     * @retval The motor stop mode.
     */  
    virtual motorStopMode_t GetStopMode(void)
    {
      return Stspin220_GetStopMode();
    }
    
    /**
     * @brief  Get the motor torque.
     * @param  torqueMode Torque mode as specified in enum motorTorqueMode_t
     * @retval the torqueValue in % (from 0 to 100)
     */       
    virtual uint8_t GetTorque(motorTorqueMode_t torqueMode)
    {
        return Stspin220_GetTorque(torqueMode);
    }
    
    /**
     * @brief  Get the torque boost feature status.
     * @param  None.
     * @retval true if enabled, false if disabled
     */       
    virtual bool GetTorqueBoostEnable(void)
    {
        return Stspin220_GetTorqueBoostEnable();
    }
    
    /**
     * @brief  Get the torque boost threshold
     * @retval The torque boost threshold above which the step mode is
     * changed to full step
     */       
    virtual uint16_t GetTorqueBoostThreshold(void)
    {
        return Stspin220_GetTorqueBoostThreshold();
    }
    
    /**
     * @brief  Going to a specified position with a specificied direction.
     * @param  direction The desired direction.
     * @param  position The desired position.
     * @retval None.
     */
    virtual void GoTo(direction_t direction, signed int position)
    {
        Stspin220_GoToDir((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD),(int32_t)position);
    }

    /**
     * @brief  Put STSPIN220 device in standby (low power consumption) by 
     * setting STBY\RESET pin to low level and changing the motion state to
     * STANDBY.
     * @param  None.
     * @retval None.
     */    
    virtual void PutDeviceInStandby(void)
    {
        Stspin220_PutDeviceInStandby();
    }

    /**
     * @brief  Release the STSPIN220 reset (Reset pin set to high level).
     * @param  None.
     * @retval None.
     */    
    virtual void ReleaseReset(void)
    {
        Stspin220_Board_ReleaseReset();
    }
    
    /**
     * @brief  Reset the STSPIN220 (Reset pin set to low level).
     * @param  None.
     * @retval None.
     */    
    virtual void Reset(void)
    {
        Stspin220_Board_Reset();
    }

    /**
     * @brief  Set the motor direction.
     * @param  direction The desired direction.
     * @retval None.
     */
    virtual void SetDirection(direction_t direction)
    { 
        Stspin220_SetDirection((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD));
    }

    /**
     * @brief  Set the frequency of the PWM for REF pin
     * @param  frequency in Hz
     * @retval None.
     */  
    virtual void SetFreqVrefPwm(uint32_t frequency)
    {
        Stspin220_VrefPwmSetFreq(frequency);
    }

    /**
     * @brief  Set the motor stop mode.
     * @param  stopMode The desired stop mode (HOLD_MODE, HIZ_MODE or
     * STANDBY_MODE).
     * @retval None.
     */       
    virtual void SetStopMode(motorStopMode_t stopMode)
    {
        Stspin220_SetStopMode(stopMode);
    }

    /**
     * @brief  Set the motor torque.
     * @param  torqueMode Torque mode as specified in enum motorTorqueMode_t
     * @param  torqueValue in % (from 0 to 100)
     * @retval None.
     */       
    virtual void SetTorque(motorTorqueMode_t torqueMode, uint8_t torqueValue)
    {
        Stspin220_SetTorque(torqueMode, torqueValue);
    }

    /**
     * @brief  Enable or disable the motor torque boost feature.
     * @param  enable enable true to enable torque boost, false to disable
     * @retval None.
     */       
    virtual void SetTorqueBoostEnable(bool enable)
    {
        Stspin220_SetTorqueBoostEnable(enable);
    }
    
    /**
     * @brief Set the torque boost threshold
     * @param[in] speedThreshold speed threshold above which the step mode is
     * changed to full step
     * @retval None.
     */       
    virtual void SetTorqueBoostThreshold(uint16_t speedThreshold)
    {
        Stspin220_SetTorqueBoostThreshold(speedThreshold);
    }

    /*** Public Interrupt Related Methods ***/

    /* ACTION 6 --------------------------------------------------------------*
     * Implement here interrupt related methods, if any.                      *
     * Note that interrupt handling is platform dependent, e.g.:              *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq(pin); //Interrupt object.                  *
     *     feature_irq.rise(callback);   //Attach a callback.                 *
     *     feature_irq.mode(PullNone);   //Set interrupt mode.                *
     *     feature_irq.enable_irq();     //Enable interrupt.                  *
     *     feature_irq.disable_irq();    //Disable interrupt.                 *
     *   + Arduino:                                                           *
     *     attachInterrupt(pin, callback, RISING); //Attach a callback.       *
     *     detachInterrupt(pin);                   //Detach a callback.       *
     *                                                                        *
     * Example (mbed):                                                        *
     *   void AttachFeatureIRQ(void (*fptr) (void))                           *
     *   {                                                                    *
     *     feature_irq.rise(fptr);                                            *
     *   }                                                                    *
     *                                                                        *
     *   void EnableFeatureIRQ(void)                                          *
     *   {                                                                    *
     *     feature_irq.enable_irq();                                          *
     *   }                                                                    *
     *                                                                        *
     *   void DisableFeatureIRQ(void)                                         *
     *   {                                                                    *
     *     feature_irq.disable_irq();                                         *
     *   }                                                                    *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Attaching an interrupt handler to the FLAG interrupt.
     * @param  fptr An interrupt handler.
     * @retval None.
     */
     
    void AttachFlagIRQ(void (*fptr)(void))
    {
        fault_and_enable_irqFunctionPointer = &(*fptr);
        fault_and_enable.fall(fptr);
        fault_and_enable.mode(PullDown);
        wait_ms(1);
    }
    
    /**
     * @brief  Enabling the FLAG interrupt handling.
     * @param  None.
     * @retval None.
     */
    void EnableFlagIRQ(void)
    {
        fault_and_enable.enable_irq();
    }
    
protected:

    /*** Protected Component Related Methods ***/

    /* ACTION 7 --------------------------------------------------------------*
     * Declare here the component's specific methods.                         *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2);                                     *
     *   + Helper methods, if any, like functions declared in the component's *
     *     source files but not pointed by the component's virtual table (3). *
     *                                                                        *
     * Example:                                                               *
     *   Status_t COMPONENT_GetValue(float *f);   //(1)                       *
     *   Status_t COMPONENT_EnableFeature(void);  //(2)                       *
     *   Status_t COMPONENT_ComputeAverage(void); //(3)                       *
     *------------------------------------------------------------------------*/
    Status_t Stspin220_Init(void *init);
    Status_t Stspin220_ReadID(uint8_t *id);
    void Stspin220_AttachErrorHandler(void (*callback)(uint16_t error));
    void Stspin220_ApplyTorque(motorTorqueMode_t torqueMode);
    void Stspin220_Disable(void);
    void Stspin220_ErrorHandler(uint16_t error);
    void Stspin220_Enable(void);
    void Stspin220_ExitDeviceFromStandby(void);
    uint16_t Stspin220_GetAcceleration(void);
    uint16_t Stspin220_GetCurrentSpeed(void);
    uint16_t Stspin220_GetDeceleration(void);
    motorState_t Stspin220_GetDeviceState(void);
    motorDir_t Stspin220_GetDirection(void);
    uint32_t Stspin220_GetFwVersion(void);
    int32_t Stspin220_GetMark(void);
    uint16_t Stspin220_GetMaxSpeed(void);
    uint16_t Stspin220_GetMinSpeed(void);
    int32_t Stspin220_GetPosition(void);
    motorStepMode_t Stspin220_GetStepMode(void);
    motorStopMode_t Stspin220_GetStopMode(void);
    uint8_t Stspin220_GetTorque(motorTorqueMode_t torqueMode);
    bool Stspin220_GetTorqueBoostEnable(void);
    uint16_t Stspin220_GetTorqueBoostThreshold(void);
    void Stspin220_GoHome(void);  
    void Stspin220_GoMark(void);
    void Stspin220_GoTo(int32_t targetPosition);
    void Stspin220_GoToDir(motorDir_t direction, int32_t targetPosition);   
    void Stspin220_HardHiZ(void);
    void Stspin220_HardStop(void);
    void Stspin220_Move(motorDir_t direction, uint32_t stepCount);
    void Stspin220_PutDeviceInStandby(void);
    void Stspin220_Run(motorDir_t direction);
    bool Stspin220_SetAcceleration(uint16_t newAcc);
    bool Stspin220_SetDeceleration(uint16_t newDec);
    void Stspin220_SetDirection(motorDir_t direction);
    void Stspin220_SetHome(void);
    void Stspin220_SetMark(void);
    bool Stspin220_SetMaxSpeed(uint16_t volatile newSpeed);
    bool Stspin220_SetMinSpeed(uint16_t volatile newSpeed);    
    bool Stspin220_SetStepMode(motorStepMode_t stepMode);
    void Stspin220_SetStopMode(motorStopMode_t stopMode);
    bool Stspin220_SoftStop(void);
    void Stspin220_SetTorque(motorTorqueMode_t torqueMode, uint8_t torqueValue);
    void Stspin220_SetTorqueBoostEnable(bool enable);
    void Stspin220_SetTorqueBoostThreshold(uint16_t speedThreshold);
    uint32_t Stspin220_VrefPwmGetFreq(void);
    void Stspin220_VrefPwmSetFreq(uint32_t newFreq);
    void Stspin220_WaitWhileActive(void);
    
    /*** Functions intended to be used only internally ***/
    void Stspin220_ApplySpeed(uint16_t newSpeed);
    void Stspin220_ComputeSpeedProfile(uint32_t nbSteps);    
    void Stspin220_SetDeviceParamsToGivenValues(Stspin220_Init_t* pInitDevicePrm);
    void Stspin220_SetDeviceParamsOtherValues(void);    
    void Stspin220_SetDeviceParamsToPredefinedValues(void);
    bool Stspin220_SetStepModeWithoutReset(motorStepMode_t stepMode);
    void Stspin220_StartMovement(void);
    void Stspin220_StepClockHandler(void);
    
    /*** Component's I/O Methods ***/

    /* ACTION 8 --------------------------------------------------------------*
     * Implement here other I/O methods beyond those already implemented      *
     * above, which are declared extern within the component's header file.   *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Making the CPU wait.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_Delay(uint32_t delay)
    {
        wait_ms(delay);
    }
    
    /**
     * @brief  Disable the power bridges (leave the output bridges HiZ).
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_Disable(void)
    {
        fault_and_enable.disable_irq();
        DigitalOut fault_and_enable(fault_and_enable_pinName);
        fault_and_enable.write(0);
    }

    /**
     * @brief  Disabling interrupts.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_DisableIrq(void)
    {
        __disable_irq();
    }
    
    /**
     * @brief  Enable the power bridges (leave the output bridges HiZ).
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_Enable(void)
    {
        DigitalOut fault_and_enable_do(fault_and_enable_pinName);
        fault_and_enable_do.write(1);
        fault_and_enable.fall(fault_and_enable_irqFunctionPointer);
        fault_and_enable.mode(PullUp);
        wait_ms(1);
        fault_and_enable.enable_irq();
    }
    
    /**
     * @brief  Enabling interrupts.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_EnableIrq(void)
    {
        __enable_irq();
    }

    /**
     * @brief  Set the duty cycle of the PwmOut used for the REF
     * reference voltage generation and actually start the pwm if the duty cycle
     * is not zero.
     * @param[in] dutyCycle 0 - 100%
     * @retval None
     */
    void Stspin220_Board_PwmRefSetDutyCycle(uint8_t dutyCycle)
    {
        pwm_ref.write(((float)(100-dutyCycle))/100);
    }

    /**
     * @brief  Set the frequency of the PwmOut used for the REF
     * reference voltage generation.
     * @param[in] newFreq in Hz.
     * @retval None.
     */
    void Stspin220_Board_PwmRefSetFreq(uint32_t newFreq)
    {
        pwm_ref.period_us(1000000.0f/newFreq);
    }
    
    /**
     * @brief  Start the PwmOut for the REF pin.
     * @param[in] frequency frequency of the PwmOut used to generate the REF
     * reference voltage for the bridges.
     * @param[in] torqueMode Torque mode as specified in enum motorTorqueMode_t    
     * @retval None.
     */ 
    void Stspin220_Board_PwmRefStart(uint32_t frequency, motorTorqueMode_t torqueMode)
    { 
        /* Setting the period of the PwmOut. */
        pwm_ref.period_us(1000000.0f/frequency);
        /* Setting the duty cycle and actually starting the of the PwmOut. */
        Stspin220_ApplyTorque(torqueMode);
        wait_ms(5*PWM_FILTER_TIME_CONSTANT);
    }

    /**
     * @brief  Exit the device from standby reset mode.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_ReleaseReset(void)
    {
        stby_reset = 1;
    }

    /**
     * @brief  Put the device in standby reset mode.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_Reset(void)
    {
        stby_reset = 0;
    }

    /**
     * @brief  Set the DIR\MODE4 pin.
     * @param  dir motor direction FORWARD or BACKWARD.
     * @retval None.
     */
    void Stspin220_Board_SetDirectionGpio(motorDir_t dir)
    {
        if (dir==BACKWARD) direction_mode4 = 0;
        else direction_mode4 = 1;
    }
 
    /**
     * @brief Select Full Step mode
     * @param  None.
     * @retval None
     */
    void Stspin220_Board_SetFullStep(void)
    {
        mode1 = 0;
        mode2 = 0;
    }
    
    /**
     * @brief Select the STSPIN220 mode1, mode2, mode3 and mode4 pins levels.
     * @param[in] modePin1Level level of the mode1 gpio (0 low, 1+ high)
     * @param[in] modePin2Level level of the mode2 gpio (0 low, 1+ high)
     * @param[in] modePin3Level level of the mode3 gpio (0 low, 1+ high)
     * @param[in] modePin4Level level of the mode4 gpio (0 low, 1+ high)
     * @retval 1
     */
    uint8_t Stspin220_Board_SetModePins(uint8_t modePin1Level,\
        uint8_t modePin2Level,\
        uint8_t modePin3Level,\
        uint8_t modePin4Level)
    {
        mode1 = modePin1Level;
        mode2 = modePin2Level;
        stck_mode3 = modePin3Level;
        direction_mode4 = modePin4Level;
        return 1;
    }

    /**
     * @brief  Reset the STCK\MODE3 pin.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_StckMode3_Reset(void)
    {
        stck_mode3 = 0;
    }

    /**
     * @brief  Set the STCK\MODE3 pin.
     * @param  None.
     * @retval None.
     */
    void Stspin220_Board_StckMode3_Set(void)
    {
        stck_mode3 = 1;
    }

    /**
     * @brief Initialises the step clock pin level
     * @param  None.
     * @retval None
     */
    void Stspin220_Board_TimStckInit(bool check)
    {
        if (monitor.is_connected()) monitor = 0;
        if (check==false) stck_mode3 = 0;
        else if (stck_mode3==1) Stspin220_ErrorHandler(STSPIN220_ERROR_STEP_CLOCK);
    }

    /**
     * @brief  Setting the Stck Timeout delay
     * and attaching a callback function to it.
     * @param  frequency The frequency corresponding to the delay.
     * @retval None.
     */
    void Stspin220_Board_TimStckSetFreq(uint16_t newFreq)
    {
        /* Computing the delay of the Timeout. */
        float delay_us = (1000000.0f / 2 )/ newFreq;
        
        /* Attaching a function which updates */
        /* the state machine after the elapsed period_us time. */
        tim_stck.attach_us(this, &STSPIN220::Stspin220_StepClockHandler, delay_us);
    }

    void Stspin220_Board_Monitor_Set(void)
    {
        if (monitor.is_connected()) monitor = 1;
    }
    
    void Stspin220_Board_Monitor_Reset(void)
    {
        if (monitor.is_connected()) monitor = 0;
    }

    /**
     * @brief  Stopping the Timeout.
     * @param  None.
     * @retval None.
     */
    uint8_t Stspin220_Board_TimStckStop(volatile uint8_t *pToggleOdd)
    {
        __disable_irq();
        if (*pToggleOdd == 1)
        {
            __enable_irq();
            return 1;
        }
        if (stck_mode3 != 0)
        {
            __enable_irq();
            return 0;
        }
        tim_stck.detach();
        return 1;
    }

    /**
     * @brief Unselect Full Step mode
     * @param  None.
     * @retval None
     */
    void Stspin220_Board_UnsetFullStep(void)
    {
        mode1 = 1;
    }

protected:

    /*** Component's Instance Variables ***/

    /* ACTION 9 --------------------------------------------------------------*
     * Declare here interrupt related variables, if needed.                   *
     * Note that interrupt handling is platform dependent, see                *
     * "Interrupt Related Methods" above.                                     *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq;                                           *
     *------------------------------------------------------------------------*/
    /* Fault Interrupt and chip enable. */
    InterruptIn fault_and_enable;
    
    /* Interrupt to toggle the MODE3\STCK pin */
    Timeout tim_stck;

    /* ACTION 10 -------------------------------------------------------------*
     * Declare here other pin related variables, if needed.                   *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut standby_reset;                                          *
     *------------------------------------------------------------------------*/
    /* STBY\RESET pin. */
    DigitalOut stby_reset;
    /* MODE4\DIR pin. */
    DigitalOut direction_mode4;
    /* MODE1 pin */
    DigitalOut mode1;
    /* MODE2 pin */
    DigitalOut mode2;
    /* MODE3\STCK pin. */
    DigitalOut stck_mode3;

    /* Pulse Width Modulation pin for REF pin */
    PwmOut pwm_ref;
 
    /* Monitoring of step clock handler duration */
    DigitalOut monitor;

    /* fault and enable pin */
    PinName fault_and_enable_pinName;
    
    /* ACTION 11 -------------------------------------------------------------*
     * Declare here communication related variables, if needed.               *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut ssel;                                                   *
     *     DevSPI &dev_spi;                                                   *
     *------------------------------------------------------------------------*/
    /* Configuration. */

    /* IO Device. */

    /* ACTION 12 -------------------------------------------------------------*
     * Declare here identity related variables, if needed.                    *
     * Note that there should be only a unique identifier for each component, *
     * which should be the "who_am_i" parameter.                              *
     *------------------------------------------------------------------------*/
    /* Identity */
    uint8_t who_am_i;

    /* ACTION 13 -------------------------------------------------------------*
     * Declare here the component's static and non-static data, one variable  *
     * per line.                                                              *
     *                                                                        *
     * Example:                                                               *
     *   float measure;                                                       *
     *   int instance_id;                                                     *
     *   static int number_of_instances;                                      *
     *------------------------------------------------------------------------*/
    /* Data. */
    void (*fault_and_enable_irqFunctionPointer)(void);
    void (*errorHandlerCallback)(uint16_t error);
    deviceParams_t devicePrm;
    uint8_t deviceInstance;
    volatile uint8_t toggleOdd;

    /* Static data. */
    static uint8_t numberOfDevices;

public:

    /* Static data. */
    
};

#endif // __STSPIN220_CLASS_H

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
