/**
 ******************************************************************************
 * @file    StepperMotor_class.h
 * @author  AST
 * @version V1.0.0
 * @date    November 12th, 2015
 * @brief   This file contains the abstract class describing the interface of a
 *          steppermotor component.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent from recursive inclusion --------------------------------*/

#ifndef __STEPPERMOTOR_CLASS_H
#define __STEPPERMOTOR_CLASS_H


/* Includes ------------------------------------------------------------------*/

#include <Component_class.h>


/* Classes  ------------------------------------------------------------------*/

/** An abstract class for StepperMotor components.
 */
class StepperMotor : public Component
{
public:
    /** Rotation modes. */
    typedef enum
    {
      BWD = 0, /* Backward. */
      FWD = 1  /* Forward. */
    } direction_t;

    /**
     * @brief Getting the status.
     * @param  None.
     * @retval None.
     */
    virtual unsigned int GetStatus(void) = 0;

   /**
     * @brief  Getting a parameter.
     * @param  parameter register adress.
     * @retval a register value.
     */
    virtual unsigned int GetParameter(unsigned int parameter) = 0;

    /* Return the current position. */
    virtual signed int GetPosition(void) = 0;

    /* Return the mark position. */
    virtual signed int GetMark(void) = 0;

    /* Return the current speed in pps. */
    virtual unsigned int GetSpeed(void) = 0;

    /* Return the maximum speed in pps. */
    virtual unsigned int GetMaxSpeed(void) = 0;

    /* Return the minimum speed in pps. */
    virtual unsigned int GetMinSpeed(void) = 0;

    /* Return the acceleration in pps^2. */
    virtual unsigned int GetAcceleration(void) = 0;

    /* Return the deceleration in pps^2. */
    virtual unsigned int GetDeceleration(void) = 0;

    /* Get the current direction of rotation. */
    virtual direction_t GetDirection(void) = 0;

   /**
     * @brief  Setting a parameter.
     * @param  parameter register adress.
     * @param  value value to be set.
     * @retval None.
     */
    virtual void SetParameter(unsigned int parameter, unsigned int value) = 0;

    /* Set the current position to be the home position. */
    virtual void SetHome(void) = 0;

    /* Set the current position to be the mark position. */
    virtual void SetMark(void) = 0;

    /* Set the maximum speed in pps. */
    virtual void SetMaxSpeed(unsigned int speed) = 0;

    /* Set the minimum speed in pps. */
    virtual void SetMinSpeed(unsigned int speed) = 0;

    /* Set the acceleration in pps^2. */
    virtual void SetAcceleration(unsigned int acceleration) = 0;

    /* Set the deceleration in pps^2. */
    virtual void SetDeceleration(unsigned int deceleration) = 0;

    /* Go to the specified position. */
    virtual void GoTo(signed int position) = 0;

    /* Go to the home position. */
    virtual void GoHome(void) = 0;

    /* Go to the marked position. */
    virtual void GoMark(void) = 0;

    /* Run the motor towards a specified direction. */
    virtual void Run(direction_t direction) = 0;

    /* Run the motor towards a specified direction for a specified number of steps. */
    virtual void Move(direction_t direction, unsigned int steps) = 0;

    /* Stop the motor. */
    virtual void SoftStop(void) = 0;

    /* Stop the motor and disable the power bridge. */
    virtual void HardStop(void) = 0;

    /* Wait while the motor is active. */
    virtual void WaitWhileActive(void) = 0;
};

#endif /* __STEPPERMOTOR_CLASS_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
