/**
 ******************************************************************************
 * @file    PDM2PCMAudio_class.cpp
 * @author  AST / Software Platforms and Cloud
 * @version V1.0
 * @date    November 10th, 2016
 * @brief   Implementation file for the PDM2PCMAudio conversion library.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

#include "PDM2PCMAudio.h"


/* Variables -----------------------------------------------------------------*/

/* Demux filter. */
const uint8_t PDM2PCMAudio::_demux_filter[DEMUX_FILTER_SIZE] =
{
    0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
    0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
    0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
    0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
    0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
    0x00, 0x01, 0x00, 0x01, 0x02, 0x03, 0x02, 0x03,
    0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
    0x04, 0x05, 0x04, 0x05, 0x06, 0x07, 0x06, 0x07,
    0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
    0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
    0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
    0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
    0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
    0x08, 0x09, 0x08, 0x09, 0x0a, 0x0b, 0x0a, 0x0b,
    0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f,
    0x0c, 0x0d, 0x0c, 0x0d, 0x0e, 0x0f, 0x0e, 0x0f
};


/* Methods -------------------------------------------------------------------*/

/**
 * @brief  Getting number of PCM samples from nummber of PDM samples.
 * @param  PCM_samples Number of PCM samples.
 * @retval Number of equivalent PDM samples.
 */
uint32_t PDM2PCMAudio::pcm2pdm_samples(uint32_t PCM_samples)
{
    return PCM_samples * (_decimation_factor >> 4);
}

/**
 * @brief  Converting audio data from PDM to PCM.
 * @param  output_buffer     Pointer to output PCM buffer data.
 * @param  input_buffer      Pointer to input PDM buffer data.
 * @param  volume            Volume level (it must be in the range [0..PDM2PCM_MAX_VOLUME]).
 * @retval COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
 */
status_t PDM2PCMAudio::convert(int16_t *output_buffer, uint16_t *input_buffer, uint32_t volume)
{
    if (!(volume <= PDM2PCM_MAX_VOLUME))
        error("Volume level not supported: it must be in the range [0..%d].\r\n", PDM2PCM_MAX_VOLUME);

#ifdef PDM2PCM_AUDIO_DEBUG
                _pdm2pcm_audio_signal = 1;
#endif
    switch (_decimation_factor)
    {
        case 64:
            for (uint32_t index = 0; index < _channels; index++) {
#ifdef USE_OPEN_PDM2PCM_LIBRARY
                Open_PDM_Filter_64(&((uint8_t *) input_buffer)[index], (uint16_t *) &(output_buffer[index]), volume, (TPDMFilter_InitStruct *) &_PDM2PCM_filter[index]);
#else                
                PDM_Filter_64_LSB(&((uint8_t *) input_buffer)[index], (uint16_t *) &(output_buffer[index]), volume, (PDMFilter_InitStruct *) &_PDM2PCM_filter[index]);
#endif
            }
            break;

        case 128:
            for (uint32_t index = 0; index < _channels; index++) {
#ifdef USE_OPEN_PDM2PCM_LIBRARY
                Open_PDM_Filter_128(&((uint8_t *) input_buffer)[index], (uint16_t *) &(output_buffer[index]), volume, (TPDMFilter_InitStruct *) &_PDM2PCM_filter[index]);
#else                
                PDM_Filter_128_LSB(&((uint8_t *) input_buffer)[index], (uint16_t *) &(output_buffer[index]), volume, (PDMFilter_InitStruct *) &_PDM2PCM_filter[index]);
#endif
            }
            break;

        default:
            error("Decimation factor not supported: it must be either 64 or 128.\r\n");
    }
#ifdef PDM2PCM_AUDIO_DEBUG
                _pdm2pcm_audio_signal = 0;
#endif

    return COMPONENT_OK;
}

/**
 * @brief  Scrambling audio data.
 * @param  output_buffer Pointer to output PDM buffer data.
 * @param  input_buffer  Pointer to input PDM buffer data.
 * @param  size          Size of the buffers (thay has to be equally sized).
 * @retval COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
 */
status_t PDM2PCMAudio::scramble(uint16_t *output_buffer, uint16_t *input_buffer, uint32_t size)
{
    for (uint32_t index = 0; index < size; index++) {
        output_buffer[index] = HTONS(input_buffer[index]);
    }

    return COMPONENT_OK;
}

/**s
 * @brief  Demuxing audio data.
 * @param  output_buffer Pointer to output PDM buffer data.
 * @param  input_buffer  Pointer to input PDM buffer data.
 * @param  size          Size of the buffers (thay has to be equally sized).
 * @retval COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
 */
status_t PDM2PCMAudio::demux(uint16_t *output_buffer, uint16_t *input_buffer, uint32_t size)
{
    for (uint32_t index = 0; index < size; index++) {
        uint8_t a = ((uint8_t *) input_buffer)[index * 2];
        uint8_t b = ((uint8_t *) input_buffer)[index * 2 + 1];
        ((uint8_t *) (output_buffer))[index * 2] = _demux_filter[a & DEMUX_FILTER_MASK] | _demux_filter[b & DEMUX_FILTER_MASK] << 4;
        ((uint8_t *) (output_buffer))[index * 2 + 1] = _demux_filter[(a >> 1) & DEMUX_FILTER_MASK] | _demux_filter[(b >> 1) & DEMUX_FILTER_MASK] << 4;
    }

    return COMPONENT_OK;
}

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/ 
