/**
 ******************************************************************************
 * @file    PDM2PCMAudio.h
 * @author  AST / Software Platforms and Cloud
 * @version V1.0
 * @date    November 10th, 2016
 * @brief   Class header file for the PDM2PCMAudio conversion library.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __PDM2PCM_AUDIO_CLASS_H
#define __PDM2PCM_AUDIO_CLASS_H


/* Definitions ---------------------------------------------------------------*/

#define USE_OPEN_PDM2PCM_LIBRARY
#define PDM2PCM_AUDIO_DEBUG


/* Includes ------------------------------------------------------------------*/

#include "mbed.h"
#include "component_def.h"
#ifdef USE_OPEN_PDM2PCM_LIBRARY
#include "OpenPDMFilter.h"
#else
#include "pdm_filter.h"
#endif


/* Definitions ---------------------------------------------------------------*/

#define DEMUX_FILTER_SIZE         128
#define DEMUX_FILTER_MASK         0x55
#define PDM2PCM_FILTER_SIZE       4
#define PDM2PCM_NOGAIN_VOLUME     4
#define PDM2PCM_MAX_VOLUME        64
#define PDM2PCM_DECIMATION_FACTOR 64


/* Classes -------------------------------------------------------------------*/

/** PDM2PCMAudio Conversion Library Class.
 */
class PDM2PCMAudio
{
public:

    /**
     * @brief Constructor.
	 * @param  frequency Audio sampling frequency.
	 * @param  channels  Number of audio channels to be recorded.
     */
    PDM2PCMAudio(uint32_t frequency, uint32_t channels) :
	    _frequency(frequency),
	    _channels(channels)
#ifdef PDM2PCM_AUDIO_DEBUG
	    , _pdm2pcm_audio_signal(D9)
#endif
    {
	    /* Enable CRC peripheral to unlock the PDM2PCMAudio library. */
	    __CRC_CLK_ENABLE();

	    /* Initializing PDM2PCMAudio Filter. */
	    for (uint32_t i = 0; i < _channels; i++)
	    {
#ifdef USE_OPEN_PDM2PCM_LIBRARY
	        /* Filter LP and HP Init */
	        _PDM2PCM_filter[i].LP_HZ = _frequency / 2;
	        _PDM2PCM_filter[i].HP_HZ = 10;
	        _PDM2PCM_filter[i].Fs = _frequency;
	        _PDM2PCM_filter[i].Out_MicChannels = _channels;
	        _PDM2PCM_filter[i].In_MicChannels = _channels;
	        _PDM2PCM_filter[i].Decimation = 64;
			_PDM2PCM_filter[i].SincN = 3;
	        Open_PDM_Filter_Init((TPDMFilter_InitStruct *) &_PDM2PCM_filter[i]);
#else
	        /* Filter LP and HP Init */
	        _PDM2PCM_filter[i].LP_HZ = _frequency / 2;
	        _PDM2PCM_filter[i].HP_HZ = 10;
	        _PDM2PCM_filter[i].Fs = _frequency;
	        _PDM2PCM_filter[i].Out_MicChannels = _channels;
	        _PDM2PCM_filter[i].In_MicChannels = _channels;
	        PDM_Filter_Init((PDMFilter_InitStruct *) &_PDM2PCM_filter[i]);
#endif
	    }
    };

	/**
	* @brief  Converting audio data from PDM to PCM.
	* @param  output_buffer     Pointer to output PCM buffer data.
	* @param  input_buffer      Pointer to input PDM buffer data.
	* @param  volume            Volume level (it must be in the range [0..64]).
	* @param  decimation_factor Decimation factor (it must be either 64 or 128).
	* @retval COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
	*/
	status_t convert(int16_t *output_buffer, uint16_t *input_buffer, uint32_t volume, uint32_t decimation_factor);

	/**
	 * @brief  Scrambling audio data.
	 * @param  output_buffer Pointer to output PDM buffer data.
	 * @param  input_buffer  Pointer to input PDM buffer data.
	 * @param  size          Size of the buffers (thay has to be equally sized).
	 * @retval COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
	 */
	status_t scramble(uint16_t *output_buffer, uint16_t *input_buffer, uint32_t size);

	/**
	 * @brief  Demuxing audio data.
	 * @param  output_buffer Pointer to output PDM buffer data.
	 * @param  input_buffer  Pointer to input PDM buffer data.
	 * @param  size          Size of the buffers (thay has to be equally sized).
	 * @retval COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
	 */
	status_t demux(uint16_t *output_buffer, uint16_t *input_buffer, uint32_t size);


protected:

	uint32_t _frequency;
	uint32_t _channels;
#ifdef USE_OPEN_PDM2PCM_LIBRARY
	TPDMFilter_InitStruct _PDM2PCM_filter[PDM2PCM_FILTER_SIZE];
#else
	PDMFilter_InitStruct _PDM2PCM_filter[PDM2PCM_FILTER_SIZE];
#endif
	static const uint8_t _demux_filter[DEMUX_FILTER_SIZE];

#ifdef PDM2PCM_AUDIO_DEBUG
	DigitalOut _pdm2pcm_audio_signal;
#endif
};

#endif /* __PDM2PCM_AUDIO_CLASS_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
