/**
******************************************************************************
* @file    STA350BW.cpp
* @author  Central Labs
* @version V1.0.0
* @date    18-August-2015
* @brief   This file provides the STA350BW SOUND TERMINAL audio driver.
******************************************************************************
* @attention
*
* <h2><center>&copy; COPYRIGHT(c) 2014 STMicroelectronics</center></h2>
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*   3. Neither the name of STMicroelectronics nor the names of its contributors
*      may be used to endorse or promote products derived from this software
*      without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************
*/


/* Generated with STM32CubeTOO -----------------------------------------------*/


/* Revision ------------------------------------------------------------------*/
/*
	Repository:       http://svn.x-nucleodev.codex.cro.st.com/svnroot/X-NucleoDev
	Branch/Trunk/Tag: trunk
	Based on:         X-CUBE-SOUNDTER1/trunk/Drivers/BSP/Components/sta350bw/sta350bw.c
	Revision:         0
*/


/* Includes ------------------------------------------------------------------*/

#include "STA350BW.h"


/* Methods -------------------------------------------------------------------*/

/**
* @brief        Initializes the STA350BW and the control interface.
* @param        init: initialization data.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_Init(void *init)
{
  uint8_t tmp = 0x00;

  /* Reset Audio Device */
  _reset = 0;
  wait_ms(100);
  _reset = 1;
  
  /* Low level I2C init */
  STA350BW_IO_Init();

  /* Set Master clock depending on sampling frequency */
  if (STA350BW_SetFrequency((*((STA350BW_init_t *) init)).frequency) != 0)
  {
    return COMPONENT_ERROR;
  }

  STA350BW_IO_Delay(500);
  
  /* Read Status Register */
  if (STA350BW_IO_Read(STA350BW_STATUS, &tmp) != 0) 
  {
    return COMPONENT_ERROR;                          
  }

#if 0
  if (tmp != 0x7F) 
  {
    /* Status register highlights undesired behaviour (PLL not locked, ...) */
    return COMPONENT_ERROR;
  }
#else
  if ((tmp & ~0x80) != 0x7F)
  {
     /*Status register highlights undesired behavior
     (betzw: excluding PLL not locked, ...) */
     return COMPONENT_ERROR;
  }
#endif
  
  /* Setup Master volume */
  uint8_t value = (*((STA350BW_init_t *) init)).volume;
  if (!(value >= MIN_VOLUME && value <= MAX_VOLUME))
      return COMPONENT_ERROR;
  if (STA350BW_SetVolume(STA350BW_CHANNEL_MASTER, (uint8_t) MAX_VOLUME - value) != 0) 
  {
    return COMPONENT_ERROR;
  }

  if (STA350BW_IO_Read(STA350BW_CONF_REGF, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }

#if 0
  tmp &= ~0x80;
  tmp |= 0x80;
  
  /* Enable Power Out Stage */
  if (STA350BW_IO_Write(STA350BW_CONF_REGF, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
#else
  /*Enable Power Out Stage*/
  if (STA350BW_PowerOn() != 0)
  {
    return COMPONENT_ERROR;
  }

#endif

  return COMPONENT_OK;
}

/**
* @brief        Deinitializes the STA350BW and the control interface.
* @param        None.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise.
*/
int32_t STA350BW::STA350BW_DeInit(void) 
{  
  if (STA350BW_PowerOff() != 0) 
  {
    return COMPONENT_ERROR;
  }
  return COMPONENT_OK;  
}

/**
* @brief        Read the device ID.
* @param        id: identifier.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_ReadID(uint8_t *id) 
{
  return COMPONENT_OK;
}

/**
* @brief        Start the audio play.
* @param        *pData: pointer to audio data.
* @param        Size: size of the data buffer.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_Play(int16_t *pData, uint16_t Size)
{
  return COMPONENT_OK;
}

/**
* @brief        Pause the audio play.
* @param        None.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_Pause(void) 
{
  /* Mute the output*/
  if (STA350BW_SetMute(STA350BW_CHANNEL_MASTER, STA350BW_ENABLE) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  return COMPONENT_OK;
}

/**
* @brief        Resume the audio play.
* @param        None.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_Resume(void) 
{
  /* Unmute the output*/
  if (STA350BW_SetMute(STA350BW_CHANNEL_MASTER, STA350BW_DISABLE) != 0)
  {
    return COMPONENT_ERROR;
  }  
  return COMPONENT_OK;
}

/**
* @brief        Control the mute features of the STA350BW.
* @param        channel: channel to be muted.
*               This parameter can be a value of @ref STA350BW_channel_define
* @param        state: enable disable parameter
*               This parameter can be a value of @ref STA350BW_state_define
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_SetMute(uint8_t channel, uint8_t state) 
{  
  uint8_t tmp;
  
  if (STA350BW_IO_Read(STA350BW_MUTE, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  if (state == STA350BW_ENABLE) 
  {
    tmp |= channel;
  } else 
  {
    tmp &= ~channel;
  }
  
  if (STA350BW_IO_Write(STA350BW_MUTE, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  return COMPONENT_OK;
}

/**
* @brief        Control the volume features of the STA350BW.
* @param        channel: channel to be controlled.
*               This parameter can be a value of @ref STA350BW_channel_define
* @param        volume: volume to be set
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_SetVolume(uint8_t channel, uint8_t value) 
{
  /*Setup volume */
  uint8_t tmp = value;
  if (STA350BW_IO_Write(STA350BW_MVOL + channel, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  return COMPONENT_OK;
}

/**
* @brief        set the sampling frequency for STA350BW.
* @param        audio_freq: audio frequency to be set.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_SetFrequency(uint32_t audio_freq) 
{
  uint8_t tmp;
  
  if (STA350BW_IO_Read(STA350BW_CONF_REGA, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  
  tmp &= ~0x1F;

  if (audio_freq == STA350BW_Fs_32000 || audio_freq == STA350BW_Fs_44100 || audio_freq == STA350BW_Fs_48000) 
  {
    tmp |= STA350BW_MCLK_256_LR_48K;
  }
  else if (audio_freq == STA350BW_Fs_88200 || audio_freq == STA350BW_Fs_96000) 
  {
    tmp |= STA350BW_MCLK_256_LR_96K;
  }
  else
    return COMPONENT_ERROR;

  if (STA350BW_IO_Write(STA350BW_CONF_REGA, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }

  /* Set I2S audio frequency. */
  dev_i2s.audio_frequency(audio_freq);

  return COMPONENT_OK;
}

/**
* @brief        Set equalization parameters for STA350BW biquad section.
* @param        ram_block: ram block to be set
* @param        filter_number: filter number
* @param        *filter_values: pointer to a uint32_t array containing filter coefficients
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_SetEq(uint8_t ram_block, uint8_t filter_number, uint32_t * filter_values) 
{  
  /*5 is due to the ram adressing: first filter is on the adresses 0x00 to 0x04; the second is on 0x05 to 0x09 ...*/
  STA350BW_WriteRAMSet(ram_block, filter_number * 5, (uint8_t *) filter_values); 
  return COMPONENT_OK;
}

/**
* @brief        Set tone value in the STA350BW tone register.
* @param        tone_gain: gain of the tone control
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_SetTone(uint8_t tone_gain) 
{
  uint8_t tmp = tone_gain;
  
  if (STA350BW_IO_Write(STA350BW_TONE, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  return COMPONENT_OK;
}

/**
* @brief        Power on the device.
* @param        None.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_PowerOn(void) 
{  
  uint8_t tmp;
  if (STA350BW_IO_Read(STA350BW_CONF_REGF, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  tmp |= 0xC0;
  if (STA350BW_IO_Write(STA350BW_CONF_REGF, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  return COMPONENT_OK;
}

/**
* @brief        Power off the device.
* @param        None.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_PowerOff(void) 
{  
  uint8_t tmp;  
  if (STA350BW_IO_Read(STA350BW_CONF_REGF, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  tmp &= ~0xC0;
  if (STA350BW_IO_Write(STA350BW_CONF_REGF, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  return COMPONENT_OK;
}

/**
* @brief        Stop audio stream.
* @param        None.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_Stop(void)
{  
  return COMPONENT_OK;
}

/**
* @brief        Reset device.
* @param        NOne.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_Reset(void)
{  
  return COMPONENT_OK;
}

/**
* @brief  This function can be used to set advanced DSP options in order to 
*         use advanced features on the STA350BW device.
* @param  option: specific option to be setted up
*         This parameter can be a value of @ref STA350BW_DSP_option_selection 
* @param  state: state of the option to be controlled. Depending on the selected 
*         DSP feature to be controlled, this value can be either ENABLE/DISABLE 
*         or a specific numerical parameter related to the specific DSP function. 
*         This parameter can be a value of @ref STA350BW_state_define   
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_SetDSPOption(uint8_t option, uint8_t state)
{
  uint8_t tmp = 0;  
  
  switch (option) 
  {
  case STA350BW_DSPB:
    {
      if (STA350BW_IO_Read(STA350BW_CONF_REGD, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x04;
      tmp |= state << 0x02;
      
      if (STA350BW_IO_Write(STA350BW_CONF_REGD, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_HPB:
    {
      if (STA350BW_IO_Read(STA350BW_CONF_REGD, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x01;
      tmp |= state << 0x00;
      
      if (STA350BW_IO_Write(STA350BW_CONF_REGD, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_DEMP:
    {
      if (STA350BW_IO_Read(STA350BW_CONF_REGD, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x02;
      tmp |= state << 0x01;
      
      if (STA350BW_IO_Write(STA350BW_CONF_REGD, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_BQL:
    {
      if (STA350BW_IO_Read(STA350BW_CONF_REGD, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x08;
      tmp |= state << 0x04;
      
      if (STA350BW_IO_Write(STA350BW_CONF_REGD, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_BQ5:
    {
      if (STA350BW_IO_Read(STA350BW_CONFX, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x04;
      tmp |= state << 0x02;
      
      if (STA350BW_IO_Write(STA350BW_CONFX, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_BQ6:
    {
      if (STA350BW_IO_Read(STA350BW_CONFX, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x02;
      tmp |= state << 0x01;
      
      if (STA350BW_IO_Write(STA350BW_CONFX, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_BQ7:
    {
      if (STA350BW_IO_Read(STA350BW_CONFX, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x01;
      tmp |= state << 0x00;
      
      if (STA350BW_IO_Write(STA350BW_CONFX, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }    
  case STA350BW_C1EQBP:
    {
      if (STA350BW_IO_Read(STA350BW_C1CFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x02;
      tmp |= state << 0x01;
      
      if (STA350BW_IO_Write(STA350BW_C1CFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }    
  case STA350BW_C2EQBP:
    {
      if (STA350BW_IO_Read(STA350BW_C2CFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x02;
      tmp |= state << 0x01;
      
      if (STA350BW_IO_Write(STA350BW_C2CFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }    
  case STA350BW_C1TCB:
    {
      if (STA350BW_IO_Read(STA350BW_C1CFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x01;
      tmp |= state << 0x00;
      
      if (STA350BW_IO_Write(STA350BW_C1CFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }    
  case STA350BW_C2TCB:
    {
      if (STA350BW_IO_Read(STA350BW_C2CFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x01;
      tmp |= state << 0x00;
      
      if (STA350BW_IO_Write(STA350BW_C2CFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }    
  case STA350BW_C1VBP:
    {
      if (STA350BW_IO_Read(STA350BW_C1CFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x04;
      tmp |= state << 0x02;
      
      if (STA350BW_IO_Write(STA350BW_C1CFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }    
  case STA350BW_C2VBP:
    {
      if (STA350BW_IO_Read(STA350BW_C2CFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x04;
      tmp |= state << 0x02;
      
      if (STA350BW_IO_Write(STA350BW_C2CFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ1:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B4B1, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x03;
      tmp |= (state>>1);
      
      if (STA350BW_IO_Write(STA350BW_CXT_B4B1, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ2:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B4B1, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x0C;
      tmp |= (state>>1) << 2;
      
      if (STA350BW_IO_Write(STA350BW_CXT_B4B1, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ3:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B4B1, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x30;
      tmp |= (state>>1) << 4;
      
      if (STA350BW_IO_Write(STA350BW_CXT_B4B1, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ4:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B4B1, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0xC0;
      tmp |= (state>>1) << 6;
      
      if (STA350BW_IO_Write(STA350BW_CXT_B4B1, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ5:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B7B5, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x03;
      tmp |= (state>>1);
      
      if (STA350BW_IO_Write(STA350BW_CXT_B7B5, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ6:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B7B5, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x0C;
      tmp |= (state>>1) << 2;
      
      if (STA350BW_IO_Write(STA350BW_CXT_B7B5, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_EXT_RANGE_BQ7:
    {
      if (STA350BW_IO_Read(STA350BW_CXT_B7B5, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x30;
      tmp |= (state>>1) << 4;
      
      if (STA350BW_IO_Write(STA350BW_CXT_B7B5, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    }
  case STA350BW_RAM_BANK_SELECT:
    {
      if (STA350BW_IO_Read(STA350BW_EQCFG, &tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }  
      tmp &= ~0x03;
      tmp |= state;
      
      if (STA350BW_IO_Write(STA350BW_EQCFG, tmp) != 0) 
      {
        return COMPONENT_ERROR;
      }
      break;
    } 
  }
  return COMPONENT_OK;
}

/**
* @brief        private function for writing a RAM set.
* @param        RAM_block: ram block to be written.
* @param        RAM_address: ram address to be written.
* @param        *pIn: pointer to the desired value to be write.
* @retval       COMPONENT_OK if correct setup, COMPONENT_ERROR otherwise
*/
int32_t STA350BW::STA350BW_WriteRAMSet(uint8_t RAM_block, uint8_t RAM_address, uint8_t * pIn) 
{
  uint8_t tmp = 0x00;
  /*choose block*/
  if (STA350BW_IO_Read(STA350BW_EQCFG, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  tmp &= ~0x03;
  RAM_block &= 0x03;
  tmp |= RAM_block;
  if (STA350BW_IO_Write(STA350BW_EQCFG, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  /*set address*/
  if (STA350BW_IO_Read(STA350BW_CFADDR, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  tmp &= ~0x3F;
  RAM_address &= 0x3F;
  tmp |= RAM_address;
  if (STA350BW_IO_Write(STA350BW_CFADDR, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  /*write*/
  if (STA350BW_IO_Write(STA350BW_B1CF1, pIn[2]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B1CF2, pIn[1]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B1CF3, pIn[0]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B2CF1, pIn[6]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B2CF2, pIn[5]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B2CF3, pIn[4]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_A1CF1, pIn[10]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_A1CF2, pIn[9]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_A1CF3, pIn[8]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_A2CF1, pIn[14]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_A2CF2, pIn[13]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_A2CF3, pIn[12]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B0CF1, pIn[18]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B0CF2, pIn[17]) != 0) 
  {
    return COMPONENT_ERROR;
  }
  if (STA350BW_IO_Write(STA350BW_B0CF3, pIn[16]) != 0) 
  {
    return COMPONENT_ERROR;
  }  
  /*Set WA PIN*/
  if (STA350BW_IO_Read(STA350BW_CFUD, &tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  tmp &= ~0x02;
  tmp = 0x02; 
  
  if (STA350BW_IO_Write(STA350BW_CFUD, tmp) != 0) 
  {
    return COMPONENT_ERROR;
  }
  return COMPONENT_OK;
}

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
