/**
******************************************************************************
* @file    STA350BW.h
* @author  Central Labs
* @version V1.0.0
* @date    18-August-2015
* @brief   This file provides the STA350BW SOUND TERMINAL audio driver.
******************************************************************************
* @attention
*
* <h2><center>&copy; COPYRIGHT(c) 2014 STMicroelectronics</center></h2>
*
* Redistribution and use in source and binary forms, with or without modification,
* are permitted provided that the following conditions are met:
*   1. Redistributions of source code must retain the above copyright notice,
*      this list of conditions and the following disclaimer.
*   2. Redistributions in binary form must reproduce the above copyright notice,
*      this list of conditions and the following disclaimer in the documentation
*      and/or other materials provided with the distribution.
*   3. Neither the name of STMicroelectronics nor the names of its contributors
*      may be used to endorse or promote products derived from this software
*      without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
* DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
* FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
* SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
* OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*
******************************************************************************
*/


/* Generated with STM32CubeTOO -----------------------------------------------*/


/* Revision ------------------------------------------------------------------*/
/*
	Repository:       http://svn.x-nucleodev.codex.cro.st.com/svnroot/X-NucleoDev
	Branch/Trunk/Tag: trunk
	Based on:         X-CUBE-SOUNDTER1/trunk/Drivers/BSP/Components/STA350BW/STA350BW.h
	Revision:         0
*/


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __STA350BW_CLASS_H
#define __STA350BW_CLASS_H


/* Configuration -------------------------------------------------------------*/

/**
 * Enable to debug the duration of functions related to the I2S and USB usage
 * through an oscilloscope.
 */
//#define X_NUCLEO_CCA01M1_DEBUG


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/		
#include "mbed.h"
#include "I2S.h"
#include "DevI2C.h"
/* ACTION 2 ------------------------------------------------------------------*
 * Include here component specific header files.                              *
 *----------------------------------------------------------------------------*/		
#include "STA350BW_def.h"
/* ACTION 3 ------------------------------------------------------------------*
 * Include here interface specific header files.                              *
 *                                                                            *
 * Example:                                                                   *
 *   #include "Humidity_class.h"                                              *
 *   #include "Temperature_class.h"                                           *
 *----------------------------------------------------------------------------*/
#include "SoundTerminal.h"


/* Definitions ---------------------------------------------------------------*/

#define MIN_VOLUME   0
#define MAX_VOLUME 128


/* Classes -------------------------------------------------------------------*/

/**
 * @brief Class representing a STA350BW component.
 */
class STA350BW : public SoundTerminal
{
public:

	/*** Constructor and Destructor Methods ***/

	/**
	 * @brief Constructor.
	 * @param reset   pin name of the RESET pin of the component.
	 * @param address I2C address of the component.
	 * @param i2c     I2C device to be used for communication.
	 * @param dpin    pin name of the DPIN pin of the I2S device to be used for audio transmission.
	 * @param clk     pin name of the CLK pin of the I2S device to be used for audio transmission.
	 * @param wsel    pin name of the WSEL pin of the I2S device to be used for audio transmission.
	 * @param fdpin   pin name of the FDPIN pin of the I2S device to be used for audio transmission.
	 * @param mck     pin name of the MCK pin of the I2S device to be used for audio transmission.
     * @note  Initialization depends on the I2S interface you wish to use:
     *          I2S1) address = STA350BW_ADDRESS_1, dpin = PB_15, clk = PB_13, wsel = PB_12, fdpin = NC, mck = PC_6;
     *          I2S2) address = STA350BW_ADDRESS_2, dpin = PC_12, clk = PC_10, wsel = PA_4, fdpin = NC, mck = PC_7.
	 */
	STA350BW(PinName reset, uint8_t address, DevI2C &i2c, PinName dpin, PinName clk, PinName wsel, PinName fdpin = NC, PinName mck = NC) :
		SoundTerminal(),
		_reset(reset),
		_address(address),
		_dev_i2c(i2c),
		dev_i2s(dpin, clk, wsel, fdpin, mck),
		_PCM_buffer(NULL),
		_PCM_buffer_bytes(0),
		_loop(false)
#ifdef X_NUCLEO_CCA01M1_DEBUG
        , _i2s_signal(D11)
#endif
	{
		/* ACTION 4 ----------------------------------------------------------*
		 * Initialize here the component's member variables, one variable per *
		 * line.                                                              *
		 *                                                                    *
		 * Example:                                                           *
		 *   measure = 0;                                                     *
		 *   instance_id = number_of_instances++;                             *
		 *--------------------------------------------------------------------*/
	}
	
	/**
	 * @brief Destructor.
	 */
	virtual ~STA350BW(void) {}
	

	/*** Public Component Related Methods ***/

	/* ACTION 5 --------------------------------------------------------------*
	 * Implement here the component's public methods, as wrappers of the C    *
	 * component's functions.                                                 *
	 * They should be:                                                        *
	 *   + Methods with the same name of the C component's virtual table's    *
	 *     functions (1);                                                     *
	 *   + Methods with the same name of the C component's extended virtual   *
	 *     table's functions, if any (2).                                     *
	 *                                                                        *
	 * Example:                                                               *
	 *   virtual int GetValue(float *f)  //(1)                                *
	 *   {                                                                    *
	 *     return COMPONENT_GetValue(float *f);                               *
	 *   }                                                                    *
	 *                                                                        *
	 *   virtual int EnableFeature(void) //(2)                                *
	 *   {                                                                    *
	 *     return COMPONENT_EnableFeature();                                  *
	 *   }                                                                    *
	 *------------------------------------------------------------------------*/
    /**
    * @brief  Initializing the STA350BW component.
    * @param  init Pointer to device specific initalization structure.
    * @retval "0" in case of success, an error code otherwise.
    */
	virtual int init(void *init = NULL)
	{
		if (STA350BW_Init((void *) init) != COMPONENT_OK)
			return COMPONENT_ERROR;

	    /* Setting I2S parameters. */
		dev_i2s.mode(MASTER_TX, true);

		return COMPONENT_OK;
	}

    /**
     * @brief  Getting the ID of the component.
     * @param  id Pointer to an allocated variable to store the ID into.
     * @retval "0" in case of success, an error code otherwise.
     */
	virtual int read_id(uint8_t *id = NULL)
	{
		return (int) STA350BW_ReadID((uint8_t *) id);
	}

    /**
    * @brief  De-initializing the STA350BW component.
    * @param  None.
    * @retval "0" in case of success, an error code otherwise.
    */
	virtual int32_t de_init(void)
	{
		return (int32_t) STA350BW_DeInit();
	}

	/*
	 * @brief  Start playing audio.
	 * @param  PCM_buffer       The buffer of data to be played.
	 * @param  PCM_buffer_bytes The size in bytes of the buffer of data to be played.
	 * @param  loop             Loops indefinitely if true, just once otherwise.
	 * @retval "0" in case of success, an error code otherwise.
	 */
	virtual int32_t play(int16_t *PCM_buffer, uint16_t PCM_buffer_bytes, bool loop = false)
	{
		/* Storing data. */
		_PCM_buffer = PCM_buffer;
		_PCM_buffer_bytes = PCM_buffer_bytes;
		_loop = loop;

		/* Sending data to the speakers via I2S. */
		int res = dev_i2s.transfer(
		    (void *) _PCM_buffer, _PCM_buffer_bytes,
		    (void *) NULL, 0,
		    event_callback_t(this, &STA350BW::i2s_callback),
		    I2S_EVENT_ALL
		);
	    if (res != 0)
	        return COMPONENT_ERROR;

	    return COMPONENT_OK;
	}

	/*
	 * @brief  Stop playing audio.
	 * @param  None.
	 * @retval None.
	 */
	virtual int32_t stop(void)
	{
		dev_i2s.abort_all_transfers();
		return (int32_t) STA350BW_Stop();
	}

	/*
	 * @brief  Set the volume of the audio.
	 * @param  channel The channel on which to set the volume of the audio.
	 * @param  value   The value of the volume in the range [MIN_VOLUME..MAX_VOLUME].
	 * @retval "0" in case of success, an error code otherwise.
	 */
	virtual int32_t set_volume(uint8_t channel, uint8_t value)
	{
		if (!(value >= MIN_VOLUME && value <= MAX_VOLUME))
			return COMPONENT_ERROR;

		return (int32_t) STA350BW_SetVolume((uint8_t) channel, (uint8_t) MAX_VOLUME - value);
	}

	/*
	 * @brief  Set the frequency of the audio.
	 * @param  audio_freq The frequency of the audio.
	 * @retval "0" in case of success, an error code otherwise.
	 */
	virtual int32_t set_frequency(uint32_t audio_freq)
	{
		return (int32_t) STA350BW_SetFrequency((uint32_t) audio_freq);
	}


	/*** Public Interrupt Related Methods ***/

	/* ACTION 6 --------------------------------------------------------------*
	 * Implement here interrupt related methods, if any.                      *
	 * Note that interrupt handling is platform dependent, e.g.:              *
	 *   + mbed:                                                              *
	 *     InterruptIn feature_irq(pin);           //Interrupt object.        *
	 *     feature_irq.fall(callback);             //Attach a callback.       *
	 *     feature_irq.mode(PullNone);             //Set interrupt mode.      *
	 *     feature_irq.enable_irq();               //Enable interrupt.        *
	 *     feature_irq.disable_irq();              //Disable interrupt.       *
	 *   + Arduino:                                                           *
	 *     attachInterrupt(pin, callback, RISING); //Attach a callback.       *
	 *     detachInterrupt(pin);                   //Detach a callback.       *
	 *                                                                        *
	 * Example (mbed):                                                        *
	 *   void AttachFeatureIRQ(void (*fptr) (void))                           *
	 *   {                                                                    *
	 *     feature_irq.fall(fptr);                                            *
	 *   }                                                                    *
	 *                                                                        *
	 *   void EnableFeatureIRQ(void)                                          *
	 *   {                                                                    *
	 *     feature_irq.enable_irq();                                          *
	 *   }                                                                    *
	 *                                                                        *
	 *   void DisableFeatureIRQ(void)                                         *
	 *   {                                                                    *
	 *     feature_irq.disable_irq();                                         *
	 *   }                                                                    *
	 *------------------------------------------------------------------------*/


protected:

	/*** Protected Component Related Methods ***/

	/* ACTION 7 --------------------------------------------------------------*
	 * Declare here the component's specific methods.                         *
	 * They should be:                                                        *
	 *   + Methods with the same name of the C component's virtual table's    *
	 *     functions (1);                                                     *
	 *   + Methods with the same name of the C component's extended virtual   *
	 *     table's functions, if any (2);                                     *
	 *   + Helper methods, if any, like functions declared in the component's *
	 *     source files but not pointed by the component's virtual table (3). *
	 *                                                                        *
	 * Example:                                                               *
	 *   status_t COMPONENT_GetValue(float *f);   //(1)                       *
	 *   status_t COMPONENT_EnableFeature(void);  //(2)                       *
	 *   status_t COMPONENT_ComputeAverage(void); //(3)                       *
	 *------------------------------------------------------------------------*/
	int32_t STA350BW_Init(void *init);
	int32_t STA350BW_ReadID(uint8_t *id);
	int32_t STA350BW_DeInit(void);
	int32_t STA350BW_Play(int16_t *pData, uint16_t Size);
	int32_t STA350BW_Stop(void);
	int32_t STA350BW_Pause(void);
	int32_t STA350BW_Resume(void);
	int32_t STA350BW_SetVolume(uint8_t channel, uint8_t value);
	int32_t STA350BW_SetFrequency(uint32_t audio_freq);
	int32_t STA350BW_PowerOn(void);
	int32_t STA350BW_PowerOff(void);
	int32_t STA350BW_Reset(void);
	int32_t STA350BW_SetEq(uint8_t ram_block, uint8_t filter_number, uint32_t * filter_values);
	int32_t STA350BW_SetTone(uint8_t tone_gain);
	int32_t STA350BW_SetMute(uint8_t channel, uint8_t state);
	int32_t STA350BW_SetDSPOption(uint8_t option, uint8_t state);
	int32_t STA350BW_WriteRAMSet(uint8_t RAM_block, uint8_t RAM_address, uint8_t * pIn);

	/**
	 * @brief  I2S callback.
	 * @param  narg Narg flag.
	 * @retval None.
	 */
	void i2s_callback(int narg)
	{
	    if (!(narg & (I2S_EVENT_TX_COMPLETE | I2S_EVENT_TX_HALF_COMPLETE)))
	        error("Unexpected transmission event.\r\n");
	    else if ((narg & I2S_EVENT_TX_COMPLETE) && !_loop)
	    	stop();

#ifdef X_NUCLEO_CCA01M1_DEBUG
	    _i2s_signal = !_i2s_signal;
#endif
	}


	/*** Component's I/O Methods ***/

	/**
	 * @brief      Utility function to read data from STA350BW.
	 * @param[out] pBuffer pointer to the buffer to read data into.
	 * @param[in]  RegisterAddr specifies the internal address register to read from.
	 * @param[in]  NumBytesToRead number of bytes to read.
	 * @retval     COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
	 */
	status_t read(uint8_t* pBuffer, uint8_t RegisterAddr, uint16_t NumBytesToRead)
	{
		if (_dev_i2c.i2c_read(pBuffer, _address, RegisterAddr, NumBytesToRead) != 0)
			return COMPONENT_ERROR;
		return COMPONENT_OK;
	}
	
	/**
	 * @brief      Utility function to write data to STA350BW.
	 * @param[in]  pBuffer pointer to the buffer of data to send.
	 * @param[in]  RegisterAddr specifies the internal address register to write to.
	 * @param[in]  NumBytesToWrite number of bytes to write.
	 * @retval     COMPONENT_OK in case of success, COMPONENT_ERROR otherwise.
	 */
	status_t write(uint8_t* pBuffer, uint8_t RegisterAddr, uint16_t NumBytesToWrite)
	{
		if (_dev_i2c.i2c_write(pBuffer, _address, RegisterAddr, NumBytesToWrite) != 0)
			return COMPONENT_ERROR;
		return COMPONENT_OK;
	}

	/* ACTION 8 --------------------------------------------------------------*
	 * Implement here other I/O methods beyond those already implemented      *
	 * above, which are declared extern within the component's header file.   *
	 *------------------------------------------------------------------------*/
	uint8_t STA350BW_IO_Init(void)
	{
		/* TO BE IMPLEMENTED BY USING TARGET PLATFORM'S APIs. */
		return (uint8_t) 0;
	}

	uint8_t STA350BW_IO_Read(uint8_t reg, uint8_t *value)
	{
		/* TO BE IMPLEMENTED BY USING TARGET PLATFORM'S APIs. */
		return (uint8_t) read(value, reg, 1);
	}

	uint8_t STA350BW_IO_Write(uint8_t reg, uint8_t value)
	{
		/* TO BE IMPLEMENTED BY USING TARGET PLATFORM'S APIs. */
		return (uint8_t) write(&value, reg, 1);
	}

	uint8_t STA350BW_IO_ReadMulti(uint8_t *pBuffer, uint8_t reg, uint16_t length)
	{
		/* TO BE IMPLEMENTED BY USING TARGET PLATFORM'S APIs. */
		return (uint8_t) read(pBuffer, reg, length);
	}

	uint8_t STA350BW_IO_WriteMulti(uint8_t *pBuffer, uint8_t reg, uint16_t length)
	{
		/* TO BE IMPLEMENTED BY USING TARGET PLATFORM'S APIs. */
		return (uint8_t) write(pBuffer, reg, length);
	}

	uint8_t STA350BW_IO_Delay(uint32_t delay_ms)
	{
		/* TO BE IMPLEMENTED BY USING TARGET PLATFORM'S APIs. */
		wait_ms(delay_ms);
		return (uint8_t) 0;
	}


	/*** Component's Instance Variables ***/

	/* ACTION 9 --------------------------------------------------------------*
	 * Declare here interrupt related variables, if needed.                   *
	 * Note that interrupt handling is platform dependent, see                *
	 * "Interrupt Related Methods" above.                                     *
	 *                                                                        *
	 * Example:                                                               *
	 *   + mbed:                                                              *
	 *     InterruptIn feature_irq;                                           *
	 *------------------------------------------------------------------------*/

	/* ACTION 10 -------------------------------------------------------------*
	 * Declare here other pin related variables, if needed.                   *
	 *                                                                        *
	 * Example:                                                               *
	 *   + mbed:                                                              *
	 *     DigitalOut standby_reset;                                          *
	 *------------------------------------------------------------------------*/
	DigitalOut _reset;

	/* ACTION 11 -------------------------------------------------------------*
	 * Declare here communication related variables, if needed.               *
	 *                                                                        *
	 * Example:                                                               *
	 *   + mbed:                                                              *
	 *     DigitalOut address;                                                *
	 *     DevI2C &dev_i2c;                                                   *
	 *------------------------------------------------------------------------*/
	/* Configuration. */
	uint8_t _address;

	/* IO Device. */
	DevI2C &_dev_i2c;

public:

    /* I2S Audio Device (exclusive). */
    I2S dev_i2s;


protected:

	/* ACTION 12 -------------------------------------------------------------*
	 * Declare here identity related variables, if needed.                    *
	 * Note that there should be only a unique identifier for each component, *
	 * which should be the "who_am_i" parameter.                              *
	 *------------------------------------------------------------------------*/

	/* ACTION 13 -------------------------------------------------------------*
	 * Declare here the component's static and non-static data, one variable  *
	 * per line.                                                              *
	 *                                                                        *
	 * Example:                                                               *
	 *   float measure;                                                       *
	 *   int instance_id;                                                     *
	 *   static int number_of_instances;                                      *
	 *------------------------------------------------------------------------*/
	/* Buffer of data to be played. */
	int16_t *_PCM_buffer;
	/* The size in bytes of the buffer of data to be played. */
	uint16_t _PCM_buffer_bytes;
	/* Loops indefinitely if true, just once otherwise. */
	bool _loop;

#ifdef X_NUCLEO_CCA01M1_DEBUG
    /* Signals for debugging purposes. */
	DigitalOut _i2s_signal;
#endif
};

#endif /* __STA350BW_CLASS_H */

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
