/**
 ******************************************************************************
 * @file    L6208.h
 * @author  IPC Rennes
 * @version V1.0.0
 * @date    March 18th, 2016
 * @brief   This file contains the class of a L6208 Motor Control component.
 * @note    (C) COPYRIGHT 2016 STMicroelectronics
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Define to prevent recursive inclusion -------------------------------------*/

#ifndef __L6208_CLASS_H
#define __L6208_CLASS_H


/* Includes ------------------------------------------------------------------*/

/* ACTION 1 ------------------------------------------------------------------*
 * Include here platform specific header files.                               *
 *----------------------------------------------------------------------------*/        
#include "mbed.h"

/* ACTION 2 ------------------------------------------------------------------*
 * Include here component specific header files.                              *
 *----------------------------------------------------------------------------*/        
#include "L6208_def.h"
/* ACTION 3 ------------------------------------------------------------------*
 * Include here interface specific header files.                              *
 *                                                                            *
 * Example:                                                                   *
 *   #include "HumiditySensor.h"                                              *
 *   #include "TemperatureSensor.h"                                           *
 *----------------------------------------------------------------------------*/
#include "StepperMotor.h"


/* Classes -------------------------------------------------------------------*/

/**
 * @brief Class representing a L6208 component.
 */
class L6208 : public StepperMotor
{
public:
    /*** Constructor and Destructor Methods ***/

    /**
     * @brief Constructor.
     * @param flag_and_enable_pin   pin name of the EN pin of the component.
     * @param reset_pin             pin name of the RESET pin of the component.
     * @param direction_pin         pin name of the CW_CCW pin of the component.
     * @param half_full_pin         pin name of the HALF_FULL pin of the component.
     * @param control_pin           pin name of the CONTROL pin of the component.
     * @param clock_pin             pin name of the CLOCK pin of the component.
     * @param vrefA_pwm_pin         pin name of the PWM connected to the VREFA pin of the component.
     * @param vrefB_pwm_pin         pin name of the PWM connected to the VREFB pin of the component.
     */
    L6208(PinName flag_and_enable_pin, PinName reset_pin, PinName direction_pin, PinName half_full_pin, PinName control_pin, PinName clock_pin, PinName vrefA_pwm_pin, PinName vrefB_pwm_pin) : StepperMotor(),
                                                                                                                                                         flag_and_enable(flag_and_enable_pin),
                                                                                                                                                         reset_pin(reset_pin),
                                                                                                                                                         direction_pin(direction_pin),
                                                                                                                                                         half_full_pin(half_full_pin),
                                                                                                                                                         control_pin(control_pin),
                                                                                                                                                         clock_pin(clock_pin),
                                                                                                                                                         vrefA_pwm(vrefA_pwm_pin),
                                                                                                                                                         vrefB_pwm(vrefB_pwm_pin)
    {
        /* Checking stackability. */
        if (numberOfDevices!=0)
            error("Instantiation of the L6208 component failed: it can't be stacked on itself.\r\n");

        /* ACTION 4 ----------------------------------------------------------*
         * Initialize here the component's member variables, one variable per *
         * line.                                                              *
         *                                                                    *
         * Example:                                                           *
         *   measure = 0;                                                     *
         *   instance_id = number_of_instances++;                             *
         *--------------------------------------------------------------------*/
        errorHandlerCallback = 0;
        deviceInstance = numberOfDevices++;
        /* default tick frequency */
        tickFreq = TIMER_TICK_FREQUENCY;
        /* waveform microstepping PWM period sample array, 90 deg shifted */
        pMicroTable2 = &(microTable1[16]);
    }
    
    /**
     * @brief Destructor.
     */
    virtual ~L6208(void) {}
    

    /*** Public Component Related Methods ***/

    /* ACTION 5 --------------------------------------------------------------*
     * Implement here the component's public methods, as wrappers of the C    *
     * component's functions.                                                 *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2).                                     *
     *                                                                        *
     * Example:                                                               *
     *   virtual int get_value(float *pData) //(1)                            *
     *   {                                                                    *
     *     return COMPONENT_get_value(float *pfData);                         *
     *   }                                                                    *
     *                                                                        *
     *   virtual int enable_feature(void) //(2)                               *
     *   {                                                                    *
     *     return COMPONENT_enable_feature();                                 *
     *   }                                                                    *
     *------------------------------------------------------------------------*/

    /**
     * @brief Public functions inherited from the Component Class
     */

    /**
     * @brief  Initialize the component.
     * @param  init Pointer to device specific initalization structure.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int init(void *init = NULL)
    {
        return (int) L6208_Init((void *) init);
    }

    /**
     * @brief  Getting the ID of the component.
     * @param  id Pointer to an allocated variable to store the ID into.
     * @retval "0" in case of success, an error code otherwise.
     */
    virtual int read_id(uint8_t *id = NULL)
    {
        return (int) L6208_ReadID((uint8_t *) id);
    }

    /**
     * @brief Public functions inherited from the StepperMotor Class
     */

    /**
     * @brief  Getting the value of the motor state .
     * @param  None.
     * @retval The motor state accoring to motorState_t in motor.h
     */
    virtual unsigned int get_status(void)
    {
        return (unsigned int) L6208_GetMotionState();
    }

    /**
     * @brief  Getting the position.
     * @param  None.
     * @retval The position.
     */
    virtual signed int get_position(void)
    {
        return (signed int)L6208_GetPosition();
    }

    /**
     * @brief  Getting the marked position.
     * @param  None.
     * @retval The marked position.
     */
    virtual signed int get_mark(void)
    {
        return (signed int)L6208_GetMark();
    }

    /**
     * @brief  Getting the current speed in pps.
     * @param  None.
     * @retval The current speed in pps.
     */
    virtual unsigned int get_speed(void)
    {
        return (unsigned int)L6208_GetCurrentSpeed();
    }

    /**
     * @brief  Getting the maximum speed in pps.
     * @param  None.
     * @retval The maximum speed in pps.
     */
    virtual unsigned int get_max_speed(void)
    {
        return (unsigned int)L6208_GetMaxSpeed();
    }

    /**
     * @brief  Getting the minimum speed in pps.
     * @param  None.
     * @retval The minimum speed in pps.
     */
    virtual unsigned int get_min_speed(void)
    {
        return (unsigned int)L6208_GetMinSpeed();
    }

    /**
     * @brief  Getting the acceleration in pps^2.
     * @param  None.
     * @retval The acceleration in pps^2.
     */
    virtual unsigned int get_acceleration(void)
    {
        return (unsigned int)L6208_GetAcceleration();
    }

    /**
     * @brief  Getting the deceleration in pps^2.
     * @param  None.
     * @retval The deceleration in pps^2.
     */
    virtual unsigned int get_deceleration(void)
    {
        return (unsigned int)L6208_GetDeceleration();
    }
    
    /**
     * @brief  Getting the direction of rotation.
     * @param  None.
     * @retval The direction of rotation.
     */
    virtual direction_t get_direction(void)
    {
        if (L6208_GetDirection()!=BACKWARD)
        {
            return FWD;
        }
        else
        {
            return BWD;
        }
    }
    
    /**
     * @brief  Setting the current position to be the home position.
     * @param  None.
     * @retval None.
     */
    virtual void set_home(void)
    {
        L6208_SetHome();
    }

    /**
     * @brief  Setting the current position to be the marked position.
     * @param  None.
     * @retval None.
     */
    virtual void set_mark(void)
    {
        L6208_SetMark();
    }

    /**
     * @brief  Setting the maximum speed in pps.
     * @param  speed The maximum speed in pps.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool set_max_speed(unsigned int speed)
    {
        if (speed <= 0xFFFF)
        {
            return L6208_SetMaxSpeed((uint16_t) speed);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the minimum speed in pps.
     * @param  speed The minimum speed in pps.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool set_min_speed(unsigned int speed)
    {
        if (speed <= 0xFFFF)
        {
            return L6208_SetMinSpeed((uint16_t) speed);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the acceleration in pps^2.
     * @param  acceleration The acceleration in pps/s^2.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool set_acceleration(unsigned int acceleration)
    {
        if (acceleration <= 0xFFFF)
        {
            return L6208_SetAcceleration((uint16_t) acceleration);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the deceleration in pps^2.
     * @param  deceleration The deceleration in pps^2.
     * @retval "true" in case of success, "false" otherwise.
     */
    virtual bool set_deceleration(unsigned int deceleration)
    {
        if (deceleration <= 0xFFFF)
        {
            return L6208_SetDeceleration((uint16_t) deceleration);
        }
        else
        {
            return false;
        }
    }

    /**
     * @brief  Setting the Step Mode.
     * @param  step_mode The Step Mode.
     * @retval "true" in case of success, "false" otherwise.
     * @note   step_mode can be one of the following:
     *           + STEP_MODE_FULL
     *           + STEP_MODE_WAVE
     *           + STEP_MODE_HALF
     *           + STEP_MODE_1_4
     *           + STEP_MODE_1_8
     *           + STEP_MODE_1_16
     */
    virtual bool set_step_mode(step_mode_t step_mode)
    {
        return L6208_SetStepMode((motorStepMode_t) step_mode);
    }

    /**
     * @brief  Going to a specified position.
     * @param  position The desired position.
     * @retval None.
     */
    virtual void go_to(signed int position)
    {
        L6208_GoTo((int32_t)position);
    }

    /**
     * @brief  Going to the home position.
     * @param  None.
     * @retval None.
     */
    virtual void go_home(void)
    {
        L6208_GoHome();
    }

    /**
     * @brief  Going to the marked position.
     * @param  None.
     * @retval None.
     */
    virtual void go_mark(void)
    {
        L6208_GoMark();
    }

    /**
     * @brief  Running the motor towards a specified direction.
     * @param  direction The direction of rotation.
     * @retval None.
     */
    virtual void run(direction_t direction)
    {
        L6208_Run((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD));
    }

    /**
     * @brief  Moving the motor towards a specified direction for a certain number of steps.
     * @param  direction The direction of rotation.
     * @param  steps The desired number of steps.
     * @retval None.
     */
    virtual void move(direction_t direction, unsigned int steps)
    {
        L6208_Move((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD), (uint32_t)steps);
    }

    /**
     * @brief  Stopping the motor through an immediate deceleration up to zero speed.
     * @param  None.
     * @retval None.
     */
    virtual void soft_stop(void)
    {
        L6208_SoftStop();
    }

    /**
     * @brief  Stopping the motor through an immediate infinite deceleration.
     * @param  None.
     * @retval None.
     */
    virtual void hard_stop(void)
    {
        L6208_HardStop();
    }

    /**
     * @brief  Disabling the power bridge after performing a deceleration to zero.
     * @param  None.
     * @retval None.
     */
    virtual void soft_hiz(void)
    {
        motorStopMode_t stopMode = L6208_GetStopMode();
        if (stopMode==HIZ_MODE)
        {
            L6208_SoftStop();
        }
        else
        {
            L6208_SetStopMode(HIZ_MODE);
            L6208_SoftStop();
            L6208_SetStopMode(stopMode);
        }
    }

    /**
     * @brief  Disabling the power bridge immediately.
     * @param  None.
     * @retval None.
     */
    virtual void hard_hiz(void)
    {
        L6208_HardHiZ();
    }

    /**
     * @brief  Waiting while the motor is active.
     * @param  None.
     * @retval None.
     */
    virtual void wait_while_active(void)
    {
        L6208_WaitWhileActive();
    }  

    /**
     * @brief Public functions NOT inherited
     */
     
    /**
     * @brief  Attaching an error handler.
     * @param  fptr An error handler.
     * @retval None.
     */
    virtual void attach_error_handler(void (*fptr)(uint16_t error))
    {
        L6208_AttachErrorHandler((void (*)(uint16_t error)) fptr);
    }

    /**
     * @brief  Checks if the device is disabled or/and has an alarm flag set 
     * by reading the EN pin position.
     * @param  None.
     * @retval One if the EN pin is low (the device is disabled or/and 
     * has an alarm flag set), otherwise zero.
     */
    virtual unsigned int check_status_hw(void)
    {
        if (!flag_and_enable.read())
        {
            return 0x01;
        }
        else
        {
            return 0x00;
        }
    }
    
    /**
     * @brief  Disabling the device.
     * @param  None.
     * @retval None.
     */
    virtual void disable(void)
    {
        L6208_Disable();
    }
    
    /**
     * @brief  Enabling the device.
     * @param  None.
     * @retval None.
     */
    virtual void enable(void)
    {
        L6208_Enable();
    }
    
    /**
     * @brief  Getting the motor decay mode.
     * @param  None.
     * @retval The motor decay mode.
     */
    virtual motorDecayMode_t get_decay_mode()
    {
         return L6208_get_decay_mode();
    }
    
    /**
     * @brief  Set the frequency of the VREFA and VREFB PWM
     * @param  frequency in Hz
     * @retval None.
     */  
    virtual uint32_t get_freq_vref_pwm(void)
    {
        return L6208_VrefPwmGetFreq();
    }    
    
    /**
     * @brief  Getting the version of the firmware.
     * @param  None.
     * @retval The version of the firmware.
     */
    virtual unsigned int get_fw_version(void)
    {
        return (unsigned int) L6208_GetFwVersion();
    }
    
    /**
     * @brief  Getting the motor step mode.
     * @param  None.
     * @retval The motor step mode.
     */
    virtual step_mode_t get_step_mode(void)
    {
      return (step_mode_t) L6208_GetStepMode();
    }
    
    /**
     * @brief  Getting the motor stop mode.
     * @param  None.
     * @retval The motor stop mode.
     */  
    virtual motorStopMode_t get_stop_mode(void)
    {
      return L6208_GetStopMode();
    }
    
    /**
     * @brief  Going to a specified position with a specificied direction.
     * @param  direction The desired direction.
     * @param  position The desired position.
     * @retval None.
     */
    virtual void go_to(direction_t direction, signed int position)
    {
        L6208_GoToDir((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD),(int32_t)position);
    }

    /**
     * @brief  Release the L6208 reset (Reset pin set to high level).
     * @param  None.
     * @retval None.
     */    
    virtual void release_reset(void)
    {
        L6208_ReleaseReset();
    }

    /**
     * @brief  Reset the device with current step mode, resets current speed,
     * positions and microstep variables.
     * @param  None.
     * @retval None.
     */    
    virtual void reset(void)
    {
        L6208_Reset();
    }
    
    /**
     * @brief  Reset the L6208 (Reset pin set to low level).
     * @param  None.
     * @retval None.
     */    
    virtual void reset_device(void)
    {
        L6208_ResetDevice();
    }
   
    /**
     * @brief  Set the motor decay mode.
     * @param  decayMode The desired decay mode (SLOW_DECAY or FAST_DECAY).
     * @retval None.
     */    
    virtual void set_decay_mode(motorDecayMode_t decayMode)
    {
        L6208_SetDecayMode(decayMode);
    }

    /**
     * @brief  Set the motor direction.
     * @param  direction The desired direction.
     * @retval None.
     */
    virtual void set_direction(direction_t direction)
    { 
        L6208_SetDirection((motorDir_t) (direction == StepperMotor::FWD ? FORWARD : BACKWARD));
    }

    /**
     * @brief  Set the frequency of the VREFA and VREFB PWM
     * @param  frequency in Hz
     * @retval None.
     */  
    virtual void set_freq_vref_pwm(uint32_t frequency)
    {
        L6208_VrefPwmSetFreq(frequency);
    }

    /**
     * @brief  Set the motor stop mode.
     * @param  stopMode The desired stop mode (HOLD_MODE or HIZ_MODE).
     * @retval None.
     */       
    virtual void set_stop_mode(motorStopMode_t stopMode)
    {
        L6208_SetStopMode(stopMode);
    }
   
    /*** Public Interrupt Related Methods ***/

    /* ACTION 6 --------------------------------------------------------------*
     * Implement here interrupt related methods, if any.                      *
     * Note that interrupt handling is platform dependent, e.g.:              *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq(pin); //Interrupt object.                  *
     *     feature_irq.rise(callback);   //Attach a callback.                 *
     *     feature_irq.mode(PullNone);   //Set interrupt mode.                *
     *     feature_irq.enable_irq();     //Enable interrupt.                  *
     *     feature_irq.disable_irq();    //Disable interrupt.                 *
     *   + Arduino:                                                           *
     *     attachInterrupt(pin, callback, RISING); //Attach a callback.       *
     *     detachInterrupt(pin);                   //Detach a callback.       *
     *                                                                        *
     * Example (mbed):                                                        *
     *   void attach_feature_irq(void (*fptr) (void))                         *
     *   {                                                                    *
     *     feature_irq.rise(fptr);                                            *
     *   }                                                                    *
     *                                                                        *
     *   void enable_feature_irq(void)                                        *
     *   {                                                                    *
     *     feature_irq.enable_irq();                                          *
     *   }                                                                    *
     *                                                                        *
     *   void disable_feature_irq(void)                                       *
     *   {                                                                    *
     *     feature_irq.disable_irq();                                         *
     *   }                                                                    *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Attaching an interrupt handler to the FLAG interrupt.
     * @param  fptr An interrupt handler.
     * @retval None.
     */
     
    void attach_flag_irq(void (*fptr)(void))
    {
        flag_and_enable.mode(PullDown);
        flag_and_enable.fall(fptr);
    }
    
    /**
     * @brief  Enabling the FLAG interrupt handling.
     * @param  None.
     * @retval None.
     */
    void enable_flag_irq(void)
    {
        flag_and_enable.enable_irq();
    }
    
protected:

    /*** Protected Component Related Methods ***/

    /* ACTION 7 --------------------------------------------------------------*
     * Declare here the component's specific methods.                         *
     * They should be:                                                        *
     *   + Methods with the same name of the C component's virtual table's    *
     *     functions (1);                                                     *
     *   + Methods with the same name of the C component's extended virtual   *
     *     table's functions, if any (2);                                     *
     *   + Helper methods, if any, like functions declared in the component's *
     *     source files but not pointed by the component's virtual table (3). *
     *                                                                        *
     * Example:                                                               *
     *   status_t COMPONENT_get_value(float *f);   //(1)                      *
     *   status_t COMPONENT_enable_feature(void);  //(2)                      *
     *   status_t COMPONENT_compute_average(void); //(3)                      *
     *------------------------------------------------------------------------*/
    status_t L6208_Init(void *init);
    status_t L6208_ReadID(uint8_t *id);
    void L6208_AttachErrorHandler(void (*callback)(uint16_t error));
    void L6208_Disable(void);
    void L6208_ErrorHandler(uint16_t error);
    void L6208_Enable(void);
    uint16_t L6208_GetAcceleration(void);
    uint16_t L6208_GetCurrentSpeed(void);
    uint16_t L6208_GetDeceleration(void);
    motorDecayMode_t L6208_get_decay_mode(void);
    motorDir_t L6208_GetDirection(void);
    uint32_t L6208_GetFwVersion(void);
    int32_t L6208_GetMark(void);
    uint16_t L6208_GetMaxSpeed(void);
    uint16_t L6208_GetMinSpeed(void);
    motorState_t L6208_GetMotionState(void);
    int32_t L6208_GetPosition(void);
    motorStepMode_t L6208_GetStepMode(void);
    motorStopMode_t L6208_GetStopMode(void);    
    void L6208_GoHome(void);  
    void L6208_GoMark(void);
    void L6208_GoTo(int32_t targetPosition);
    void L6208_GoToDir(motorDir_t direction, int32_t targetPosition);   
    void L6208_HardHiZ(void);
    void L6208_HardStop(void);
    void L6208_Move(motorDir_t direction, uint32_t stepCount);
    void L6208_ReleaseReset(void);
    void L6208_Reset(void);
    void L6208_ResetDevice(void);  
    void L6208_Run(motorDir_t direction);
    bool L6208_SetAcceleration(uint16_t newAcc);
    void L6208_SetDecayMode(motorDecayMode_t decayMode);
    bool L6208_SetDeceleration(uint16_t newDec);
    void L6208_SetDirection(motorDir_t direction);
    void L6208_SetHome(void);
    void L6208_SetMark(void);
    bool L6208_SetMaxSpeed(uint16_t volatile newSpeed);
    bool L6208_SetMinSpeed(uint16_t volatile newSpeed);    
    bool L6208_SetStepMode(motorStepMode_t stepMode);
    void L6208_SetStopMode(motorStopMode_t stopMode);    
    bool L6208_SoftStop(void);
    void L6208_TickHandler(void);
    uint32_t L6208_VrefPwmGetFreq(void);
    void L6208_VrefPwmSetFreq(uint32_t newFreq);
    void L6208_WaitWhileActive(void);
    
    /*** Functions intended to be used only internally ***/
    
    void L6208_ClearSysFlag(uint32_t mask);
    uint32_t L6208_ComputeNbAccOrDecSteps(uint16_t accOrDecRate);
    uint16_t L6208_ConvertAcceDecelRateValue(uint16_t newAccOrDecRate);
    void L6208_DoAccel(void);
    void L6208_DoDecel(void);
    void L6208_DoRun(void);
    uint8_t L6208_GetMicrostepSample2Scale(void);
    void L6208_Indexmodeinit(void);
    bool L6208_IsSysFlag(uint32_t mask);
    void L6208_ResetSteps(void);
    uint32_t L6208_ScaleWaveformSample(uint8_t sampleIndex);
    void L6208_ScaleWaveformTable(void);
    void L6208_SetDeviceParamsToGivenValues(l6208_init_t* pInitDevicePrm);
    void L6208_SetDeviceParamsToPredefinedValues(void);
    void L6208_SetMicrostepSample2Scale(uint8_t value);
    void L6208_SetMicrostepSample2Update(uint8_t value);
    void L6208_SetMotionState(motorState_t newMotionState);
    bool L6208_SetSpeed(uint16_t newSpeed, uint32_t volatile *pSpeed);
    void L6208_SetSysFlag(uint32_t mask);
    bool L6208_StartMovement(void);
    void L6208_UpdateScanWaveformTable(void);
    void L6208_UstepWaveformHandling(void);
    bool L6208_VectorCalc(uint8_t newTorque);
    bool L6208_VrefPwmComputePulseWidth(uint8_t bridgeId, uint16_t value, bool valueIsPwmDutyCycle);
    void L6208_VrefPwmUpdatePulseWidth(void);
    
    /*** Component's I/O Methods ***/

    /* ACTION 8 --------------------------------------------------------------*
     * Implement here other I/O methods beyond those already implemented      *
     * above, which are declared extern within the component's header file.   *
     *------------------------------------------------------------------------*/
    /**
     * @brief  Reset the clock pin.
     * @param  None.
     * @retval None.
     */    
    void L6208_Board_CLOCK_PIN_Reset(void)
    {
        clock_pin = 0;
    }    
    /**
     * @brief  Set the clock pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_CLOCK_PIN_Set(void)
    {
        clock_pin = 1;
    }

    /**
     * @brief  Set the control pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_CONTROL_PIN_Set(void)
    {
        control_pin = 1;
    }

    /**
     * @brief  Reset the control pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_CONTROL_PIN_Reset(void)
    {
        control_pin = 0;
    }
    
    /**
     * @brief  Making the CPU wait.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_Delay(uint32_t delay)
    {
        wait_ms(delay);
    }
    
    /**
     * @brief  Reset the dir pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_DIR_PIN_Reset(void)
    {
        direction_pin = 0;
    }
        
    /**
     * @brief  Set the dir pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_DIR_PIN_Set(void)
    {
        direction_pin = 1;
    }
    
    /**
     * @brief  Disable the power bridges (leave the output bridges HiZ).
     * @param  None.
     * @retval None.
     */
    void L6208_Board_Disable(void)
    {
        flag_and_enable.disable_irq();
        flag_and_enable.mode(PullDown);
    }

    /**
     * @brief  Disabling interrupts.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_DisableIrq(void)
    {
        __disable_irq();
    }
    
    /**
     * @brief  Enable the power bridges (leave the output bridges HiZ).
     * @param  None.
     * @retval None.
     */
    void L6208_Board_Enable(void)
    {     
        flag_and_enable.mode(PullUp);
        flag_and_enable.enable_irq();
    }
    
    /**
     * @brief  Enabling interrupts.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_EnableIrq(void)
    {
        __enable_irq();
    }
    
    /**
     * @brief  Reset the half full pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_HALF_FULL_PIN_Reset(void)
    {
        half_full_pin = 0;
    }
    
    /**
     * @brief  Set the half full pin.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_HALF_FULL_PIN_Set(void)
    {
        half_full_pin = 1;
    }

    /**
     * @brief  Initialising the the VREFA or VREFB PWM.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_VrefPwmInit(uint8_t bridgeId, uint32_t pwmFreq) {}
        
    /**
     * @brief  Exit the device from reset mode.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_ReleaseReset(void)
    {
        reset_pin = 1;
    }

    /**
     * @brief  Put the device in reset mode.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_Reset(void)
    {
        reset_pin = 0;
    }

    /**
     * @brief  Get the tick timer frequency in Hz.
     * @param  None.
     * @retval The tick timer frequency in Hz.
     */    
    uint32_t L6208_Board_TickGetFreq(void)
    {
       return TIMER_TICK_FREQUENCY;
    }

    /**
     * @brief  Initialising the tick.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_TickInit(void) {}

    /**
     * @brief  Starting the tick timer, setting its frequency
     * and attaching a tick handler function to it.
     * @param  frequency the frequency of the tick.
     * @retval None.
     */
    void L6208_Board_TickStart(uint16_t frequency)
    {
        /* Computing the period of the tick. */
        double period = 1.0f / frequency;
        
        /* Attaching a tick handler function which updates */
        /* the state machine every elapsed period time. */
        ticker.attach(this, &L6208::L6208_TickHandler, period);
    }

    /**
     * @brief  Stopping the tick.
     * @param  None.
     * @retval None.
     */
    void L6208_Board_TickStop(void)
    {
        ticker.detach();
    }

    /**
     * @brief Set the pulse width of the VREFA PWM.
     * @param[in] pulseWidthInUs pulse width of the PWM in microsecond.
     * @retval None.
     */     
    void L6208_Board_VrefPwmSetPulseWidthA(uint8_t pulseWidthInUs)
    {
      vrefA_pwm.pulsewidth_us(pulseWidthInUs);
    }
 
    /**
     * @brief Set the pulse width of the VREFB PWM.
     * @param[in] pulseWidthInUs pulse width of the PWM in microsecond.
     * @retval None.
     */       
    void L6208_Board_VrefPwmSetPulseWidthB(uint8_t pulseWidthInUs)
    {
      vrefB_pwm.pulsewidth_us(pulseWidthInUs);
    }    
    
    /**
     * @brief  Start the timer for the VREFA or VREFB PWM.
     * @param[in] bridgeId
     *            0 for BRIDGE_A
     *            1 for BRIDGE_B
     * @param[in] pwmPeriod period of the PWM used to generate the reference
     * voltage for the bridge.
     * @retval "true" in case of success, "false" otherwise.
     * @note the unit is 1/256th of a microsecond. The VREFA PWM must be started
     * before the VREFB PWM.
     */   
    bool L6208_Board_VrefPwmStart(uint8_t bridgeId, uint16_t pwmPeriod)
    { 
        pwmPeriod>>=8;
        /* Setting the period and the duty-cycle of PWM. */
        if (bridgeId == 0)
        {
            vrefA_pwm.period_us(pwmPeriod);         
        }
        else if (bridgeId == 1)
        {
            vrefB_pwm.period_us(pwmPeriod);
        }
        else
        {
            return false;
        }
        return true;
    }
    
    /**
     * @brief  Stop the timer for the VREFA or VREFB PWM.
     * @param[in] bridgeId
     *            0 for BRIDGE_A
     *            1 for BRIDGE_B
     * @retval "true" in case of success, "false" otherwise.
     */       
    bool L6208_Board_VrefPwmStop(uint8_t bridgeId)
    {
        if (bridgeId == 0)
        {
            vrefA_pwm.period_us(0);
        }
        else if (bridgeId == 1)
        {
            vrefB_pwm.period_us(0);
        }
        else
        {
            return false;
        }
        return true;
    }

protected:

    /*** Component's Instance Variables ***/

    /* ACTION 9 --------------------------------------------------------------*
     * Declare here interrupt related variables, if needed.                   *
     * Note that interrupt handling is platform dependent, see                *
     * "Interrupt Related Methods" above.                                     *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     InterruptIn feature_irq;                                           *
     *------------------------------------------------------------------------*/
    /* Flag Interrupt and chip enable. */
    InterruptIn flag_and_enable;

    /* ACTION 10 -------------------------------------------------------------*
     * Declare here other pin related variables, if needed.                   *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut standby_reset;                                          *
     *------------------------------------------------------------------------*/
    /* RESET pin. */
    DigitalOut reset_pin;
    /* CW_CCW pin. */
    DigitalOut direction_pin;
    /* HALF_FULL pin */
    DigitalOut half_full_pin;
    /* CONTROL pin */
    DigitalOut control_pin;
    /* CLOCK pin */
    DigitalOut clock_pin;

    /* Pulse Width Modulation pin for VREFA pin */
    PwmOut vrefA_pwm;
    /* Pulse Width Modulation pin for VREFA pin */
    PwmOut vrefB_pwm;
    
    /* Timer for the tick */
    Ticker ticker;
    
    /* ACTION 11 -------------------------------------------------------------*
     * Declare here communication related variables, if needed.               *
     *                                                                        *
     * Example:                                                               *
     *   + mbed:                                                              *
     *     DigitalOut ssel;                                                   *
     *     DevSPI &dev_spi;                                                   *
     *------------------------------------------------------------------------*/
    /* Configuration. */

    /* IO Device. */

    /* ACTION 12 -------------------------------------------------------------*
     * Declare here identity related variables, if needed.                    *
     * Note that there should be only a unique identifier for each component, *
     * which should be the "who_am_i" parameter.                              *
     *------------------------------------------------------------------------*/
    /* Identity */
    uint8_t who_am_i;

    /* ACTION 13 -------------------------------------------------------------*
     * Declare here the component's static and non-static data, one variable  *
     * per line.                                                              *
     *                                                                        *
     * Example:                                                               *
     *   float measure;                                                       *
     *   int instance_id;                                                     *
     *   static int number_of_instances;                                      *
     *------------------------------------------------------------------------*/
    /* Data. */
    void (*errorHandlerCallback)(uint16_t error);
    deviceParams_t devicePrm;
    uint8_t deviceInstance;
    uint32_t tickFreq;
    /// microstepping PWM period and torque scaled waveform samples array
    uint16_t updatedMicroTable[L6208_USTEPS_PER_QUARTER_PERIOD+1];
    /// waveform scanning microstepping PWM period sample arrays for VREFA wave
    uint16_t microTable1[L6208_USTEPS_PER_QUARTER_PERIOD*3+1];
    /// waveform scanning microstepping PWM period sample array for VREFB wave
    uint16_t *pMicroTable2;

    /* Static data. */
    static uint8_t numberOfDevices;
    static const uint16_t RefMicroTable[L6208_USTEPS_PER_QUARTER_PERIOD*3];
};

#endif // __L6208_CLASS_H

/************************ (C) COPYRIGHT STMicroelectronics *****END OF FILE****/
