/**
 ******************************************************************************
 * @file    main.cpp
 * @author  SRA
 * @version V1.0.0
 * @date    5-March-2019
 * @brief   Simple Example application for using the X_NUCLEO_IKS01A3
 *          MEMS Inertial & Environmental Sensor Nucleo expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2019 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/

/* Includes */
#include "mbed.h"
#include "rtos.h"
#include "XNucleoIKS01A3.h"

/* Instantiate the expansion board */
static XNucleoIKS01A3 *mems_expansion_board = XNucleoIKS01A3::instance(D14, D15, D4, D5, A3, D6, A4);

/* Retrieve the composing elements of the expansion board */
static STTS751Sensor *t_sensor = mems_expansion_board->t_sensor;

DigitalOut myled(LED1);

volatile int mems_event = 0;
uint32_t previous_tick = 0;
uint32_t current_tick = 0;
uint8_t high = 0, low = 0;
float temperature = 0.0f;
char buffer[32];

void INT_cb();

/* Helper function for printing floats & doubles */
static char *print_double(char *str, double v, int decimalDigits = 2)
{
    int i = 1;
    int intPart, fractPart;
    int len;
    char *ptr;

    /* prepare decimal digits multiplicator */
    for (; decimalDigits != 0; i *= 10, decimalDigits--);

    /* calculate integer & fractinal parts */
    intPart = (int)v;
    fractPart = (int)((v - (double)(int)v) * i);

    /* fill in integer part */
    sprintf(str, "%i.", intPart);

    /* prepare fill in of fractional part */
    len = strlen(str);
    ptr = &str[len];

    /* fill in leading fractional zeros */
    for (i /= 10; i > 1; i /= 10, ptr++) {
        if (fractPart >= i) {
            break;
        }
        *ptr = '0';
    }

    /* fill in (rest of) fractional part */
    sprintf(ptr, "%i", fractPart);

    return str;
}

/* Simple main function */
int main()
{
    /* Attach callback to STTS751 INT */
    t_sensor->attach_int_irq(&INT_cb);

    /* Enable STTS751 temperature sensor */
    t_sensor->enable();
    /* Set ODR to 4Hz */
    t_sensor->set_odr(4.0f);
    /* Set Low Temperature Threshold */
    t_sensor->set_low_temp_thr(22.0f);
    /* Set High Temperature Threshold */
    t_sensor->set_high_temp_thr(28.0f);
    /* Enable Event pin */
    t_sensor->set_event_pin(1);
    /* Get beginning status */
    t_sensor->get_temp_limit_status(NULL, NULL, NULL);

    previous_tick = clock();

    printf("\r\n--- Starting new run ---\r\n");

    while (1) {
        if (mems_event) {
            mems_event = 0;
            uint8_t high_temp = 0, low_temp = 0;
            t_sensor->get_temp_limit_status(&high_temp, &low_temp, NULL);
            if (high_temp) {
                high = 1;
                low = 0;
            }
            if (low_temp) {
                low = 1;
                high = 0;
            }

            t_sensor->get_temperature(&temperature);
            myled = 1;
            ThisThread::sleep_for(100);
            myled = 0;
        }

        current_tick = clock();
        if (((current_tick - previous_tick) / CLOCKS_PER_SEC) >= 2) {
            if (!high && !low) {
                t_sensor->get_temperature(&temperature);
            }
            printf("Temp[C]: ");
            printf("%7s C", print_double(buffer, temperature));
            if (high) {
                printf(" High temperature detected!(>28C) \r\n");
                high = 0;
            } else if (low) {
                printf(" Low temperature detected!(<22C) \r\n");
                low = 0;
            } else {
                printf("\r\n");
            }
            previous_tick = clock();
        }
    }
}

void INT_cb()
{
    mems_event = 1;
}
