/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Fabio Brembilla
 * @version V2.0.0
 * @date    February, 2016
 * @brief   SunTracker + RemoteControl Vertical Application
 *          This application use IHM01A1, 6180XA1, IKS01A1, IDB0XA1 expansion boards
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/

// Mbed specific header files
#include "mbed.h"

// Helper header files
#include "DevSPI.h"
#include "DevI2C.h"

// Component specific header files
#include "l6474_class.h"
#include "x_nucleo_6180xa1.h"
#include "x_nucleo_iks01a1.h"

// C header files
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/* BlueTooth -----------------------------------------------------------------*/

#include "debug.h" // Need for PRINTF
#include "Utils.h" // Need for STORE_LE_16 and _32

typedef struct {
    int32_t AXIS_X;
    int32_t AXIS_Y;
    int32_t AXIS_Z;
} AxesRaw_TypeDef;

typedef enum ConnectionStatus_t {
    DISCONNECTED    =0,
    CONNECTED       =1
}cns_t;

const unsigned   LENGTH_OF_LONG_UUID = 16;
typedef uint16_t ShortUUIDBytes_t;
typedef uint8_t  LongUUIDBytes_t[LENGTH_OF_LONG_UUID];

#include "CustomControlService.h"
#include "CustomSensorsService.h"

static BLE *p_BLEdev = NULL;

#define BLE_DEV_NAME "SunTracker"
#define BLE_DEV_MAC 0xAA,0xBB,0xCC,0xDD,0xEE,0xFF
#define BLE_ADVERTISING_INTERVAL 1000

static CustomControlService *p_customcontrolservice = NULL;
static CustomSensorService  *p_customsensorservice = NULL;

/* Definitions ---------------------------------------------------------------*/

#define SET_ACC 400     // Set Motor Acceleration
#define SET_DEC 400     // Set Motor Deceleration
#define SET_MAX 200     // Set Motor MaxSpeed
#define SET_MIN 100     // Set Motor MinSpeed
#define STOP 1000       // Set Motor Stop Position
#define TOLLERANCE 100  // Tollerance between Left and Right before Start Movement
#define RANGE_1 200     // Range 1 for Motor Speed
#define RANGE_2 500     // Range 2 for Motor Speed

/* Variables -----------------------------------------------------------------*/

int16_t dir=0;          // Motor Rotation Direction: 0 = Stop, 1 = Anticlockwise, 2 = Clockwise
int16_t changedir=0;    // Change Direction: 0 = No, 1 = Yes
int16_t babybear=0;     // Difference (in Lux) between Left and Right
int acc_data[3];        // Difference of Accelerometer
int16_t diff=0;         // Abs of Babybear or Accelerometer difference
int16_t left=0;         // Left Command for Rotate Direction
int16_t right=0;        // Right Command for Rotate Direction
int16_t start=0;        // Waiting User Button Push
int32_t pos=0;          // Motor Position
char DisplayStr[5];     // Status Display
int16_t Display=0;      // Shown on Display: 0 = Motor Speed, 1 = Solar Panel Value, 2 = Manual Control
int16_t status, status_t, status_b, status_l, status_r; // Babybear Status

/* ---------------------------------------------------------------------------*/  

//void DISP_ExecLoopBody(void) {};

//AnalogIn analog_read(A1); // A1 Conflict with BLE SPI_CS --> Changed in A????


/* Bluetooth CallBack ---------------------------------------------------------*/

static void onUpdatesEnabledCallback(GattAttribute::Handle_t handle)
{
    
}

static void onUpdatesDisabledCallback(Gap::Handle_t handle)
{
    
}

static void onDataReadCallback(const GattReadCallbackParams *eventDataP)
{
    
}

static void myonDataWriteCallback(const GattWriteCallbackParams *eventDataP)
{
    printf ("myonDataWriteCallback attr_handle: %x  att_data[3]: %x  data_length: %d\n\r", eventDataP->handle, eventDataP->data[3], eventDataP->len );
    
    if (p_customcontrolservice->isContHandle(eventDataP->handle)) // Only if write come from this Service
    {
        uint8_t Command = eventDataP->data[3];  // Riceve 8 byte: data[0] + data[1] + data[2] + data[3]
        printf("myonDataWriteCallback (Command %x)\r\n", Command);
    } 

}

static void onConnectionCallback(const Gap::ConnectionCallbackParams_t * connectionParams)
{
    printf("onConnectionCallback (Line %d)\r\n", __LINE__);   
}

static void onDisconnectionCallback(const Gap::DisconnectionCallbackParams_t * disConnectionReason)
{
    printf("onDisconnectionCallback (Line %d)\r\n", __LINE__);
    p_BLEdev->gap().startAdvertising();
}

/* Main ----------------------------------------------------------------------*/

int main()
{

    // Printing to the console
    printf("SunTracker by Fabio Brembilla\r\n\n");
    
    // BLE Initialization
    p_BLEdev = new BLE;
    if (!p_BLEdev) { printf("\r\nBLE Device creation failed\r\n"); }
    const Gap::Address_t BLE_address_BE = {BLE_DEV_MAC};        
    p_BLEdev->gap().setAddress(BLEProtocol::AddressType::PUBLIC, BLE_address_BE);
    
    p_BLEdev->init();

    // BLE CallBack Functions
    p_BLEdev->gattServer().onUpdatesEnabled(onUpdatesEnabledCallback);
    p_BLEdev->gattServer().onUpdatesDisabled(onUpdatesDisabledCallback);
    p_BLEdev->gattServer().onDataRead(onDataReadCallback);
    p_BLEdev->gattServer().onDataWritten(myonDataWriteCallback);
    p_BLEdev->gap().onConnection(onConnectionCallback);
    p_BLEdev->gap().onDisconnection(onDisconnectionCallback);

    // BLE Services
    p_customcontrolservice =  new CustomControlService(*p_BLEdev);   
    p_customsensorservice = new CustomSensorService(*p_BLEdev);

    // BLE Advertising
    const static char DEVICE_NAME[] = BLE_DEV_NAME;
    p_BLEdev->gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    uint8_t dat[] = {0x01,0x80,0x00,0xFC,0x00,0x00};  
    p_BLEdev->gap().accumulateScanResponse(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA,dat,6);    
    p_BLEdev->gap().accumulateAdvertisingPayload(GapAdvertisingData::UNKNOWN);
    p_BLEdev->gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    p_BLEdev->gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    p_BLEdev->gap().setAdvertisingInterval(BLE_ADVERTISING_INTERVAL);
    p_BLEdev->gap().startAdvertising();
      
    printf("BLE_Initialization OK (Line %d)\r\n", __LINE__);
    
    // Initializing SPI bus
    //DevSPI dev_spi(D11, D12, D13);
    
    // Initializing I2C bus
    DevI2C dev_i2c(D14, D15);  
    
    dev_i2c.frequency(100000);
    
    // Initializing Babybear Component 6180XA1
    static X_NUCLEO_6180XA1 *board;
    MeasureData_t data_sensor_top, data_sensor_left, data_sensor_right;
    
    // Initializing Babybear Component
    board=X_NUCLEO_6180XA1::Instance(&dev_i2c, PA_4, PA_4, PA_4, PA_4); // Non usare NC altrimenti non funziona!!! InterruptIn fa casino!!!

    status=board->InitBoard();
    if(status) VL6180x_ErrLog("Failed to init the board!\n\r");

    // Put GPIO not used as Interrupt in Hi-Z
    status_t=board->sensor_top->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF);
    //status_b=board->sensor_botton->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF); No Present
    //status_l=board->sensor_left->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF);
    //status_r=board->sensor_right->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF);
    
    // Set Babybears
    //status_l=board->sensor_left->AlsSetAnalogueGain(3);
    //status_r=board->sensor_right->AlsSetAnalogueGain(3);
    //status_l=board->sensor_left->StartMeasurement(als_continuous_polling, NULL, NULL, NULL);
    //status_r=board->sensor_right->StartMeasurement(als_continuous_polling, NULL, NULL, NULL);
    
    status_t=board->sensor_top->AlsSetAnalogueGain(3);
    status_t=board->sensor_top->StartMeasurement(als_continuous_polling, NULL, NULL, NULL);

    printf("Initialization OK (Line %d)\r\n", __LINE__);

    printf("Start Main Loop\r\n");

    //static int INTLOOP=0;

    // Main Loop
    while(true) {
        
        status_t=board->sensor_top->GetMeasurement(als_continuous_polling, &data_sensor_top);
        //status_l=board->sensor_left->GetMeasurement(als_continuous_polling, &data_sensor_left);
        //status_r=board->sensor_right->GetMeasurement(als_continuous_polling, &data_sensor_right);
    
        //babybear = data_sensor_right.lux - data_sensor_left.lux;
    
        //diff = abs(babybear);
        
        diff = data_sensor_top.lux;
        
        //INTLOOP++;
        //if (INTLOOP==100) { p_customsensorservice->sendEnvPosition(rand(),0); INTLOOP=0; }
        
        wait_ms(100);
        printf("babybear %d\r\n", diff);
        
        //sprintf(DisplayStr, "%d", diff);
        //board->display->DisplayString(DisplayStr, 4);
            
        //strcpy(DisplayStr,"pusH");
        //board->display->DisplayString(DisplayStr, 4);
            
        p_BLEdev->waitForEvent();
        }
}

// Configuro prima BLE e poi Babybear funziona, ma quando mi collego con il BLE l'app va in crash con il commento.
// “mbed assertation failed: (hz > 0) && (hz <= 400000), file: C:\Code\git_repo\github\mbed-official\libraries\mbed\targets\hal\TARGET_STM\TARGET_STM32F4\i2c_api.c, line 119”

// Configuro prima Babybear e poi BLE non funziona.
