/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Fabio Brembilla
 * @version V2.0.0
 * @date    February, 2016
 * @brief   SunTracker + RemoteControl Vertical Application
 *          This application use IHM01A1, 6180XA1, IKS01A1, IDB0XA1 expansion boards
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */

/* Includes ------------------------------------------------------------------*/

// Mbed specific header files
#include "mbed.h"

// Helper header files
#include "DevSPI.h"
#include "DevI2C.h"

// Component specific header files
#include "l6474_class.h"
#include "x_nucleo_6180xa1.h"
#include "x_nucleo_iks01a1.h"

// C header files
#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <assert.h>

/* BlueTooth -----------------------------------------------------------------*/

#include "debug.h" // Need for PRINTF
#include "Utils.h" // Need for STORE_LE_16 and _32

typedef struct {
    int32_t AXIS_X;
    int32_t AXIS_Y;
    int32_t AXIS_Z;
} AxesRaw_TypeDef;

typedef enum ConnectionStatus_t {
    DISCONNECTED    =0,
    CONNECTED       =1
}cns_t;

const unsigned   LENGTH_OF_LONG_UUID = 16;
typedef uint16_t ShortUUIDBytes_t;
typedef uint8_t  LongUUIDBytes_t[LENGTH_OF_LONG_UUID];

#include "CustomControlService.h"
#include "CustomSensorsService.h"

static BLE *p_BLEdev = NULL;

#define BLE_DEV_NAME "SunTracker"
#define BLE_DEV_MAC 0xAA,0xBB,0xCC,0xDD,0xEE,0xFF
#define BLE_ADVERTISING_INTERVAL 1000

/* Definitions ---------------------------------------------------------------*/

#define SET_ACC 400     // Set Motor Acceleration
#define SET_DEC 400     // Set Motor Deceleration
#define SET_MAX 200     // Set Motor MaxSpeed
#define SET_MIN 100     // Set Motor MinSpeed
#define STOP 1000       // Set Motor Stop Position
#define TOLLERANCE 100  // Tollerance between Left and Right before Start Movement
#define RANGE_1 200     // Range 1 for Motor Speed
#define RANGE_2 500     // Range 2 for Motor Speed

/* Variables -----------------------------------------------------------------*/

int16_t dir=0;          // Motor Rotation Direction: 0 = Stop, 1 = Anticlockwise, 2 = Clockwise
int16_t changedir=0;    // Change Direction: 0 = No, 1 = Yes
int16_t babybear=0;     // Difference (in Lux) between Left and Right
int acc_data[3];        // Difference of Accelerometer
int16_t diff=0;         // Abs of Babybear or Accelerometer difference
int16_t left=0;         // Left Command for Rotate Direction
int16_t right=0;        // Right Command for Rotate Direction
int16_t start=0;        // Waiting User Button Push
int32_t pos=0;          // Motor Position
char DisplayStr[5];     // Status Display
int16_t Display=0;      // Shown on Display: 0 = Motor Speed, 1 = Solar Panel Value, 2 = Manual Control
int16_t status, status_t, status_b, status_l, status_r; // Babybear Status

/* ---------------------------------------------------------------------------*/

// Motor Control Component
static L6474 *motor;

// Initializing SPI bus
DevSPI dev_spi(D11, D12, D13);

// Initializing I2C bus
DevI2C dev_i2c(D14, D15);

// Instance board 6180XA1
//static X_NUCLEO_6180XA1 *board=X_NUCLEO_6180XA1::Instance(&dev_i2c, NC, NC, NC, NC);
//MeasureData_t data_sensor_top, data_sensor_bottom, data_sensor_left, data_sensor_right;
static X_NUCLEO_6180XA1 *board;
MeasureData_t data_sensor_top, data_sensor_bottom, data_sensor_left, data_sensor_right;
    
// Instance mems IKS01A1
//static X_NUCLEO_IKS01A1 *mems=X_NUCLEO_IKS01A1::Instance(&dev_i2c);
//MotionSensor *accelerometer = mems->GetAccelerometer();
static X_NUCLEO_IKS01A1 *mems;
MotionSensor *accelerometer;

void DISP_ExecLoopBody(void) {};

//AnalogIn analog_read(A1); // A1 Conflict with BLE SPI_CS --> Changed in A????

InterruptIn mybutton(USER_BUTTON);

/* User_Button_Pressed -------------------------------------------------------*/

void User_Button_Pressed()
{

    if (start>0) {
        Display++;
    }
    if (Display>2) {
        Display=0;
    }
    if (start==0) {
        start=1;
    }
    
    printf("PUSH Display %d Start %d\r\n", Display, start);

}

/* Initialization ------------------------------------------------------------*/

bool Initialization(void)
{
   
    // Initializing MEMS Component
    mems=X_NUCLEO_IKS01A1::Instance(&dev_i2c);
    accelerometer = mems->GetAccelerometer();
   
//----

    // Initializing Babybear Component
    board=X_NUCLEO_6180XA1::Instance(&dev_i2c, NC, NC, NC, NC);

/* FABIO
    status=board->InitBoard();
    if(status) VL6180x_ErrLog("Failed to init the board!\n\r");
*/

    // Put GPIO not used as Interrupt in Hi-Z
    status_t=board->sensor_top->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF);
    //status_b=board->sensor_botton->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF); No Present
    status_l=board->sensor_left->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF);
    status_r=board->sensor_right->SetGPIOxFunctionality(1, GPIOx_SELECT_OFF);    
    
    // Set Babybears
    status_l=board->sensor_left->AlsSetAnalogueGain(3);
    status_r=board->sensor_right->AlsSetAnalogueGain(3);
    status_l=board->sensor_left->StartMeasurement(als_continuous_polling, NULL, NULL, NULL);
    status_r=board->sensor_right->StartMeasurement(als_continuous_polling, NULL, NULL, NULL);

//----

    // Initializing Motor Component
    motor = new L6474(D2, D8, D7, D9, D10, dev_spi);
    if (motor->Init(NULL) != COMPONENT_OK)
        return false;

    motor->SetStepMode(STEP_MODE_1_8);  // Default is STEP_MODE_1_16

    // Set defaults Motor Speed
    motor->SetAcceleration(SET_ACC);
    motor->SetDeceleration(SET_DEC);
    motor->SetMaxSpeed(SET_MAX);        // Variable by Light/Mems Sensors
    motor->SetMinSpeed(SET_MIN);

    return true;

}

/* Measure_Babybear ----------------------------------------------------------*/

void Measure_Babybear(void)
{

    status_l=board->sensor_left->GetMeasurement(als_continuous_polling, &data_sensor_left);
    status_r=board->sensor_right->GetMeasurement(als_continuous_polling, &data_sensor_right);

    babybear = data_sensor_right.lux - data_sensor_left.lux;

    diff = abs(babybear);

    if (babybear>0) {
        left=0;
        right=1;
    }
    if (babybear<0) {
        left=1;
        right=0;
    }

}

/* Measure_Accelerometer -----------------------------------------------------*/

void Measure_Accelerometer(void)
{

    accelerometer->Get_X_Axes(acc_data);

    diff = abs(acc_data[0]);

    if (acc_data[0]>0) {
        left=0;
        right=1;
    }
    if (acc_data[0]<0) {
        left=1;
        right=0;
    }

}

/* Control_Motor -------------------------------------------------------------*/

void Control_Motor(void)
{

    //printf("Diff: %d lux/mems\n\r", diff);
    motor->SetMaxSpeed(diff);

    if (diff>TOLLERANCE) {
        if (diff <=RANGE_1)
        {
            if (left)  { strcpy(DisplayStr,"E___"); }
            if (right) { strcpy(DisplayStr,"___3"); }
        }
        else if (diff >RANGE_1 & diff <=RANGE_2)
        {
            if (left)  { strcpy(DisplayStr,"E==="); }
            if (right) { strcpy(DisplayStr,"===3"); }
        }
        else if (diff >RANGE_2)
        {
            if (left)  { strcpy(DisplayStr,"E~~~"); }
            if (right) { strcpy(DisplayStr,"~~~3"); }
        }

        // In Case of Change Direction
        if (left & dir==2)  { changedir=1; }
        if (right & dir==1) { changedir=1; }

        // Run only if Stop or Change Direction
        if (diff>TOLLERANCE & (dir==0 | changedir==1)) {
            if (left)   { motor->Run(StepperMotor::FWD); dir=1; changedir=0; }
            if (right)  { motor->Run(StepperMotor::BWD); dir=2; changedir=0; }
        }
    }

    // Get Motor Position and Control Rotation Block
    pos = motor->GetPosition();
    if (pos>STOP | pos<-STOP) {
        if (pos>0) { motor->GoTo(STOP); }
        if (pos<0) { motor->GoTo(-STOP); }
        printf("GOTO\n\r"); // Without this command, the motor remain in stop
    }

    // Stop Motor
    if (diff<=TOLLERANCE) { // It continues to send the command to stop the motor. Think to do it just one time
        motor->HardStop();
        if (Display==0) { strcpy(DisplayStr,"----"); }
        if (Display==2) { strcpy(DisplayStr,"E  3"); }
        dir=0;
        changedir=0;
        //printf("STOP\n\r");
    }

}

/* Measure_SolarPanel --------------------------------------------------------*/

void Measure_SolarPanel(void)
{

    // AnalogIn: 0V return 0.0 , 3.3V return 1.0
//    float measure = analog_read.read() * 3300;
    float measure = 0;
    //printf("Measure = %.0f mV\r\n", measure);
    //board->display->DisplayDigit("A", 0);

    if (Display==1) {
        sprintf(DisplayStr, "%.0f", measure);
    }

    board->display->DisplayString(DisplayStr, 4);
    printf("%s\n\r", DisplayStr);

}

/* Bluetooth CallBack ---------------------------------------------------------*/

static void onUpdatesEnabledCallback(GattAttribute::Handle_t handle)
{
    
}

static void onUpdatesDisabledCallback(Gap::Handle_t handle)
{
    
}

static void onDataReadCallback(const GattReadCallbackParams *eventDataP)
{
    
}

static void myonDataWriteCallback(const GattWriteCallbackParams *eventDataP)
{
    
}

static void onConnectionCallback(const Gap::ConnectionCallbackParams_t * connectionParams)
{
    
}

static void onDisconnectionCallback(const Gap::DisconnectionCallbackParams_t * disConnectionReason)
{
    
}

/* Bluetooth Initialization ---------------------------------------------------*/

void BLE_Initialization(void)
{

    p_BLEdev = new BLE;
    if (!p_BLEdev) { printf("\r\nBLE Device creation failed\r\n"); }
    const Gap::Address_t BLE_address_BE = {BLE_DEV_MAC};        
    p_BLEdev->gap().setAddress(BLEProtocol::AddressType::PUBLIC, BLE_address_BE);  

    printf("BLE Init (Line %d)\r\n", __LINE__);

    p_BLEdev->init();
    
    printf("BLE Init OK (Line %d)\r\n", __LINE__);

    // Set BLE CallBack Functions
    p_BLEdev->gattServer().onUpdatesEnabled(onUpdatesEnabledCallback);
    p_BLEdev->gattServer().onUpdatesDisabled(onUpdatesDisabledCallback);
    p_BLEdev->gattServer().onDataRead(onDataReadCallback);
    p_BLEdev->gattServer().onDataWritten(myonDataWriteCallback);
    p_BLEdev->gap().onConnection(onConnectionCallback);
    p_BLEdev->gap().onDisconnection(onDisconnectionCallback);
    //p_BLEdev->gattServer().onConfirmationReceived(onConfirmationReceivedCallback);    
    //p_BLEdev->gattServer().onDataSent(onDataSentCallback);    
    //p_BLEdev->gap().onTimeout(onTimeoutCallback);

    // Setup BLE Advertising
    const static char DEVICE_NAME[] = BLE_DEV_NAME;
    p_BLEdev->gap().accumulateAdvertisingPayload(GapAdvertisingData::BREDR_NOT_SUPPORTED | GapAdvertisingData::LE_GENERAL_DISCOVERABLE);
    #ifdef USE_SENSOR_FUSION_LIB        
    uint8_t dat[] = {0x01,0x80,0x00,0xFC,0x01,0x80};
    #else
    uint8_t dat[] = {0x01,0x80,0x00,0xFC,0x00,0x00};
    #endif      
    p_BLEdev->gap().accumulateScanResponse(GapAdvertisingData::MANUFACTURER_SPECIFIC_DATA,dat,6);    
    p_BLEdev->gap().accumulateAdvertisingPayload(GapAdvertisingData::UNKNOWN);
    p_BLEdev->gap().accumulateAdvertisingPayload(GapAdvertisingData::COMPLETE_LOCAL_NAME, (uint8_t *)DEVICE_NAME, sizeof(DEVICE_NAME));
    p_BLEdev->gap().setAdvertisingType(GapAdvertisingParams::ADV_CONNECTABLE_UNDIRECTED);
    p_BLEdev->gap().setAdvertisingInterval(BLE_ADVERTISING_INTERVAL);
    p_BLEdev->gap().startAdvertising();

}

/* Main ----------------------------------------------------------------------*/

int main()
{

    // Printing to the console
    printf("SunTracker by Fabio Brembilla\r\n\n");

    Initialization();

    printf("Initialization OK (Line %d)\r\n", __LINE__);    
    
    BLE_Initialization();
        
    printf("BLE_Initialization OK (Line %d)\r\n", __LINE__);
        
    mybutton.fall(&User_Button_Pressed);
    
    printf("Main Initializations OK (Line %d)\r\n", __LINE__);
    printf("Wait Push Button\r\n");

    // Loop until push User Button to Set 0 Point
    strcpy(DisplayStr,"pusH");
    while(start<1) {
        board->display->DisplayString(DisplayStr, 4);
        printf("%s\n\r", DisplayStr);
        //wait(0.1); // Need one command otherwise remain always in loop
    }

    printf("Start Main Loop\r\n");

    // Main Loop
    while(true) {
        if (Display==0 | Display==1)    {
            Measure_Babybear();
        }
        if (Display==2)                 {
            Measure_Accelerometer();
        }

        Control_Motor();
        Measure_SolarPanel();
        
        p_BLEdev->waitForEvent();
    }

    //status_l=board->sensor_left->StopMeasurement(als_continuous_polling);
    //status_r=board->sensor_right->StopMeasurement(als_continuous_polling);

}
