/******************************************************************************
  * @file    CustomSensorsService.h
  * @author  Fabio Brembilla
  * @version V1.0.0
  * @date    January, 2016
  * @brief   SunTracker Custom Service for BlueTooth (IDB0XA1 expansion board)
  *****************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
*/

#ifndef __CUSTOM_BLE_SENSORS_SERVICE_H__
#define __CUSTOM_BLE_SENSORS_SERVICE_H__
#include "BLE.h"
#include "UUID.h"

/* BLE Services: Primary + 2 Secondary (Char Desk) */
const LongUUIDBytes_t SENS_SERVICE_UUID_128 =       {0x00,0x00,0x00,0x00,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //Primary 
const LongUUIDBytes_t SENS_POSITION_CHAR_UUID_128 = {0x00,0x00,0x00,0x01,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //Motor Position
const LongUUIDBytes_t SENS_SUNPANEL_CHAR_UUID_128 = {0x00,0x00,0x00,0x02,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //SunPanel Voltage

#define POSITION_DATA_LEN   2+2
#define SUNPANEL_DATA_LEN   2+2

/* Custom Sensors Service */
class CustomSensorService  {
public:
    CustomSensorService(BLEDevice &_ble) :
        ble(_ble),
                positionCharacteristic(SENS_POSITION_CHAR_UUID_128, envPosition, POSITION_DATA_LEN, POSITION_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
                sunpanelCharacteristic(SENS_SUNPANEL_CHAR_UUID_128, envSunpanel, SUNPANEL_DATA_LEN, SUNPANEL_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE)          
        {  
        static bool serviceAdded = false; // To be sure to define just one time
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&positionCharacteristic, &sunpanelCharacteristic};                 
                                                                                                                                                         
        GattService   envService(SENS_SERVICE_UUID_128, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));               
                
        ble.gattServer().addService(envService);             

        isEnabledPositionNotify         = false;
        isEnabledSunpanelNotify         = false;            

        memset (envPosition, 0, POSITION_DATA_LEN);  
        memset (envSunpanel, 0, SUNPANEL_DATA_LEN);                                
                                                                         
        isBTLEConnected                 = DISCONNECTED;                                                                  
        serviceAdded                    = true;                                                                                                                                  
    }

// Tests Method
//----------------INIZIO  
/*

    void sendEnvTemperature (int16_t Temp, uint16_t TimeStamp) {
            STORE_LE_16(envTemperature,TimeStamp);
            STORE_LE_16(envTemperature+2,Temp);
            PRINTF("sendEnvTemperature!!  handle: %d\n\r",   envTemperatureCharacteristic.getValueAttribute().getHandle());
            memcpy (pastenvTemperature, envTemperature, TEMP_DATA_LEN);
            ble.gattServer().write(envTemperatureCharacteristic.getValueAttribute().getHandle(), envTemperature, TEMP_DATA_LEN, 0);         
    }               
        
    void updateEnvTemperature (int16_t Temp, uint16_t TimeStamp) {
                if (memcmp (&pastenvTemperature[2], &Temp, 2) != 0) {
                    sendEnvTemperature (Temp,  TimeStamp);              
                }
        }
                
    void sendEnvHumidity(uint16_t Hum, uint16_t TimeStamp) {        
            STORE_LE_16(envHumidity,TimeStamp);
            STORE_LE_16(envHumidity+2,Hum);
            memcpy (pastenvHumidity, envHumidity, HUM_DATA_LEN);                
            ble.gattServer().write(envHumidityCharacteristic.getValueAttribute().getHandle(), envHumidity, HUM_DATA_LEN, 0);            
            
        }
        
    void updateEnvHumidity(uint16_t Hum, uint16_t TimeStamp) {
                if (memcmp (&pastenvHumidity[2], &Hum, 2) != 0) {               
                    sendEnvHumidity(Hum, TimeStamp);
                }
    }       
                
    void sendEnvPressure(uint32_t Press, uint16_t TimeStamp) {
            STORE_LE_16(envPressure,TimeStamp);         
            STORE_LE_32(envPressure+2,Press);
            memcpy (pastenvPressure, envPressure, PRES_DATA_LEN);                           
            ble.gattServer().write(envPressureCharacteristic.getValueAttribute().getHandle(), envPressure, PRES_DATA_LEN, 0);           
        }
        
    void updateEnvPressure(uint32_t Press, uint16_t TimeStamp) {
                if (memcmp (&pastenvPressure[2], &Press, 2) != 0) {             
                    sendEnvPressure(Press, TimeStamp);
                }
    }       
    
    void sendEnvMagnetometer(AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor magOffset) {         
            STORE_LE_16(envMagn,TimeStamp);
            STORE_LE_16(envMagn+2,(Magn->AXIS_X - magOffset.magOffX));
            STORE_LE_16(envMagn+4,(Magn->AXIS_Y - magOffset.magOffY));
            STORE_LE_16(envMagn+6,(Magn->AXIS_Z - magOffset.magOffZ));          
            ble.gattServer().write(envMagnetometerCharacteristic.getValueAttribute().getHandle(), envMagn, MAG_DATA_LEN, 0);                
    }
        
    void updateEnvMagnetometer(AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor magOffset) {   
            if (isMagNotificationEn()) sendEnvMagnetometer(Magn, TimeStamp, magOffset);
    }               
        
    void sendEnvAccelerometer (AxesRaw_TypeDef *Acc, uint16_t TimeStamp) {          
            STORE_LE_16(envAcce,TimeStamp);
            STORE_LE_16(envAcce+2,Acc->AXIS_X);
            STORE_LE_16(envAcce+4,Acc->AXIS_Y);
            STORE_LE_16(envAcce+6,Acc->AXIS_Z);
            ble.gattServer().write(envAccelerometerCharacteristic.getValueAttribute().getHandle(), envAcce, ACC_DATA_LEN, 0);           
    }
        
    void updateEnvAccelerometer (AxesRaw_TypeDef *Acc, uint16_t TimeStamp) {            
            if (isAccNotificationEn()) sendEnvAccelerometer (Acc, TimeStamp);
    }                       

    void sendEnvGyroscope (AxesRaw_TypeDef *Gyro, uint16_t TimeStamp) {     
            STORE_LE_16(envGyro,TimeStamp);
            STORE_LE_16(envGyro+2,Gyro->AXIS_X);
            STORE_LE_16(envGyro+4,Gyro->AXIS_Y);
            STORE_LE_16(envGyro+6,Gyro->AXIS_Z);
            ble.gattServer().write(envGyroCharacteristic.getValueAttribute().getHandle(), envGyro, GYRO_DATA_LEN, 0);           
    }       
        
    void updateEnvGyroscope (AxesRaw_TypeDef *Gyro, uint16_t TimeStamp) {       
            if (isGyroNotificationEn()) sendEnvGyroscope (Gyro, TimeStamp);             
    }                       

    void sendEnvAccGyroMag (AxesRaw_TypeDef *Acc, AxesRaw_TypeDef *Gyro, AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor magOffset) {
            STORE_LE_16(envAccGyroMag,TimeStamp);
            STORE_LE_16(envAccGyroMag+2,Acc->AXIS_X);
            STORE_LE_16(envAccGyroMag+4,Acc->AXIS_Y);
            STORE_LE_16(envAccGyroMag+6,Acc->AXIS_Z);
            
            STORE_LE_16(envAccGyroMag+8,Gyro->AXIS_X);
            STORE_LE_16(envAccGyroMag+10,Gyro->AXIS_Y);
            STORE_LE_16(envAccGyroMag+12,Gyro->AXIS_Z);
            
            STORE_LE_16(envAccGyroMag+14,(Magn->AXIS_X  - magOffset.magOffX));
            STORE_LE_16(envAccGyroMag+16,(Magn->AXIS_Y  - magOffset.magOffY));
            STORE_LE_16(envAccGyroMag+18,(Magn->AXIS_Z  - magOffset.magOffZ));                      
            ble.gattServer().write(envAccGyroMagCharacteristic.getValueAttribute().getHandle(), envAccGyroMag, ACCGYROMAG_DATA_LEN, 0);         
    }
        
    void updateEnvAccGyroMag (AxesRaw_TypeDef *Acc, AxesRaw_TypeDef *Gyro, AxesRaw_TypeDef *Magn, uint16_t TimeStamp, osxMFX_calibFactor magOffset) {       
            if (isAccGyroMagNotificationEn())sendEnvAccGyroMag (Acc, Gyro, Magn, TimeStamp, magOffset);              
    } 
    
*/    
//----------------FINE  
                                  
        
    void enNotify (Gap::Handle_t handle) {
            if (isPositionHandle(handle)) { isEnabledPositionNotify = true; memset(envPosition,0,POSITION_DATA_LEN); return; }
            if (isSunpanelHandle(handle)) { isEnabledSunpanelNotify = true; memset(envSunpanel,0,SUNPANEL_DATA_LEN); return; }                     
    }

    void disNotify (Gap::Handle_t handle) {
            if (isPositionHandle(handle)) { isEnabledPositionNotify = false; memset(envPosition,0,POSITION_DATA_LEN); return; }        
            if (isSunpanelHandle(handle)) { isEnabledSunpanelNotify = false; memset(envSunpanel,0,SUNPANEL_DATA_LEN); return; }                   
    }
        
    bool isPositionNotificationEn (void) {
            return isEnabledPositionNotify;
    }
        
    bool isSunpanelNotificationEn (void) {
            return isEnabledSunpanelNotify;
    }
                 
    bool isPositionHandle (Gap::Handle_t handle) {
            if (handle == positionCharacteristic.getValueAttribute().getHandle()) return true;
            return false;
    }

    bool isSunpanelHandle (Gap::Handle_t handle) {
            if (handle == sunpanelCharacteristic.getValueAttribute().getHandle()) return true;
            return false;
    }
        
    void updateConnectionStatus(ConnectionStatus_t status) {   
            isEnabledPositionNotify = false;
            isEnabledSunpanelNotify = false;    
            
            memset (envPosition, 0, POSITION_DATA_LEN);  
            memset (envSunpanel, 0, SUNPANEL_DATA_LEN);                             
            isBTLEConnected = status; 
    }
                
// Variables Initialization     
private:

    BLEDevice            &ble;
    uint8_t              envPosition [POSITION_DATA_LEN];
    uint8_t              envSunpanel [SUNPANEL_DATA_LEN];        

    GattCharacteristic   positionCharacteristic;     
    GattCharacteristic   sunpanelCharacteristic;   
 
    ConnectionStatus_t   isBTLEConnected;

    bool                 isEnabledPositionNotify;
    bool                 isEnabledSunpanelNotify;
               
};

#endif /* #ifndef __CUSTOM_BLE_SENSORS_SERVICE_H__*/
