/******************************************************************************
  * @file    CustomSensorsService.h
  * @author  Fabio Brembilla
  * @version V1.0.0
  * @date    April, 2016
  * @brief   SunTracker Custom Service for BlueTooth (IDB0XA1 expansion board)
  *****************************************************************************
  * @attention
  *
  * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
*/

#include "BLE.h"
#include "UUID.h"

/* BLE Services: Primary + 3 Secondary (Char Desk) */
//const LongUUIDBytes_t SENS_SERVICE_UUID           = {0x00,0x00,0x00,0x00,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //Primary 
//const LongUUIDBytes_t SENS_DIFFERENCE_CHAR_UUID   = {0x00,0x00,0x00,0x01,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //Difference Light/Mems
//const LongUUIDBytes_t SENS_POSITION_CHAR_UUID     = {0x00,0x00,0x00,0x02,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //Motor Position
//const LongUUIDBytes_t SENS_SUNPANEL_CHAR_UUID     = {0x00,0x00,0x00,0x03,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B,0x0B}; //SunPanel Voltage
//const ShortUUIDBytes_t SENS_SERVICE_UUID            = 0xB000; //Primary 
//const ShortUUIDBytes_t SENS_STATUS_CHAR_UUID        = 0xB001; //Status
//const ShortUUIDBytes_t SENS_DIFFERENCE_CHAR_UUID    = 0xB002; //Difference Light/Mems Sensors
//const ShortUUIDBytes_t SENS_POSITION_CHAR_UUID      = 0xB003; //Motor Position
//const ShortUUIDBytes_t SENS_SUNPANEL_CHAR_UUID      = 0xB004; //SunPanel Voltage    

/* BLE Services: Primary + 4 Secondary (Char Desk) */
const UUID::LongUUIDBytes_t SENS_SERVICE_UUID           = {0x00,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Primary 
const UUID::LongUUIDBytes_t SENS_STATUS_CHAR_UUID       = {0x00,0x00,0x08,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Status                 0x00000800
const UUID::LongUUIDBytes_t SENS_DIFFERENCE_CHAR_UUID   = {0x00,0x00,0x10,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Difference Light/Mems  0x00001000
const UUID::LongUUIDBytes_t SENS_POSITION_CHAR_UUID     = {0x00,0x00,0x20,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //Motor Position         0x00002000
const UUID::LongUUIDBytes_t SENS_SUNPANEL_CHAR_UUID     = {0x00,0x00,0x40,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b}; //SunPanel Voltage       0x00004000
                      
//#define DIFFERENCE_DATA_LEN     2
//#define POSITION_DATA_LEN       2
//#define SUNPANEL_DATA_LEN       2
// 1 byte   0xFF     8 bit      int8
// 2 byte   0xFFFF   16 bit     int16

#define STATUS_DATA_LEN         2+2 // TimeStamp + Status
#define DIFFERENCE_DATA_LEN     2+2 // TimeStamp + Difference
#define POSITION_DATA_LEN       2+2 // TimeStamp + Position
#define SUNPANEL_DATA_LEN       2+2 // TimeStamp + SunPanel

/* Custom Sensors Service */
class CustomSensorService  {
public:
    CustomSensorService(BLEDevice &_ble) :
        ble(_ble),
                statusCharacteristic(SENS_STATUS_CHAR_UUID, envStatus, STATUS_DATA_LEN, STATUS_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
                differenceCharacteristic(SENS_DIFFERENCE_CHAR_UUID, envDifference, DIFFERENCE_DATA_LEN, DIFFERENCE_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
                positionCharacteristic(SENS_POSITION_CHAR_UUID, envPosition, POSITION_DATA_LEN, POSITION_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
                sunpanelCharacteristic(SENS_SUNPANEL_CHAR_UUID, envSunpanel, SUNPANEL_DATA_LEN, SUNPANEL_DATA_LEN,
                                       GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)         
        {  
        static bool serviceAdded = false; // To be sure to define just one time
        if (serviceAdded) {
            return;
        }

        GattCharacteristic *charTable[] = {&statusCharacteristic, &differenceCharacteristic, &positionCharacteristic, &sunpanelCharacteristic};                                                                                                                          
        GattService   envService(SENS_SERVICE_UUID, charTable, sizeof(charTable) / sizeof(GattCharacteristic *));
        ble.gattServer().addService(envService);             

        isEnabledStatusNotify           = false;
        isEnabledDifferenceNotify       = false;
        isEnabledPositionNotify         = false;
        isEnabledSunpanelNotify         = false;
        memset (envStatus,     0, STATUS_DATA_LEN);          
        memset (envDifference, 0, DIFFERENCE_DATA_LEN);  
        memset (envPosition,   0, POSITION_DATA_LEN);  
        memset (envSunpanel,   0, SUNPANEL_DATA_LEN);                                                       
        isBTLEConnected                 = DISCONNECTED;                                                                  
        serviceAdded                    = true;                                                                                                                                  
    }

// Tests Method

    void sendEnvStatus (uint16_t Sta, uint16_t TimeStamp) {
            STORE_LE_16(envStatus,TimeStamp);
            STORE_LE_16(envStatus+2,Sta);
            PRINTF("sendEnvStatus handle: %d\n\r", statusCharacteristic.getValueAttribute().getHandle());
            memcpy (pastenvStatus, envStatus, STATUS_DATA_LEN);
            ble.gattServer().write(statusCharacteristic.getValueAttribute().getHandle(), envStatus, STATUS_DATA_LEN, 0);         
            printf(" (Update)");
    } 

    void updateEnvStatus (uint16_t Sta, uint16_t TimeStamp) {
            if (ble.getGapState().connected && isEnabledStatusNotify ) { 
                if (memcmp (&pastenvStatus[2], &Sta, 2) != 0) {
                    sendEnvStatus (Sta, TimeStamp);
                }
            }
    }

//--

    void sendEnvDifference (uint16_t Dif, uint16_t TimeStamp) {
            STORE_LE_16(envDifference,TimeStamp);
            STORE_LE_16(envDifference+2,Dif);            
            PRINTF("sendEnvDifference handle: %d\n\r", differenceCharacteristic.getValueAttribute().getHandle());
            memcpy (pastenvDifference, envDifference, DIFFERENCE_DATA_LEN);
            ble.gattServer().write(differenceCharacteristic.getValueAttribute().getHandle(), envDifference, DIFFERENCE_DATA_LEN, 0);         
            printf(" (Update)");
    } 

    void updateEnvDifference (uint16_t Dif, uint16_t TimeStamp) {
            if (ble.getGapState().connected && isEnabledDifferenceNotify ) { 
                if (memcmp (&pastenvDifference[2], &Dif, 2) != 0) {
                    sendEnvDifference (Dif, TimeStamp);
                }
            }
    }
        
//--

    void sendEnvPosition (uint16_t Pos, uint16_t TimeStamp) {
            STORE_LE_16(envPosition,TimeStamp);
            STORE_LE_16(envPosition+2,Pos);             
            PRINTF("sendEnvPosition handle: %d\n\r", positionCharacteristic.getValueAttribute().getHandle());
            memcpy (pastenvPosition, envPosition, POSITION_DATA_LEN);
            ble.gattServer().write(positionCharacteristic.getValueAttribute().getHandle(), envPosition, POSITION_DATA_LEN, 0);         
            printf(" (Update)");
    } 

    void updateEnvPosition (uint16_t Pos, uint16_t TimeStamp) {
            if (ble.getGapState().connected && isEnabledPositionNotify ) { 
                if (memcmp (&pastenvPosition[2], &Pos, 2) != 0) {
                    sendEnvPosition (Pos, TimeStamp);
                }
            }
    }

//--

    void sendEnvSunpanel (uint16_t Sun, uint16_t TimeStamp) {
            STORE_LE_16(envSunpanel,TimeStamp);
            STORE_LE_16(envSunpanel+2,Sun);              
            PRINTF("sendEnvSunpanel handle: %d\n\r", sunpanelCharacteristic.getValueAttribute().getHandle());
            memcpy (pastenvSunpanel, envSunpanel, SUNPANEL_DATA_LEN);
            ble.gattServer().write(sunpanelCharacteristic.getValueAttribute().getHandle(), envSunpanel, SUNPANEL_DATA_LEN, 0);         
            printf(" (Update)");
    } 

    void updateEnvSunpanel (uint16_t Sun, uint16_t TimeStamp) {
            if (ble.getGapState().connected && isEnabledSunpanelNotify ) { 
                if (memcmp (&pastenvSunpanel[2], &Sun, 2) != 0) {
                    sendEnvSunpanel (Sun, TimeStamp);            
                }
            }
    }

//---- 
                                        
    void enNotify (Gap::Handle_t handle) {
            printf("\n\r\n\renNotify (handle %d)", handle);
            if (isStatusHandle(handle)) { isEnabledStatusNotify = true; memset(envStatus,0,STATUS_DATA_LEN); return; }
            if (isDifferenceHandle(handle)) { isEnabledDifferenceNotify = true; memset(envDifference,0,DIFFERENCE_DATA_LEN); return; }
            if (isPositionHandle(handle)) { isEnabledPositionNotify = true; memset(envPosition,0,POSITION_DATA_LEN); return; }
            if (isSunpanelHandle(handle)) { isEnabledSunpanelNotify = true; memset(envSunpanel,0,SUNPANEL_DATA_LEN); return; }                     
    }

    void disNotify (Gap::Handle_t handle) {
            printf("\n\r\n\rdisNotify (handle %d)", handle);
            if (isStatusHandle(handle)) { isEnabledStatusNotify = false; memset(envStatus,0,STATUS_DATA_LEN); return; }
            if (isDifferenceHandle(handle)) { isEnabledDifferenceNotify = false; memset(envDifference,0,DIFFERENCE_DATA_LEN); return; }
            if (isPositionHandle(handle)) { isEnabledPositionNotify = false; memset(envPosition,0,POSITION_DATA_LEN); return; }        
            if (isSunpanelHandle(handle)) { isEnabledSunpanelNotify = false; memset(envSunpanel,0,SUNPANEL_DATA_LEN); return; }                   
    }
    
//----    
      
    bool isStatusNotificationEn (void) {
            return isEnabledStatusNotify;
    }  
        
    bool isDifferenceNotificationEn (void) {
            return isEnabledDifferenceNotify;
    }

    bool isPositionNotificationEn (void) {
            return isEnabledPositionNotify;
    }
        
    bool isSunpanelNotificationEn (void) {
            return isEnabledSunpanelNotify;
    }

//----
           
    bool isStatusHandle (Gap::Handle_t handle) {
            if (handle == statusCharacteristic.getValueAttribute().getHandle()) return true;
            return false;
    }

    bool isDifferenceHandle (Gap::Handle_t handle) {
            if (handle == differenceCharacteristic.getValueAttribute().getHandle()) return true;
            return false;
    }

    bool isPositionHandle (Gap::Handle_t handle) {
            if (handle == positionCharacteristic.getValueAttribute().getHandle()) return true;
            return false;
    }

    bool isSunpanelHandle (Gap::Handle_t handle) {
            if (handle == sunpanelCharacteristic.getValueAttribute().getHandle()) return true;
            return false;
    }
        
    void updateConnectionStatus(ConnectionStatus_t status) {
            isEnabledStatusNotify = false; 
            isEnabledDifferenceNotify = false;
            isEnabledPositionNotify = false;
            isEnabledSunpanelNotify = false;
            memset (envStatus, 0, STATUS_DATA_LEN);  
            memset (envDifference, 0, DIFFERENCE_DATA_LEN);  
            memset (envPosition, 0, POSITION_DATA_LEN);  
            memset (envSunpanel, 0, SUNPANEL_DATA_LEN);                             
            isBTLEConnected = status; 
    }
                
// Variables Initialization     
private:

    BLEDevice            &ble;
    uint8_t              envStatus [STATUS_DATA_LEN];
    uint8_t              envDifference [DIFFERENCE_DATA_LEN];
    uint8_t              envPosition [POSITION_DATA_LEN];
    uint8_t              envSunpanel [SUNPANEL_DATA_LEN];    
    uint8_t              pastenvStatus [STATUS_DATA_LEN];   
    uint8_t              pastenvDifference [POSITION_DATA_LEN];
    uint8_t              pastenvPosition [POSITION_DATA_LEN];
    uint8_t              pastenvSunpanel [SUNPANEL_DATA_LEN];
    
    GattCharacteristic   statusCharacteristic;
    GattCharacteristic   differenceCharacteristic;   
    GattCharacteristic   positionCharacteristic;     
    GattCharacteristic   sunpanelCharacteristic;   
    
    ConnectionStatus_t   isBTLEConnected;

    bool                 isEnabledStatusNotify;
    bool                 isEnabledDifferenceNotify;
    bool                 isEnabledPositionNotify;
    bool                 isEnabledSunpanelNotify;
               
};
