/**
 ******************************************************************************
 * @file    main.cpp
 * @author  CLab
 * @version V1.0.0
 * @date    2-December-2016
 * @brief   Simple Example application for using the X_NUCLEO_IKS01A2 
 *          MEMS Inertial & Environmental Sensor Nucleo expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 

/* Includes */
#include "mbed.h"
#include "x_nucleo_iks01a2.h"

/* Instantiate the expansion board */
static X_NUCLEO_IKS01A2 *mems_expansion_board = X_NUCLEO_IKS01A2::Instance(D14, D15, D4, D5);

/* Retrieve the composing elements of the expansion board */
static LSM6DSLSensor *acc_gyro = mems_expansion_board->acc_gyro;

InterruptIn mybutton(USER_BUTTON);
DigitalOut myled(LED1);

volatile int mems_event = 0;
volatile int step_count_reset_request = 0;
uint32_t previous_tick = 0;
uint32_t current_tick = 0;
uint16_t step_count = 0;

void pressed_cb();
void INT1_cb();

/* Simple main function */
int main() {
  /* Attach callback to User button press */
  mybutton.fall(&pressed_cb);
  /* Attach callback to LSM6DSL INT1 */
  acc_gyro->AttachINT1IRQ(&INT1_cb);
  
  /* Enable LSM6DSL accelerometer */
  acc_gyro->Enable_X();
  /* Enable Pedometer. */
  acc_gyro->Enable_Pedometer();
  
  previous_tick = clock();
  
  printf("\r\n--- Starting new run ---\r\n");
 
  while(1) {
    if (mems_event) {
      mems_event = 0;
      LSM6DSL_Event_Status_t status;
      acc_gyro->Get_Event_Status(&status);
      if (status.StepStatus) {
        /* New step detected, so print the step counter */
        acc_gyro->Get_Step_Counter(&step_count);
        printf("Step counter: %d\r\n", step_count);
      
        /* Led blinking. */
        myled = 1;
        wait(0.1);
        myled = 0;
      }
    }

    if(step_count_reset_request) {
      step_count_reset_request = 0;
      acc_gyro->Reset_Step_Counter();
    }

    /* Print the step counter in any case every 3s */
    current_tick = clock();
    if(((current_tick - previous_tick)/CLOCKS_PER_SEC) >= 3) {
      acc_gyro->Get_Step_Counter(&step_count);
      printf("Step counter: %d\r\n", step_count);
      previous_tick = clock();
    }
  }
}

void pressed_cb() {
  step_count_reset_request = 1;
}

void INT1_cb() {
  mems_event = 1;
}
