/**
 *******************************************************************************
 * @file    CustomService.h
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    October 31st, 2017
 * @brief   mbed test application for the STMicroelectronics X-NUCLEO-IHM01A1
 *          Motor Control Expansion Board and the X-NUCLEO-IDB05A1 Bluetooth
 *          Low energy Expansion Board.
 *******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2018 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 */
 

#ifndef __BLE_CUSTOM_SERVICE_H__
#define __BLE_CUSTOM_SERVICE_H__

#include "ble_utils.h"

#define STATE_DATA_LENGTH   (sizeof(uint16_t) + sizeof(uint8_t))
#define COMMAND_DATA_LENGTH (sizeof(uint8_t) + sizeof(uint32_t))
#define MAX_DATA_LENGTH     (COMMAND_DATA_LENGTH)

const UUID::LongUUIDBytes_t CUSTOM_SERVICE_UUID                      = {0x00,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b};
const UUID::LongUUIDBytes_t CUSTOM_STEPPER_MOTOR_CHARACTERISTIC_UUID = {0x00,0x00,0x20,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b};

class CustomService {
public:

    /* Motor states. */
    typedef enum
    {
        MOTOR_INACTIVE = 0,  // Motor not running.
        MOTOR_RUNNING        // Motor running.
    } motor_state_t;

    /* Motor commands. */
    typedef enum
    {
        MOTOR_STOP_RUNNING_WITHOUT_TORQUE = 0,  // Stops running with HiZ.
        MOTOR_STOP_RUNNING_WITH_TORQUE,         // Stops running with torque applied.
        MOTOR_RUN_FORWARD,                      // Runs forward indefinitely.
        MOTOR_RUN_BACKWARD,                     // Runs backward indefinitely.
        MOTOR_MOVE_STEPS_FORWARD,               // Moves steps forward.
        MOTOR_MOVE_STEPS_BACKWARD               // Moves steps backward.
    } motor_command_t;

    CustomService(BLEDevice &_ble) :
        ble(_ble),
        state_command(CUSTOM_STEPPER_MOTOR_CHARACTERISTIC_UUID, packed_state_command, MAX_DATA_LENGTH, MAX_DATA_LENGTH,
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE)
    {
        GattCharacteristic *char_table[] = {&state_command};
        GattService motor_service(CUSTOM_SERVICE_UUID, char_table, sizeof(char_table) / sizeof(GattCharacteristic *));
        ble.addService(motor_service);
        memset (packed_state_command, 0, MAX_DATA_LENGTH);
    }

    void send_state(uint16_t time_stamp, uint8_t current_state) {
        memset (packed_state_command, 0, MAX_DATA_LENGTH);
        STORE_LE_16(packed_state_command, time_stamp);
        packed_state_command[2] = current_state;
        ble.gattServer().write(state_command.getValueAttribute().getHandle(), (uint8_t *) &packed_state_command, STATE_DATA_LENGTH, 0);
    }

    GattAttribute::Handle_t getValueHandle() const
    {
        return state_command.getValueAttribute().getHandle();
    }

private:
    BLEDevice &ble;
    GattCharacteristic state_command;
    uint8_t packed_state_command[MAX_DATA_LENGTH];
};

#endif /* #ifndef __BLE_CUSTOM_SERVICE_H__ */
