/**
 *******************************************************************************
 * @file    CustomService.h
 * @author  Davide Aliprandi, STMicroelectronics
 * @version V1.0.0
 * @date    March 31st, 2018
 * @brief   mbed test application for the STMicroelectronics X-NUCLEO-IDB05A1
 *          Bluetooth Low energy Expansion Board.
 *******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2018 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright
 *      notice, this list of conditions and the following disclaimer in the
 *      documentation and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its
 *      contributors may be used to endorse or promote products derived from
 *      this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS 
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 *******************************************************************************
 */
 

#ifndef __BLE_CUSTOM_SERVICE_H__
#define __BLE_CUSTOM_SERVICE_H__


/* Includes ------------------------------------------------------------------*/

#include "ble_utils.h"


/* Definitions ---------------------------------------------------------------*/

#define TIMESTAMP_LENGTH        (sizeof(uint16_t))
#define SWITCH_LENGTH           (sizeof(uint8_t))
#define PRESSURE_LENGTH         (sizeof(uint32_t))
#define HUMIDITY_LENGTH         (sizeof(uint16_t))
#define TEMPERATURE_LENGTH      (sizeof(uint16_t))
#define ACCELEROMETER_LENGTH    (sizeof(uint16_t))
#define GYROSCOPE_LENGTH        (sizeof(uint16_t))
#define MAGNETOMETER_LENGTH     (sizeof(uint16_t))

#define SWITCH_DATA_INDEX       (TIMESTAMP_LENGTH)
#define SWITCH_DATA_LENGTH      (TIMESTAMP_LENGTH + SWITCH_LENGTH)
#define ENV_SENSORS_DATA_LENGTH (TIMESTAMP_LENGTH + PRESSURE_LENGTH + HUMIDITY_LENGTH + TEMPERATURE_LENGTH)
#define INE_SENSORS_DATA_LENGTH (TIMESTAMP_LENGTH + (3 * ACCELEROMETER_LENGTH) + (3 * GYROSCOPE_LENGTH) + (3 * MAGNETOMETER_LENGTH))


/* Types ---------------------------------------------------------------------*/

/* Axes. */
typedef struct
{
    int32_t x;
    int32_t y;
    int32_t z;
} axes_t;


/* Constants -----------------------------------------------------------------*/

const UUID::LongUUIDBytes_t CUSTOM_SERVICE_UUID                    = {0x00,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0x9a,0xb4,0x00,0x02,0xa5,0xd5,0xc5,0x1b};
const UUID::LongUUIDBytes_t CUSTOM_SWITCH_CHARACTERISTIC_UUID      = {0x20,0x00,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b};
const UUID::LongUUIDBytes_t CUSTOM_ENV_SENSORS_CHARACTERISTIC_UUID = {0x00,0x1c,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b};
const UUID::LongUUIDBytes_t CUSTOM_INE_SENSORS_CHARACTERISTIC_UUID = {0x00,0xe0,0x00,0x00,0x00,0x01,0x11,0xe1,0xac,0x36,0x00,0x02,0xa5,0xd5,0xc5,0x1b};


/* Classes -------------------------------------------------------------------*/

class CustomService {
public:

    /* Switch states. */
    typedef enum
    {
        SWITCH_OFF = 0,
        SWITCH_ON
    } switch_state_t;

    CustomService(BLEDevice &_ble) :
        ble(_ble),
        state_command(CUSTOM_SWITCH_CHARACTERISTIC_UUID, packed_state_command, SWITCH_DATA_LENGTH, SWITCH_DATA_LENGTH,
            /*GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | */GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_WRITE | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        env_sensors(CUSTOM_ENV_SENSORS_CHARACTERISTIC_UUID, packed_env_sensors, ENV_SENSORS_DATA_LENGTH, ENV_SENSORS_DATA_LENGTH,
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY),
        ine_sensors(CUSTOM_INE_SENSORS_CHARACTERISTIC_UUID, packed_ine_sensors, INE_SENSORS_DATA_LENGTH, INE_SENSORS_DATA_LENGTH,
            GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_READ | GattCharacteristic::BLE_GATT_CHAR_PROPERTIES_NOTIFY)
    {
        GattCharacteristic *char_table[] = {&state_command, &env_sensors, &ine_sensors};
        GattService iot_service(CUSTOM_SERVICE_UUID, char_table, sizeof(char_table) / sizeof(GattCharacteristic *));
        ble.addService(iot_service);
        memset (packed_state_command, 0, SWITCH_DATA_LENGTH);
        memset (packed_env_sensors, 0, ENV_SENSORS_DATA_LENGTH);
        memset (packed_ine_sensors, 0, INE_SENSORS_DATA_LENGTH);
    }

    void send_state(uint16_t time_stamp, uint8_t current_state) {
        memset (packed_state_command, 0, SWITCH_DATA_LENGTH);
        STORE_LE_16(packed_state_command, time_stamp);
        packed_state_command[SWITCH_DATA_INDEX] = current_state;
        ble.gattServer().write(get_state_command_handle(), (uint8_t *) &packed_state_command, SWITCH_DATA_LENGTH, 0);
    }

    void send_env_sensors(
        uint16_t time_stamp,
        float pressure, float humidity, float temperature
    ) {
        memset (packed_env_sensors, 0, ENV_SENSORS_DATA_LENGTH);
        int p = 0;
        STORE_LE_16(&packed_env_sensors[p], time_stamp);
        p += TIMESTAMP_LENGTH;
        STORE_LE_32(&packed_env_sensors[p], (uint32_t) (pressure * 10));
        p += PRESSURE_LENGTH;
        STORE_LE_16(&packed_env_sensors[p], (uint16_t) (humidity * 10));
        p += HUMIDITY_LENGTH;
        STORE_LE_16(&packed_env_sensors[p], (uint16_t) (temperature * 10));
        ble.gattServer().write(get_env_sensors_handle(), (uint8_t *) &packed_env_sensors, ENV_SENSORS_DATA_LENGTH, 0);
    }

    void send_ine_sensors(
        uint16_t time_stamp,
        axes_t *accelerometer, axes_t *gyroscope, axes_t *magnetometer
    ) {
        memset (packed_ine_sensors, 0, INE_SENSORS_DATA_LENGTH);
        int p = 0;
        STORE_LE_16(&packed_ine_sensors[p], time_stamp);
        p += TIMESTAMP_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) accelerometer->x);
        p += ACCELEROMETER_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) accelerometer->y);
        p += ACCELEROMETER_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) accelerometer->z);
        p += ACCELEROMETER_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) gyroscope->x);
        p += GYROSCOPE_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) gyroscope->y);
        p += GYROSCOPE_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) gyroscope->z);
        p += GYROSCOPE_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) magnetometer->x);
        p += MAGNETOMETER_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) magnetometer->y);
        p += MAGNETOMETER_LENGTH;
        STORE_LE_16(&packed_ine_sensors[p], (int16_t) magnetometer->z);
        ble.gattServer().write(get_ine_sensors_handle(), (uint8_t *) &packed_ine_sensors, INE_SENSORS_DATA_LENGTH, 0);
    }

    GattAttribute::Handle_t get_state_command_handle() const
    {
        return state_command.getValueAttribute().getHandle();
    }

    GattAttribute::Handle_t get_env_sensors_handle() const
    {
        return env_sensors.getValueAttribute().getHandle();
    }

    GattAttribute::Handle_t get_ine_sensors_handle() const
    {
        return ine_sensors.getValueAttribute().getHandle();
    }

private:
    BLEDevice &ble;
    GattCharacteristic state_command;
    GattCharacteristic env_sensors;
    GattCharacteristic ine_sensors;
    uint8_t packed_state_command[SWITCH_DATA_LENGTH];
    uint8_t packed_env_sensors[ENV_SENSORS_DATA_LENGTH];
    uint8_t packed_ine_sensors[INE_SENSORS_DATA_LENGTH];
};

#endif /* #ifndef __BLE_CUSTOM_SERVICE_H__ */
