/**
  ******************************************************************************
  * @file       SampleSynch_writeAndChangeAll.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo write an ndef message different records, when the user press the buttun
  *             read the tag, change some data and write it again
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "NDefLib/NDefNfcTag.h"

#include "NDefLib/RecordType/RecordAAR.h"
#include "NDefLib/RecordType/RecordSMS.h"
#include "NDefLib/RecordType/RecordGeo.h"
#include "NDefLib/RecordType/RecordURI.h"
#include "NDefLib/RecordType/RecordMail.h"
#include "NDefLib/RecordType/RecordText.h"
#include "NDefLib/RecordType/RecordMimeType.h"
#include "NDefLib/RecordType/RecordVCard.h"
#include "NDefLib/RecordType/RecordWifiConf.h"

#include "XNucleoNFC01A1.h"

/**
 * Shift the led status between the 3 leds.
 */
static void shift_led(DigitalOut &led1,DigitalOut &led2,DigitalOut &led3){
    const uint8_t prevLed1=led1;
    const uint8_t prevLed2=led2;
    const uint8_t prevLed3=led3;
    led1=prevLed3;
    led2=prevLed1;
    led3=prevLed2;
}

/**
 * Create a message that contains all the possible records, and write it in the tag.
 * @param tag Nfc tag where write the message.
 */
static void write_nfc_tag(NDefLib::NDefNfcTag &tag){

    bool writeStatus=false;
    bool closeStatus=false;
    if(tag.open_session()){
        printf("Open session\r\n");
        NDefLib::Message msg;

        NDefLib::RecordAAR rAAR("com.st.BlueMS");
        msg.add_record(&rAAR);

        NDefLib::RecordSMS rSMS("123456789","st.com.BlueMS");
        msg.add_record(&rSMS);

        NDefLib::RecordGeo rGeo(123.123,-456.789);
        msg.add_record(&rGeo);

        NDefLib::RecordURI rUri(NDefLib::RecordURI::HTTP_WWW,"http://www.st.com");
        msg.add_record(&rUri);

        NDefLib::RecordMail rMail("mail@st.com","ciao","da nfc tag");
        msg.add_record(&rMail);

        NDefLib::RecordMimeType rText1("text/plain",(const uint8_t*)"ciao",4);
        msg.add_record(&rText1);

        NDefLib::RecordText rText3(NDefLib::RecordText::UTF8,"it","ciao");
        msg.add_record(&rText3);

        NDefLib::RecordWifiConf rWifi("OpenNetworkd");
        msg.add_record(&rWifi);

        NDefLib::RecordVCard::VCardInfo_t cardInfo;
        cardInfo[NDefLib::RecordVCard::FORMATTED_NAME]="prova prova1";
        cardInfo[NDefLib::RecordVCard::ADDRESS_HOME]=";;1 Main St.;Springfield;IL;12345;USA";
        cardInfo[NDefLib::RecordVCard::ADDRESS_WORK]=";;2 Main St.;Springfield;IL;12345;USA";
        cardInfo[NDefLib::RecordVCard::EMAIL_WORK]="workmail@st.com";
        cardInfo[NDefLib::RecordVCard::EMAIL_HOME]="homemail@st.com";
        cardInfo[NDefLib::RecordVCard::GEO]="39.95;-75.1667";
        cardInfo[NDefLib::RecordVCard::IMPP]="aim:johndoe@aol.com";
        cardInfo[NDefLib::RecordVCard::NAME]="prova2;prova3";
        cardInfo[NDefLib::RecordVCard::NICKNAME]="test";
        cardInfo[NDefLib::RecordVCard::NOTE]="A good test";
        cardInfo[NDefLib::RecordVCard::ORGANIZATION]="STM";
        cardInfo[NDefLib::RecordVCard::TEL_HOME]="123";
        cardInfo[NDefLib::RecordVCard::TEL_MOBILE]="456";
        cardInfo[NDefLib::RecordVCard::TEL_WORK]="789";
        cardInfo[NDefLib::RecordVCard::TITLE]="King";
        cardInfo[NDefLib::RecordVCard::URL]="www.st.com";
        cardInfo[NDefLib::RecordVCard::PHOTO_URI]="http://www.st.com/st-web-ui/static/active/en/fragment/multimedia/image/picture/customer_focus.jpg";
        NDefLib::RecordVCard rVCard(cardInfo);
        msg.add_record(&rVCard);

        writeStatus = tag.write(msg);

        closeStatus = tag.close_session();

    } else {
        printf("Error open Session\n\r");
    }

    if (writeStatus) {
        printf("Write Done\n\r");
    } else {
        printf("Write Fail\n\r");
    }

    if (closeStatus) {
        printf("Close Done\n\r");
    } else {
        printf("Close Fail\n\r");
    }
}

/**
 * Print the record content.
 * @param r Record to print.
 */
static void print_record(NDefLib::Record *const r){
    using namespace NDefLib;
    switch(r->get_type()){
        case Record::TYPE_TEXT: {
            const RecordText *const temp = ( RecordText* )r;
            printf("Read Text: %s\r\n",temp->get_text().c_str());
            break; }
        case Record::TYPE_AAR:{
            const RecordAAR *const temp = ( RecordAAR* )r;
            printf("Read ARR: %s\r\n",temp->get_package().c_str());
            break; }
        case Record::TYPE_MIME:{
            const RecordMimeType *const temp = ( RecordMimeType* )r;
            printf("Read mimeType: %s\r\n",temp->get_mime_type().c_str());
            printf("Read mimeData: %s\r\n",
                    std::string((const char*)temp->get_mime_data(),
                            temp->get_mime_data_lenght()).c_str());
            break;}
        case Record::TYPE_URI:{
            RecordURI *const temp = (RecordURI*)r;
            printf("Read uriId: %d\r\n",temp->get_uri_id());
            printf("Read uriType: %s\r\n",temp->get_uri_type().c_str());
            printf("Read uriContent: %s\r\n",temp->get_content().c_str());
            break;}
        case Record::TYPE_URI_MAIL:{
            const RecordMail*const temp = (RecordMail*)r;
            printf("Read Dest: %s\r\n",temp->get_to_address().c_str());
            printf("Read Subject: %s\r\n",temp->get_subject().c_str());
            printf("Read Body: %s\r\n",temp->get_body().c_str());
            break;}
        case Record::TYPE_URI_SMS:{
            const RecordSMS*const temp = (RecordSMS*)r;
            printf("Read number: %s\r\n",temp->get_number().c_str());
            printf("Read message: %s\r\n",temp->get_messagge().c_str());
            break;}
        case Record::TYPE_URI_GEOLOCATION:{
            const RecordGeo*const temp = (RecordGeo*)r;
            printf("Read lat: %f\r\n",temp->get_latitude());
            printf("Read long: %f\r\n",temp->get_longitude());
            break;}
        case Record::TYPE_MIME_VCARD:{
            const RecordVCard *const temp = (RecordVCard*)r;
            printf("Read Name: %s\r\n",(*temp)[RecordVCard::NAME].c_str());
            printf("Read Mail: %s\r\n",(*temp)[RecordVCard::EMAIL_WORK].c_str());
            printf("Read ORG: %s\r\n",(*temp)[RecordVCard::ORGANIZATION].c_str());
            break;}
        case Record::TYPE_WIFI_CONF:{
            const RecordWifiConf *const temp = (RecordWifiConf*)r;
            printf("Nework Name: %s\r\n",temp->get_network_ssid().c_str());
            printf("Nework Key: %s\r\n",temp->get_network_key().c_str());
            printf("Nework Auth: %X\r\n",temp->get_auth_type());
            printf("Nework Enc: %X\r\n",temp->get_encryption());
            break;}
        case Record::TYPE_UNKNOWN:{
            printf("Unknown record\r\n");
            break;}
    }//switch
}

/**
 * Change the record content.
 * @param r Record to change.
 */
static void change_record(NDefLib::Record const* r){
    using namespace NDefLib;
    switch(r->get_type()){
        case Record::TYPE_TEXT: {
            RecordText *temp = (RecordText*)r;
            temp->set_text("Hello");
            break; }
        case Record::TYPE_AAR:{
            RecordAAR *temp = (RecordAAR*)r;
            temp->set_package("set Package Ok");
            break; }
        case Record::TYPE_MIME:{
            RecordMimeType *temp = (RecordMimeType*)r;
            temp->copy_mime_data((const uint8_t *)"String2",sizeof("String2"));
            break;}
        case Record::TYPE_URI:{
            RecordURI *temp = (RecordURI*)r;
            temp->set_content("mbed.com");
            break;}
        case Record::TYPE_URI_MAIL:{
            RecordMail *temp = (RecordMail*)r;
            temp->set_to_address("newMail@st.com");
            temp->set_subject("tag change");
            temp->set_body("read/change Works!");
            break;}
        case Record::TYPE_URI_SMS:{
            RecordSMS *temp = (RecordSMS*)r;
            temp->set_message("Message Change");
            temp->set_number("0987654321");
            break;}
        case Record::TYPE_URI_GEOLOCATION:{
            RecordGeo *temp = (RecordGeo*)r;
            temp->set_latitude(-temp->get_latitude());
            temp->set_longitude(-temp->get_longitude());
            break;}
        case Record::TYPE_MIME_VCARD:{
            RecordVCard *temp = (RecordVCard*)r;
            (*temp)[RecordVCard::NAME]="name change";
            (*temp)[RecordVCard::NICKNAME]="nic change";
            break;}
        case Record::TYPE_WIFI_CONF:{
            RecordWifiConf * temp = (RecordWifiConf*)r;
            temp->set_network_ssid("hackMe");
            temp->set_network_key("qwerty");
            temp->set_auth_type(RecordWifiConf::AUTH_WPA2_PSK);
            temp->set_encryption_type(RecordWifiConf::ENC_TYPE_AES_TKIP);
            break;}
        case Record::TYPE_UNKNOWN:{
            printf("Unknown record\r\n");
            break;}
    }//switch
}

/**
 * Read the nfc message and print the content on the serial console
 * @param tag Nfc tag where read the content
 */
static void read_and_print_nfc_tag(NDefLib::NDefNfcTag &tag){
    using namespace NDefLib;

    if(tag.open_session()){
        printf("Open Session\r\n");
        NDefLib::Message readMsg;

        tag.read(&readMsg);
        printf("Message Read\r\n");
        
        if(readMsg.get_N_records()==0){
            printf("Error Read\r\n");
        }else{
            for(uint32_t i=0;i<readMsg.get_N_records();i++){
                Record *r = readMsg[i];
                print_record(r);
                delete r;
            }//for
        }//if-else

        tag.close_session();
        printf("Close session\r\n");
    }else{
        printf("Error open read Session\n\r");
    }
}

/**
 * Read a nfc message, change the content of each record and write the new message.
 * @param tag Tag where read and write the nfc message.
 */
static void read_and_change_nfc_tag(NDefLib::NDefNfcTag &tag){
    using NDefLib::Record;
    using NDefLib::Message;

    if(tag.open_session()){
        printf("Open Session\r\n");
        Message readMsg;

        tag.read(&readMsg);
        printf("Read Message\r\n");
        
        if(readMsg.get_N_records()==0){
            printf("Error Read\r\n");
        }else{
            for(uint32_t i=0;i<readMsg.get_N_records();i++){
                Record *r = readMsg[i];
                change_record(r);
            }//for
            tag.write(readMsg);
            printf("Message Wrote\r\n");
        }//if-else

        tag.close_session();
        printf("Close Session\r\n");
    }else{
        printf("Error open SessionChange\n\r");
    }
}

static volatile bool buttonPress=false; /// true when the user press the message

/**
 * Call back called when the user press the button
 */
static void set_button_press(){
    buttonPress=true;
}//if buttonPress

/**
 * Write a message and when the user press the button it read the message, change it and update it.
 */
void sampleSync_writeAndChangeAll() {
    
    //instance the board with the default paramiters
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel);
    //retrieve the NdefLib interface
    NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();
    
    //switch on the first led    
    nfcNucleo->get_led1()=1;
    nfcNucleo->get_led2()=0;
    nfcNucleo->get_led3()=0;
    
    printf("Init Done\r\n");

    //write the message
    write_nfc_tag(tag);
    //read the message and write it on console
    read_and_print_nfc_tag(tag);

    //enable the button
    #if defined(TARGET_STM)
        InterruptIn userButton(USER_BUTTON);
    #else
        InterruptIn userButton(SW2);
    #endif
    userButton.fall(set_button_press);

    //each second change the led status and see if the user press the button
    while(true) {               
        if(buttonPress){
            //update the message content
            read_and_change_nfc_tag(tag);
            //write the new message on console
            read_and_print_nfc_tag(tag);
            shift_led(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
            buttonPress=false;
        }//if
    }//while
}