/**
  ******************************************************************************
  * @file       SampleSync_lockWriteTag.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo define a lock/unlock the tag content when the user press
  *  the user button
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"

#include "XNucleoNFC01A1.h"

static volatile bool buttonPress=false;

static void setButtonPress(){
    printf("button\r\n");
    buttonPress=true;
}//if buttonPress

/**
 * return the led that tell if the tag is read only
 * @param nfcNucleo board where the lead lives
 * @return let to switch on if the tag is read only
 */
static inline DigitalOut& getReadOnlyLed(XNucleoNFC01A1 *nfcNucleo){
	return nfcNucleo->get_led2();
}

/**
 * return the led that tell if the tag is write only
 * @param nfcNucleo board where the lead lives
 * @return let to switch on if the tag is write only
 */
static inline DigitalOut& getWriteableLed(XNucleoNFC01A1 *nfcNucleo){
	return nfcNucleo->get_led3();
}

/**
 * set the tag as read only and switch the led on
 * @param nfcNucleo board where the lead lives
 * @param tag helper class to manage the nfc tag
 * @return true if the change has effect
 */
bool enableReadOnly( XNucleoNFC01A1 *nfcNucleo){
	if(nfcNucleo->get_M24SR().enable_read_only(M24SR::DEFAULT_PASSWORD) == M24SR::M24SR_SUCCESS){
		printf("Enable Read Only\r\n");
		getReadOnlyLed(nfcNucleo)=true;
		getWriteableLed(nfcNucleo)=false;
		return true;
	}else{
		printf("Error: enabling Read Only\r\n");
		return false;
	}
}

/**
 * set the tag as read/write and switch the led on
 * @param nfcNucleo board where the lead lives
 * @param tag helper class to manage the nfc tag
 * @return true if the change has effect
 */
bool disableReadOnly( XNucleoNFC01A1 *nfcNucleo){

	if(nfcNucleo->get_M24SR().disable_read_only(M24SR::DEFAULT_PASSWORD)== M24SR::M24SR_SUCCESS){
		printf("Disable Read Only\r\n");
		getReadOnlyLed(nfcNucleo)=false;
		getWriteableLed(nfcNucleo)=true;
		return true;
	}else{
		printf("Error: Disabling Read Only\r\n");
		return false;
	}
}

void sampleSync_lockTagContent() {

    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);

    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel);
    
    NDefLib::NDefNfcTag &tag = nfcNucleo->get_M24SR().get_NDef_tag();

    //when the user press the button set buttonPress to true
    
	#if defined(TARGET_STM)
        InterruptIn userButton(USER_BUTTON);
    #else
        InterruptIn userButton(SW2);
    #endif
        
    userButton.fall(setButtonPress);
	
	printf("Init Done\r\n");
	
    //restore the board as read/write
    if(!tag.open_session(true)){
    	printf("Error open session\r\n");
    	return;
    }
    if(nfcNucleo->get_M24SR().disable_all_password(M24SR::DEFAULT_PASSWORD)!= M24SR::M24SR_SUCCESS){
    	printf("Error disable_all_password\r\n");
    	return;
    }
    if(!tag.close_session()){
    	printf("Error close session\r\n");
    	return;
    }
    
    getReadOnlyLed(nfcNucleo)=false;
    getWriteableLed(nfcNucleo)=true;
    bool tagIsLock=false;
	
	printf("All Protection removed\r\n");
	
	nfcNucleo->get_led1()=true;
    while(true){

        if(buttonPress){
        		printf("button pressed\r\n");
        	if(tag.open_session()){
        		bool changeStatusResult;
				printf("session open\r\n");
				if(!tagIsLock) {
					changeStatusResult=enableReadOnly(nfcNucleo);
				} else {
					changeStatusResult=disableReadOnly(nfcNucleo);
				}

				if(changeStatusResult) {
					tagIsLock=!tagIsLock;
				} else {
					printf("Error change the tag status\r\n");
				}
				tag.close_session();
        	}else{
        		printf("Error opening the session, tag is in use\r\n");
        	}
        	buttonPress=false;
        }//if

        //wait next event
        __WFE();
    }//while

}//sample_countClick
