/**
  ******************************************************************************
  * @file       SampleSync_countClick.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo define a custom record that contains the number of time the user
  *             press the user button
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "XNucleoNFC01A1.h"
#include "NDefLib/NDefNfcTag.h"
#include "MyRecord.h"

/* Write a nfc message into a tag
 * @param nfcNucleo Board where write the data.
 * @param msg Message to write.
 */
static void write_message(XNucleoNFC01A1 *nfcNucleo,NDefLib::Message &msg){
    NDefLib::NDefNfcTag& tag = nfcNucleo->get_M24SR().get_NDef_tag();
    //open the i2c session with the nfc chip
    if(tag.open_session()){
        printf("Session opened\r\n");

        nfcNucleo->get_led1()=! nfcNucleo->get_led1();
        
        //write the tag
        if(tag.write(msg)){
            printf("message wrote\r\n");
            nfcNucleo->get_led2()=!nfcNucleo->get_led2();
        }//if

        //close the i2c session
        if(tag.close_session()){
            printf("Session closed\r\n");
            nfcNucleo->get_led3()=!nfcNucleo->get_led3();
        }
    }//if open session
}//writeMessage


static volatile bool buttonPress=false; /// true when the user press the message

/**
 * Call back called when the user press the button.
 */
static void set_button_press(){
    buttonPress=true;
}//if buttonPress

/**
 * Write a castum record that count how many times the user press the button.
 */
void sampleSync_countClick() {
    
    //instance the board with the default paramiters
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel);

    //set the button interrupt
    #if defined(TARGET_STM)
        InterruptIn userButton(USER_BUTTON);
    #else
        InterruptIn userButton(SW2);
    #endif
    userButton.fall(set_button_press);
    
    printf("Init Done\r\n");
    
    //create the NDef message and record
    NDefLib::Message msg;
    MyRecord rClickCount;
    msg.add_record(&rClickCount);
    
    write_message(nfcNucleo,msg);
    
    printf("Start Main Loop\r\n");
    while(true){
        
        if(buttonPress){
            //change the record content
            rClickCount.increment_click();
            //write the new record content
            write_message(nfcNucleo,msg);  
            //wait a new button press          
            buttonPress=false;
        }//if
        //wait next event
        __WFE();
    }//while

}//sample_countClick
