/**
  ******************************************************************************
  * @file       SampleAsync_writeUrl.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo write an ndef message with an url inside
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "XNucleoNFC01A1.h"
#include "NDefLib/NDefNfcTag.h"
#include "NDefLib/RecordType/RecordURI.h"

/**
 * Chain of callback that will crate a Uri record and write it.
 * After each operation the class will switch on a led
 */
class WriteUriCallbacks : public NDefLib::NDefNfcTag::Callbacks{

	DigitalOut &mOnOpenSession;
	DigitalOut &mOnWrite;
	DigitalOut &mOnCloseSession;

	NDefLib::Message msg;

public:

	/**
	 * create the callback chain
	 * @param onOpenSession led to switch on when the session open
	 * @param onWrite led to switch on when the write end
	 * @param onCloseSession led to switch on when the session end
	 */
	WriteUriCallbacks(DigitalOut &onOpenSession,DigitalOut &onWrite,
			DigitalOut &onCloseSession):mOnOpenSession(onOpenSession),
					mOnWrite(onWrite),mOnCloseSession(onCloseSession){};

	/**
	 * crate the new message and write it
	 * @param tag tag where write the message
	 * @param success true if the session correctly open
	 */
	virtual void on_session_open(NDefLib::NDefNfcTag *tag,bool success){
		if(!success){
			printf("Error OpenSession\n\r");
		}//else
		printf("Session Open\n\r");
		//ask to have an interrupt when the command finish
		mOnOpenSession=1;

		NDefLib::RecordURI *rUri = 
			new NDefLib::RecordURI(NDefLib::RecordURI::HTTP_WWW,"http://www.st.com");
		msg.add_record(rUri);

		tag->write(msg);
	}

	/**
	 * request to close the session
	 * @param tag tag where close the session
	 * @param success true if the message is correctly wrote
	 * @param message wrote
	 */
	virtual void on_message_write(NDefLib::NDefNfcTag *tag,bool success){
		NDefLib::Message::remove_and_delete_all_record(msg);
		if(!success) {
			printf("Error writing tag!\n\r");
		} else {
			printf("Tag Written!\n\r");
			mOnWrite=1;
		}//if-else
		tag->close_session();
	}

	/**
	 * switch on the led
	 * @param tag where the session is closed
	 * @param success true if the session is correctly close
	 */
	virtual void on_session_close(NDefLib::NDefNfcTag*,bool success){
		if(success) {
			printf("Session closed\n\r");
			mOnCloseSession=1;
		} else {
			printf("Error Session closed\n\r");
		}
	}

};


/** variable set to true when we receive an interrupt from the nfc component*/
static volatile bool nfcInterruptFlag;

/** Nfc ISR called when the nfc component has a message ready*/
static void nfcInterruptCallback(){
	nfcInterruptFlag=true;
}//nfcInterruptCallback

void sampleAsync_writeUrl(){
	//create the nfc component
	I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
	XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,&nfcInterruptCallback);


    M24SR &nfcTag =nfcNucleo->get_M24SR();

    //No call back needed since default behavior is sync
    if(nfcTag.get_session()==M24SR::M24SR_SUCCESS) {
    	nfcTag.manage_I2C_GPO(M24SR::I2C_ANSWER_READY);//Set async mode
	}

	NDefLib::NDefNfcTag &tag = nfcNucleo->get_M24SR().get_NDef_tag();

	//crate the callback to use for write a tag
	WriteUriCallbacks NDefCallback(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
	tag.set_callback(&NDefCallback); //set the callback
	tag.open_session(); //start the callback chain

	printf("Start Main Loop\n\r");
	while(true){
		if(nfcInterruptFlag){
			nfcInterruptFlag=false;
			//manage an async event from the nfc component
			nfcTag.manage_event();
		}//if
		__WFE();
	}//while
}
