/**
  ******************************************************************************
  * @file       SampleAsync_writeAndChangeAll.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo write an ndef message different records, when the user press the buttun
  *             read the tag, change some data and write it again
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "NDefLib/NDefNfcTag.h"

#include "NDefLib/RecordType/RecordAAR.h"
#include "NDefLib/RecordType/RecordSMS.h"
#include "NDefLib/RecordType/RecordGeo.h"
#include "NDefLib/RecordType/RecordURI.h"
#include "NDefLib/RecordType/RecordMail.h"
#include "NDefLib/RecordType/RecordText.h"
#include "NDefLib/RecordType/RecordMimeType.h"
#include "NDefLib/RecordType/RecordVCard.h"
#include "NDefLib/RecordType/RecordWifiConf.h"

#include "XNucleoNFC01A1.h"

/**
 * Shift the led status between the 3 leds.
 */
static void shift_led(DigitalOut &led1,DigitalOut &led2,DigitalOut &led3){
    const uint8_t prevLed1=led1;
    const uint8_t prevLed2=led2;
    const uint8_t prevLed3=led3;
    led1=prevLed3;
    led2=prevLed1;
    led3=prevLed2;
}

/**
 * Print in the console some data about the record.
 * @param r Record to print.
 */
static void print_record(NDefLib::Record *const r){
    using namespace NDefLib;
    switch(r->get_type()){
        case Record::TYPE_TEXT: {
            const RecordText *const temp = ( RecordText* )r;
            printf("Read Text: %s\r\n",temp->get_text().c_str());
            break; }
        case Record::TYPE_AAR:{
            const RecordAAR *const temp = ( RecordAAR* )r;
            printf("Read ARR: %s\r\n",temp->get_package().c_str());
            break; }
        case Record::TYPE_MIME:{
            const RecordMimeType *const temp = ( RecordMimeType* )r;
            printf("Read mimeType: %s\r\n",temp->get_mime_type().c_str());
            printf("Read mimeData: %s\r\n",
                    std::string((const char*)temp->get_mime_data(),
                            temp->get_mime_data_lenght()).c_str());
            break;}
        case Record::TYPE_URI:{
            RecordURI *const temp = (RecordURI*)r;
            printf("Read uriId: %d\r\n",temp->get_uri_id());
            printf("Read uriType: %s\r\n",temp->get_uri_type().c_str());
            printf("Read uriContent: %s\r\n",temp->get_content().c_str());
            break;}
        case Record::TYPE_URI_MAIL:{
            const RecordMail*const temp = (RecordMail*)r;
            printf("Read Dest: %s\r\n",temp->get_to_address().c_str());
            printf("Read Subject: %s\r\n",temp->get_subject().c_str());
            printf("Read Body: %s\r\n",temp->get_body().c_str());
            break;}
        case Record::TYPE_URI_SMS:{
            const RecordSMS*const temp = (RecordSMS*)r;
            printf("Read number: %s\r\n",temp->get_number().c_str());
            printf("Read message: %s\r\n",temp->get_messagge().c_str());
            break;}
        case Record::TYPE_URI_GEOLOCATION:{
            const RecordGeo*const temp = (RecordGeo*)r;
            printf("Read lat: %f\r\n",temp->get_latitude());
            printf("Read long: %f\r\n",temp->get_longitude());
            break;}
        case Record::TYPE_MIME_VCARD:{
            const RecordVCard *const temp = (RecordVCard*)r;
            printf("Read Name: %s\r\n",(*temp)[RecordVCard::NAME].c_str());
            printf("Read Mail: %s\r\n",(*temp)[RecordVCard::EMAIL_WORK].c_str());
            printf("Read ORG: %s\r\n",(*temp)[RecordVCard::ORGANIZATION].c_str());
            break;}
        case Record::TYPE_WIFI_CONF:{
			const RecordWifiConf *const temp = (RecordWifiConf*)r;
			printf("Nework Name: %s\r\n",temp->get_network_ssid().c_str());
			printf("Nework Key: %s\r\n",temp->get_network_key().c_str());
			printf("Nework Auth: %X\r\n",temp->get_auth_type());
			printf("Nework Enc: %X\r\n",temp->get_encryption());
			break;}
        case Record::TYPE_UNKNOWN:{
            printf("Unknown record\r\n");
            break;}
    }//switch
}//printRecord

/**
 * Change the record content.
 * @param r Record to change.
 */
static void change_record(NDefLib::Record const* r){
    using namespace NDefLib;
    switch(r->get_type()){
        case Record::TYPE_TEXT: {
            RecordText *temp = (RecordText*)r;
            temp->set_text("Hello");
            break; }
        case Record::TYPE_AAR:{
            RecordAAR *temp = (RecordAAR*)r;
            temp->set_package("set Package Ok");
            break; }
        case Record::TYPE_MIME:{
            RecordMimeType *temp = (RecordMimeType*)r;
            temp->copy_mime_data((const uint8_t *)"String2",sizeof("String2"));
            break;}
        case Record::TYPE_URI:{
            RecordURI *temp = (RecordURI*)r;
            temp->set_content("mbed.com");
            break;}
        case Record::TYPE_URI_MAIL:{
            RecordMail *temp = (RecordMail*)r;
            temp->set_to_address("newMail@st.com");
            temp->set_subject("tag change");
            temp->set_body("read/change Works!");
            break;}
        case Record::TYPE_URI_SMS:{
            RecordSMS *temp = (RecordSMS*)r;
            temp->set_message("Message Change");
            temp->set_number("0987654321");
            break;}
        case Record::TYPE_URI_GEOLOCATION:{
            RecordGeo *temp = (RecordGeo*)r;
            temp->set_latitude(-temp->get_latitude());
            temp->set_longitude(-temp->get_longitude());
            break;}
        case Record::TYPE_MIME_VCARD:{
            RecordVCard *temp = (RecordVCard*)r;
            (*temp)[RecordVCard::NAME]="name change";
            (*temp)[RecordVCard::NICKNAME]="nic change";
            break;}
        case Record::TYPE_WIFI_CONF:{
			RecordWifiConf * temp = (RecordWifiConf*)r;
			temp->set_network_ssid("hackMe");
			temp->set_network_key("qwerty");
			temp->set_auth_type(RecordWifiConf::AUTH_WPA2_PSK);
			temp->set_encryption_type(RecordWifiConf::ENC_TYPE_AES_TKIP);
			break;}
        case Record::TYPE_UNKNOWN:{
            printf("Unknown record\r\n");
            break;}
    }//switch
}//changeRecord


/**
 * Class that print read a message and print it on console,
 * and enable the user button when it finish
 */
class ReadMessageCallback : public NDefLib::NDefNfcTag::Callbacks{

	NDefLib::Message *mMsg; /// Message where read
	bool *mDisableButton; /// enable the user button

	void on_finish(const char *const msg){
		printf(msg);
		*mDisableButton=false;
	}

public:
	/**
	 *
	 * @param disableEnable set to false when the read finish
	 */
	ReadMessageCallback(bool *disableEnable):mMsg(NULL),
		mDisableButton(disableEnable){}

	/**
	 * Create the message and ask to read it
	 */
	virtual void on_session_open(NDefLib::NDefNfcTag *tag,bool success){
		if(!success){
			return on_finish("Error Opening Session\r\n");
		}//if
		mMsg = new NDefLib::Message;
		tag->read(mMsg);
	}

	/**
	 * Print all the record inside the message
	 */
	virtual void on_message_read(NDefLib::NDefNfcTag *tag,bool success,
			const NDefLib::Message *readMsg){
		if(!success || readMsg->get_N_records()==0){
			delete mMsg;
			return on_finish("Error Reading\r\n");
		}else{
			printf("Message Read\r\n\n");
			for(uint32_t i=0;i<readMsg->get_N_records();i++){
				NDefLib::Record *r = (*readMsg)[i];
				print_record(r);
				delete r;
			}//for
			delete mMsg;
		}//if-else
		tag->close_session();
	}

	/**
	 * Enable the button
	 */
	virtual void on_session_close(NDefLib::NDefNfcTag *,bool success){
		if(success)
			on_finish("Read Session close\r\n");
		else
			on_finish("Read Session close Error\r\n");
	}

};

/**
 * Read the message, change some data and write it back
 */
class ChangeMessageCallback : public NDefLib::NDefNfcTag::Callbacks{

	ReadMessageCallback* mReadMessage;
	NDefLib::Message *mMsg;

	/**
	 * Change all the record in the message.
	 * @param readMsg Message to change.
	 */
	static void change_content(const NDefLib::Message *readMsg){
		for(uint32_t i=0;i<readMsg->get_N_records();i++){
			change_record((*readMsg)[i]);
		}//for
	}//changeContent

	/**
	 * Delete all the record and the message.
	 */
	void delete_message(){
		NDefLib::Message::remove_and_delete_all_record(*mMsg);
		delete mMsg;
	}//deleteMessage

public:

	/**
	 * @param readCallback Callback needed to print the tag content after the change
	 */
	ChangeMessageCallback(ReadMessageCallback* readCallback):
		mReadMessage(readCallback),mMsg(NULL){}

	/**
	 * Ask to read the tag content
	 */
	virtual void on_session_open(NDefLib::NDefNfcTag *tag,bool success){
		if(!success){
			printf("Error Opening the session");
			return;
		}//else
		mMsg = new NDefLib::Message;
		tag->read(mMsg);
	}

	/**
	 * Change the message content and write it back
	 */
	virtual void on_message_read(NDefLib::NDefNfcTag *tag,bool success,
			const NDefLib::Message *readMsg){
		if(!success || readMsg->get_N_records()==0){
			printf("Error Reading\r\n");
			delete_message();
		}else{
			printf("Message Read: change message content\r\n");
			change_content(readMsg);
			printf("Start write new message\r\n");
			tag->write(*mMsg);
		}//if-else
	}

	/**
	 * Delete the Message and close the session
	 */
	virtual void on_message_write(NDefLib::NDefNfcTag *tag,bool success){
		delete_message();
		if(success){
			printf("Message Wrote\r\n");
			tag->close_session();
		}else
			printf("Error Writing\r\n");
	}//onMessageWrite

	/**
	 * Set the callback for print the tag content and open a new session
	 */
	virtual void on_session_close(NDefLib::NDefNfcTag *tag,bool success){
		if(success){
			printf("Change Session close\r\n");
			tag->set_callback(mReadMessage);
			tag->open_session();
		}else
			printf("Change Session close Error\r\n");
	}

};

/**
 * Create and write a message in a nfc tag
 */
class WriteMessageCallback : public NDefLib::NDefNfcTag::Callbacks{

	ReadMessageCallback* mReadMessage;
	NDefLib::Message mMsg;

public:

	/**
	 *
	 * @param readCallback Callbacks to use for print the tag content
	 */
	WriteMessageCallback(ReadMessageCallback* readCallback):
		mReadMessage(readCallback){}

	/**
	 *
	 */
	virtual void on_session_open(NDefLib::NDefNfcTag *tag,bool success){
		if(!success){
			printf("Error Opening the Session\r\n");
			return;
		}//else

		printf("Session open\r\n");

		NDefLib::RecordAAR *rAAR = 
			new NDefLib::RecordAAR("com.st.BlueMS");
		mMsg.add_record(rAAR);

		NDefLib::RecordSMS *rSMS = 
			new NDefLib::RecordSMS("123456789","st.com.BlueMS");
		mMsg.add_record(rSMS);

		NDefLib::RecordGeo *rGeo = 
			new NDefLib::RecordGeo(123.123,-456.789);
		mMsg.add_record(rGeo);

		NDefLib::RecordURI *rUri = 
			new NDefLib::RecordURI(NDefLib::RecordURI::HTTP_WWW,"http://www.st.com");
		mMsg.add_record(rUri);

		NDefLib::RecordMail *rMail = 
			new NDefLib::RecordMail("mail@st.com","ciao","da nfc tag");
		mMsg.add_record(rMail);

		NDefLib::RecordMimeType *rText1 =
			new NDefLib::RecordMimeType("text/plain",(const uint8_t*)"Ciao",4);
		mMsg.add_record(rText1);

		NDefLib::RecordText *rText2 =
			new NDefLib::RecordText(NDefLib::RecordText::UTF8,"it","ciao");
		mMsg.add_record(rText2);

		NDefLib::RecordWifiConf *rWifi =
			new NDefLib::RecordWifiConf("OpenNetwork");
		mMsg.add_record(rWifi);

		NDefLib::RecordVCard::VCardInfo_t cardInfo;
		cardInfo[NDefLib::RecordVCard::FORMATTED_NAME]="prova prova1";
		cardInfo[NDefLib::RecordVCard::ADDRESS_HOME]=";;1 Main St.;Springfield;IL;12345;USA";
		cardInfo[NDefLib::RecordVCard::ADDRESS_WORK]=";;2 Main St.;Springfield;IL;12345;USA";
		cardInfo[NDefLib::RecordVCard::EMAIL_WORK]="workmail@st.com";
		cardInfo[NDefLib::RecordVCard::EMAIL_HOME]="homemail@st.com";
		cardInfo[NDefLib::RecordVCard::GEO]="39.95;-75.1667";
		cardInfo[NDefLib::RecordVCard::IMPP]="aim:johndoe@aol.com";
		cardInfo[NDefLib::RecordVCard::NAME]="prova2;prova3";
		cardInfo[NDefLib::RecordVCard::NICKNAME]="test";
		cardInfo[NDefLib::RecordVCard::NOTE]="A good test";
		cardInfo[NDefLib::RecordVCard::ORGANIZATION]="STM";
		cardInfo[NDefLib::RecordVCard::TEL_HOME]="123";
		cardInfo[NDefLib::RecordVCard::TEL_MOBILE]="456";
		cardInfo[NDefLib::RecordVCard::TEL_WORK]="789";
		cardInfo[NDefLib::RecordVCard::TITLE]="King";
		cardInfo[NDefLib::RecordVCard::URL]="www.st.com";
		cardInfo[NDefLib::RecordVCard::PHOTO_URI]="http://www.st.com/st-web-ui/static/active/en/fragment/multimedia/image/picture/customer_focus.jpg";
		NDefLib::RecordVCard *rVCard = new NDefLib::RecordVCard(cardInfo);
		mMsg.add_record(rVCard);

		//write it
		tag->write(mMsg);
	}


	/**
	 * Close the session
	 */
	virtual void on_message_write(NDefLib::NDefNfcTag *tag,bool success){
		NDefLib::Message::remove_and_delete_all_record(mMsg);
		if(!success){
			printf("Error Writing\r\n");
			return;
		}
		printf("Message wrote\r\n");
		tag->close_session();
	}

	/**
	 * Set the callback to print the tag content and open a new session
	 */
	virtual void on_session_close(NDefLib::NDefNfcTag *tag,bool success){
		if(success){
			printf("Write Session close\r\n");
			tag->set_callback(mReadMessage);
			tag->open_session();
		}else
			printf("Write Session close Error\r\n");
	}//onSessionClose

};


static bool buttonPress=false; /// true when the user press the message

/**
 * Call back called when the user press the button
 */
static void set_button_press(){
    buttonPress=true;
}//if buttonPress

static volatile bool nfcEvent=false; /// true if there is an nfc interrupt

/**
 * Call back called when the user press the button
 */
static void set_nfc_event(){
	nfcEvent=true;
}//

/**
 * Write a message and when the user press the button it read the message, change it and update it.
 */
void sampleAsync_writeAndChangeAll() {
    DigitalOut nucleoLed(LED1);
    
    //instance the board with the default parameters
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,&set_nfc_event);

    //retrieve the Nfc component
    M24SR &nfc = nfcNucleo->get_M24SR();
    //retrieve the NdefLib interface
    NDefLib::NDefNfcTag& tag = nfc.get_NDef_tag();

    //switch on the first led    
    nfcNucleo->get_led1()=1;
    nfcNucleo->get_led2()=0;
    nfcNucleo->get_led3()=0;

    ReadMessageCallback readMessageCallback(&buttonPress);
    WriteMessageCallback writeBigMessageCallback(&readMessageCallback);
    ChangeMessageCallback changeMessageCallback(&readMessageCallback);

    //Enable async mode
    if(nfc.get_session()==M24SR::M24SR_SUCCESS)
        nfc.manage_I2C_GPO(M24SR::I2C_ANSWER_READY);

    //write the message
    tag.set_callback(&writeBigMessageCallback);
    tag.open_session();

    //enable the button
	#if defined(TARGET_STM)
        InterruptIn userButton(USER_BUTTON);
    #else
        InterruptIn userButton(SW2);
    #endif
    userButton.fall(set_button_press);

	printf("Start main loop\r\n");
    while(true) {
        if(buttonPress){
        	buttonPress=false;
        	shift_led(nfcNucleo->get_led1(),nfcNucleo->get_led2(),nfcNucleo->get_led3());
        	tag.set_callback(&changeMessageCallback);
        	tag.open_session();
        }else if (nfcEvent){
        	nfcEvent=false;
        	nucleoLed=!nucleoLed;
        	nfcNucleo->get_M24SR().manage_event();
        }//if-else
        __WFE();
    }//while
}