/**
  ******************************************************************************
  * @file       SampleAsync_lockWriteTag.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo define a lock/unlock the tag content when the user press
  *  the user button
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"

#include "XNucleoNFC01A1.h"

static volatile bool buttonPress=false; /// true when the user press the message
static volatile bool nfcEvent=false; /// true when the user press the message

static void set_nfc_event_callback(){
	nfcEvent=true;
}//if buttonPress

/**
 * Call back called when the user press the button.
 */
static void set_button_press(){
    buttonPress=true;
}//if buttonPress

/**
 * Class with the command chain needed to remove all the read/write password
 */
class RemoveAllPasswordStatusCallback : public M24SR::Callbacks {

public:

	RemoveAllPasswordStatusCallback():mIsComamndRunning(false){}

	/**
	 * True when the command chain finish, and the tag can now be read and write
	 * @return true if the command chain finish
	 */
	bool is_finish(){
		return !mIsComamndRunning;
	}


private:
	bool mIsComamndRunning; /// false when the command chain finish

	/**
	 * end the command chain and print an error message
	 * @param status error code of the last command
	 */
	void on_error(const M24SR::StatusTypeDef status){
		mIsComamndRunning=false;
		printf("Error Code: %X \r\n",status);
	}

	virtual void on_session_open(M24SR *tag,M24SR::StatusTypeDef status){
		if(status!=M24SR::M24SR_SUCCESS) {
			return on_error(status);
		}
		mIsComamndRunning=true;
		printf("Session Opened\r\n");
		tag->select_application();
	}

	virtual void on_selected_application(M24SR *tag,M24SR::StatusTypeDef status){
		if(status!=M24SR::M24SR_SUCCESS) {
			return on_error(status);
		}
			
		printf("Select application\r\n");
		tag->select_NDEF_file(0x0001);
	}

	virtual void on_selected_NDEF_file(M24SR *tag,M24SR::StatusTypeDef status){
		if(status!=M24SR::M24SR_SUCCESS) {
			return on_error(status);
		}

		printf("Select NDef\r\n");
		tag->disable_all_password(M24SR::DEFAULT_PASSWORD);
	}

	virtual void on_disable_all_password(M24SR *tag, M24SR::StatusTypeDef status){
		if(status!=M24SR::M24SR_SUCCESS) {
			return on_error(status);
		}
			
		printf("Disable all Password\r\n");
		tag->deselect();
	}

	virtual void on_deselect(M24SR *,M24SR::StatusTypeDef status){
		if(status!=M24SR::M24SR_SUCCESS) {
			return on_error(status);
		}
			
		printf("Close Application\r\n");
		mIsComamndRunning=false;
	}

};

/**
 * Chain of call needed to change the tag status between read only and write/read
 */
class ChangeTagStatusCallback : public M24SR::Callbacks {

	M24SR &mNfcTag; /// object to change
	DigitalOut& mReadOnlyLed; /// led to switch on when it is in read only mode
	DigitalOut& mReadWriteWriteLed; /// led to switch on when is in read/write mode

	bool mIsReadOnly; /// current tag status
	bool mIsComamndRunning; /// true if the chain is running

	public:

		/**
		 * Build a chain of callback needed to change the tag status.
		 * @param nfcTag Tag to change.
		 * @param readLed Led to switch on when in read only mode.
		 * @param writeLed Led to switch on when in read/write mode.
		 */
		ChangeTagStatusCallback(M24SR &nfcTag, DigitalOut& readLed, DigitalOut& writeLed):
			mNfcTag(nfcTag),mReadOnlyLed(readLed),mReadWriteWriteLed(writeLed),mIsReadOnly(false),
			mIsComamndRunning(false){
			mReadOnlyLed=false;
			mReadWriteWriteLed=true;
		}

		/**
		 * Change the tag status
		 */
		void change_status(){
			mIsReadOnly=!mIsReadOnly;
			//if it is not running a previous command
			if(!mIsComamndRunning){
				mNfcTag.get_session();
			}//if
		}//changeStatus;

	private:

		/**
		 * When an error happen switch on both the led and print the error code.
		 * @param status Error code.
		 */
		void on_error(const M24SR::StatusTypeDef status){
			mIsComamndRunning=false;
			mReadOnlyLed=true;
			mReadWriteWriteLed=true;
			printf("Error %X changing the status\r\n",status);
		}

		virtual void on_session_open(M24SR *,M24SR::StatusTypeDef status){
			if(status!=M24SR::M24SR_SUCCESS) {
				return on_error(status);
			}
			printf("Session Open\r\n");
			mNfcTag.select_application();
		}

		virtual void on_selected_application(M24SR *,M24SR::StatusTypeDef status){
			if(status!=M24SR::M24SR_SUCCESS) {
				return on_error(status);
			}
	
			printf("Select Application\r\n");
			mNfcTag.select_NDEF_file(0x0001);
		}

		virtual void on_selected_NDEF_file(M24SR *,M24SR::StatusTypeDef status){
			if(status!=M24SR::M24SR_SUCCESS) {
				return on_error(status);
			}
			printf("Select NDef file\r\n");
			if(mIsReadOnly){
				mNfcTag.enable_read_only(M24SR::DEFAULT_PASSWORD);
			}else{
				mNfcTag.disable_read_only(M24SR::DEFAULT_PASSWORD);
			}
		}


		virtual void on_enable_read_only(M24SR *,M24SR::StatusTypeDef status){
			if(status!=M24SR::M24SR_SUCCESS) {
				return on_error(status);
			}
			printf("Ready only Enabled\r\n");
			if(!mIsReadOnly) {//if in the meantime the status change
				mNfcTag.disable_read_only(M24SR::DEFAULT_PASSWORD);
			} else {
				mReadOnlyLed=true;
				mReadWriteWriteLed=false;
				mNfcTag.deselect();
			}
		}

		virtual void on_disable_read_only(M24SR *,M24SR::StatusTypeDef status){
			if(status!=M24SR::M24SR_SUCCESS) {
				return on_error(status);
			}
			printf("Ready only Disabled\r\n");
			if(mIsReadOnly) {//if in the meantime the status change
				mNfcTag.enable_read_only(M24SR::DEFAULT_PASSWORD);
			} else {
				mReadOnlyLed=false;
				mReadWriteWriteLed=true;
				mNfcTag.deselect();
			}
		}

		virtual void on_deselect(M24SR *,M24SR::StatusTypeDef status){
			if(status!=M24SR::M24SR_SUCCESS) {
				return on_error(status);
			}
			printf("Close Application\r\n");
			mIsComamndRunning=false;
		}
};

/**
 * When the user press the button enable/disable the tag write protection.
 */
void sampleAsync_lockTagContent() {

    //instance the board with the default parameters
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,&set_nfc_event_callback);
	
    //when the user press the button set buttonPress to true
	#if defined(TARGET_STM)
        InterruptIn userButton(USER_BUTTON);
    #else
        InterruptIn userButton(SW2);
    #endif
    userButton.fall(set_button_press);
    
    M24SR &nfcTag =nfcNucleo->get_M24SR();

    //set async mode
    if(nfcTag.get_session()==M24SR::M24SR_SUCCESS) {
    	nfcTag.manage_I2C_GPO(M24SR::I2C_ANSWER_READY);
	}


    ChangeTagStatusCallback changeTagStatus(nfcTag,nfcNucleo->get_led2(),nfcNucleo->get_led3());
    RemoveAllPasswordStatusCallback removeAllPasswordStatusCallback;

    nfcTag.set_callback(&removeAllPasswordStatusCallback);
    nfcTag.get_session();
    nfcNucleo->get_led1()=true;

    while(true){

        if(buttonPress && removeAllPasswordStatusCallback.is_finish()){
        	buttonPress=false;
            nfcTag.set_callback(&changeTagStatus);
        	changeTagStatus.change_status();
        }//if
        if(nfcEvent){
        	nfcEvent=false;
        	nfcTag.manage_event();
        }
        //wait next event
        __WFE();
    }//while

}//sampleAsync_lockTagContent
