/**
  ******************************************************************************
  * @file       SampleAsync_countClick.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo define a custom record that contains the number of time the user
  *             press the user button
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "XNucleoNFC01A1.h"
#include "NDefLib/NDefNfcTag.h"

#include "MyRecord.h"

/**
 * class with the callback needed to update the record in the tag
 */
class WriteMyRecordCallback : public NDefLib::NDefNfcTag::Callbacks {

	/**
	 * board where change the led status
	 */
	XNucleoNFC01A1 *mNfcNucleo;

	/**
	 * message to write
	 */
	NDefLib::Message &mMsg;

	/**
	 * true if the message is wrote in the tag
	 */
	bool mCommandFinish;

	/**
	 * true if an update request is done while writing a tag
	 */
	bool mRequestRefresh;


	public:

		/**
		 * build an object write an message into a nfc tag
		 * @param nfcNucleo board with the leds and nfc tag
		 * @param msg message to write
		 */
		WriteMyRecordCallback(XNucleoNFC01A1 *nfcNucleo,NDefLib::Message &msg):
			mNfcNucleo(nfcNucleo),mMsg(msg),mCommandFinish(true),
			mRequestRefresh(false){}

		/**
		 * start the write process
		 */
		void update_message(){
			if(mCommandFinish){
				mNfcNucleo->get_M24SR().get_NDef_tag().open_session();
			}else{// if it is doing something remember this request
				mRequestRefresh=true;
			}
		}//updateMessage

	private:

		void on_error(){
			printf("Error updating the tag\r\n");
			mCommandFinish=false;
		}

		/**
		 * when the session is open change the led status and ask to write the message
		 */
		virtual void on_session_open(NDefLib::NDefNfcTag *tag,bool success){
			if(!success)
				return on_error();
			mNfcNucleo->get_led1()=!mNfcNucleo->get_led1();
			printf("Session opened\r\n");
			tag->write(mMsg);
		}

		/**
		 * if the user ask to update the value it write again the message, otherwise close the session
		 */
		virtual void on_message_write(NDefLib::NDefNfcTag *tag,bool success){
			if(!success) {
				return on_error();
			}
			mNfcNucleo->get_led2()=!mNfcNucleo->get_led2();
			printf("Tag wrote\r\n");
			if(mRequestRefresh){
				mRequestRefresh=false;
				tag->write(mMsg);
			}else {
				tag->close_session();
			}
			//if-else

		};

		/**
		 * if the user ask to update the value it open a new session
		 */
		virtual void on_session_close(NDefLib::NDefNfcTag *tag,bool success){
			if(!success)
				return on_error();
			mNfcNucleo->get_led3()=!mNfcNucleo->get_led3();
			printf("Session Close\r\n");
			if(mRequestRefresh){
				mRequestRefresh=false;
				tag->open_session();
			}else {
				mCommandFinish=true;
			}
			//if-else
		};

};


static volatile bool buttonPress=false; /// true when the user press the message
static volatile bool nfcEvent=false; /// true when the user press the message

/**
 * Call back called when the user press the button.
 */
static void set_button_press(){
    buttonPress=true;
}//if buttonPress

static void set_nfc_event_callback(){
	nfcEvent=true;
}//if buttonPress

/**
 * Write a custom record that count how many times the user press the button.
 */
void sampleAsync_countClick() {
    
    //instance the board with the default parameters
    I2C i2cChannel(XNucleoNFC01A1::DEFAULT_SDA_PIN,XNucleoNFC01A1::DEFAULT_SDL_PIN);
    XNucleoNFC01A1 *nfcNucleo = XNucleoNFC01A1::instance(i2cChannel,&set_nfc_event_callback);

    //set the button interrupt
	#if defined(TARGET_STM)
        InterruptIn userButton(USER_BUTTON);
    #else
        InterruptIn userButton(SW2);
    #endif
    userButton.fall(set_button_press);
    
    M24SR &nfcTag =nfcNucleo->get_M24SR();

   //No call back needed since default behavior is sync
    if(nfcTag.get_session()==M24SR::M24SR_SUCCESS) {
    	nfcTag.manage_I2C_GPO(M24SR::I2C_ANSWER_READY);//Set async mode
	}

    //create the NDef message and record
    NDefLib::Message msg;
    MyRecord rClickCount;
    msg.add_record(&rClickCount);
    
    WriteMyRecordCallback writeMyRecordCallback(nfcNucleo,msg);

    nfcNucleo->get_M24SR().get_NDef_tag().set_callback(&writeMyRecordCallback);
    writeMyRecordCallback.update_message();
    while(true){
        
        if(buttonPress){
        	buttonPress=false;
            //change the record content
            rClickCount.increment_click();
            printf("upgade Ndef message\r\n");
            writeMyRecordCallback.update_message();
        }//if
        if(nfcEvent){
        	nfcEvent=false;
        	nfcNucleo->get_M24SR().manage_event();
        }//if
        //wait next event
        __WFE();
    }//while

}//sample_countClick
