/*
 * Type4NfcTagSTM24SR.h
 *
 *  Created on: Nov 5, 2015
 *      Author: giovanni visentini
 */

#ifndef TYPE4NFCTAGSTM24SR_H_
#define TYPE4NFCTAGSTM24SR_H_
#include <stdint.h>
#include "m24sr/m24sr_class.h"

#include "NDefLib/Type4NfcTag.h"

/**
 * implement the abstract method for use the NDefLib
 */
class Type4NfcTagSTM24SR: public NDefLib::Type4NfcTag {

public:
	/**
	 *
	 * @param device device where write the Ndef tags
	 */
	Type4NfcTagSTM24SR(M24SR &device) :
			mDevice(device), mMaxReadBytes(0xFF), mMaxWriteBytes(
					0xFF) {
	}


	virtual bool openSession(bool force = false);
	virtual bool closeSession();

	/**
	 * close the open session
	 */
	virtual ~Type4NfcTagSTM24SR() {
		if(isSessionOpen())
			closeSession();
	}//~Type4NfcTagSTM24SR

	bool enableReadPassword(const uint8_t* pCurrentWritePassword,
			const uint8_t* pNewPassword) {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::WritePwd, 0x10, pCurrentWritePassword)
				== NFC_SUCCESS) {
			/* Set new password */
			if (mDevice.ChangeReferenceData(M24SR::ReadPwd, pNewPassword)
					== NFC_SUCCESS)
				return mDevice.EnableVerificationRequirement(M24SR::ReadPwd)
						== NFC_SUCCESS;
		} //else
		/* M24SR already lock but password not known */
		return false;

	}

	bool disableReadPassword(const uint8_t* pCurrentWritePassword) {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::WritePwd, 0x10, pCurrentWritePassword)
				== NFC_SUCCESS) {
			/* Set new password */
			return mDevice.DisableVerificationRequirement(M24SR::ReadPwd)
					== NFC_SUCCESS;
		} else {
			/* M24SR already lock but password not known */
			return false;
		}
	}

	bool enableWritePassword(const uint8_t* pCurrentWritePassword,
			uint8_t* pNewPassword) {
		if (!isSessionOpen())
			return false;

		/* check we have the good password */
		if (mDevice.Verify(M24SR::WritePwd, 0x10, pCurrentWritePassword)
				== NFC_SUCCESS) {
			/* Set new password */
			if (mDevice.ChangeReferenceData(M24SR::WritePwd, pNewPassword)
					== NFC_SUCCESS)
			return mDevice.EnableVerificationRequirement(M24SR::WritePwd)
					== NFC_SUCCESS;
		}
		return false;
	}

	bool disableWritePassword(const uint8_t* pCurrentWritePassword) {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::WritePwd, 0x10, pCurrentWritePassword)
				== NFC_SUCCESS)
			return mDevice.DisableVerificationRequirement(M24SR::WritePwd)
					== NFC_SUCCESS;
		return false;
	}

	/**
	 * @brief  This fonction desactivate the need of read and write password for next access
	 * @param	pSuperUserPassword : I2C super user password to overwrite read and write password
	 * @retval SUCCESS : M24SR access is now free (no password needed)
	 * @retval ERROR : operation does not complete
	 */
	bool disableAllPassword(const uint8_t* pSuperUserPassword) {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::WritePwd, 0x10, pSuperUserPassword)
				== NFC_SUCCESS) {
			mDevice.DisablePermanentState(M24SR::ReadPwd);
			mDevice.DisablePermanentState(M24SR::WritePwd);

			mDevice.DisableVerificationRequirement(M24SR::ReadPwd);
			mDevice.DisableVerificationRequirement(M24SR::WritePwd);

			/* reset password */
			mDevice.ChangeReferenceData(M24SR::ReadPwd, pSuperUserPassword);
			mDevice.ChangeReferenceData(M24SR::WritePwd, pSuperUserPassword);
			return true;
		}
		return false;
	}

	/**
	 * @brief  This fonction enable read only mode
	 * @param	pCurrentWritePassword : Write password is needed to have right to enable read only mode
	 * @retval SUCCESS : M24SR access is now forbidden in write mode
	 * @retval ERROR : operation does not complete
	 */
	bool enableReadOnly(const uint8_t* pCurrentWritePassword) {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::WritePwd, 0x10, pCurrentWritePassword)
				== NFC_SUCCESS) {
			/* lock write to have read only */
			return mDevice.EnablePermanentState(M24SR::WritePwd) == NFC_SUCCESS;
		}
		return false;
	}

	/**
	 * @brief  This fonction disable read only mode
	 * @param	pCurrentWritePassword : Write password is needed to have right to disable read only mode
	 * @retval SUCCESS : M24SR write access is now allowed
	 * @retval ERROR : operation does not complete
	 */
	bool disableReadOnly() {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::I2CPwd, 0x10, M24SR::DEFAULT_PASSWORD)
				== NFC_SUCCESS) {
			/* disable write protection to disable read only mode */
			if (mDevice.DisablePermanentState(M24SR::WritePwd) == NFC_SUCCESS)
				return mDevice.DisableVerificationRequirement(M24SR::WritePwd)
						== NFC_SUCCESS;
		} //if
		return false;
	}

	/**
	 * @brief  This fonction enable write only mode
	 * @param	pCurrentWritePassword : Write password is needed to have right to enable write only mode
	 * @retval SUCCESS : M24SR access is now forbidden in read mode
	 * @retval ERROR : operation does not complete
	 */
	bool enableWriteOnly(const uint8_t* pCurrentWritePassword) {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::WritePwd, 0x10, pCurrentWritePassword)
				== NFC_SUCCESS) {
			/* disable read access and keep write */
			return mDevice.EnablePermanentState(M24SR::ReadPwd) == NFC_SUCCESS;
		}
		return false;
	}

	/**
	 * @brief  This fonction disable write only mode
	 * @param	pCurrentWritePassword : Write password is needed to have right to disable write only mode
	 * @retval SUCCESS : M24SR read access is now allowed
	 * @retval ERROR : operation does not complete
	 */
	bool disableWriteOnly() {
		if (!isSessionOpen())
			return false;

		if (mDevice.Verify(M24SR::I2CPwd, 0x10, M24SR::DEFAULT_PASSWORD)
				== NFC_SUCCESS) {
			/* disable write only -> enable write acces */
			if (mDevice.DisablePermanentState(M24SR::ReadPwd) == NFC_SUCCESS)
				return mDevice.DisableVerificationRequirement(M24SR::ReadPwd)
						== NFC_SUCCESS;
		}
		return false;
	}

protected:
	virtual bool writeByte(const uint8_t *buffer, const uint16_t length, uint16_t offset);
	virtual bool readByte(const uint16_t byteOffset, const uint16_t byteLength,
			uint8_t *buffer);

private:

	M24SR &mDevice;

	/**
	 * max length for a read operation
	 */
	uint16_t mMaxReadBytes;

	/**
	 * max length for a write operation
	 */
	uint16_t mMaxWriteBytes;
};

#endif /* TYPE4NFCTAGSTM24SR_H_ */
