/**
  ******************************************************************************
  * @file       Sample_lockWriteTag.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo define a lock/unlock the tag content when the user press
  *  the user button
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */

#include "mbed.h"

#include "X_NUCLEO_NFC01A1.h"

static volatile bool buttonPress=false; /// true when the user press the message

/**
 * call back called when the user press the button
 */
static void setButtonPress(){
    buttonPress=true;
}//if buttonPress

/**
 * return the led that tell if the tag is read only
 * @param nfcNucleo board where the lead lives
 * @return let to switch on if the tag is read only
 */
static inline DigitalOut& getReadOnlyLed(X_NUCLEO_NFC01A1 *nfcNucleo){
	return nfcNucleo->getLed2();
}

/**
 * return the led that tell if the tag is write only
 * @param nfcNucleo board where the lead lives
 * @return let to switch on if the tag is write only
 */
static inline DigitalOut& getWriteableLed(X_NUCLEO_NFC01A1 *nfcNucleo){
	return nfcNucleo->getLed3();
}

/**
 * set the tag as read only and switch the led on
 * @param nfcNucleo board where the lead lives
 * @return true if the change has effect
 */
bool enableReadOnly( X_NUCLEO_NFC01A1 *nfcNucleo){

	if(nfcNucleo->getM24SR().enableReadOnly()==NFC_SUCCESS){
		getReadOnlyLed(nfcNucleo)=true;
		getWriteableLed(nfcNucleo)=false;
		return true;
	}else
		return false;
}

/**
 * set the tag as read/write and switch the led on
 * @param nfcNucleo board where the lead lives
 * @return true if the change has effect
 */
bool disableReadOnly( X_NUCLEO_NFC01A1 *nfcNucleo){

	if(nfcNucleo->getM24SR().disableReadOnly()==NFC_SUCCESS){
		getReadOnlyLed(nfcNucleo)=false;
		getWriteableLed(nfcNucleo)=true;
		return true;
	}else
		return false;
}

/**
 * When the user press the button enable/disable the tag write protetion
 */
void sample_lockTagContent() {
	Serial console(SERIAL_TX, SERIAL_RX);
	
    //instance the board with the default paramiters
    I2C i2cChannel(X_NUCLEO_NFC01A1::DEFAULT_SDA_PIN,X_NUCLEO_NFC01A1::DEFAULT_SDL_PIN);
    X_NUCLEO_NFC01A1 *nfcNucleo = X_NUCLEO_NFC01A1::Instance(i2cChannel);
	
	//get the wrapper for use the NdefLib
	NDefLib::NDefNfcTag& tag = nfcNucleo->getM24SR().getNDefTag();

    //when the user press the button set buttonPress to true
    InterruptIn userButton(USER_BUTTON);
    userButton.fall(setButtonPress);
    
    //restore the board as read/write, set the led1 if it is ok
    tag.openSession(true);
    nfcNucleo->getLed1()= (nfcNucleo->getM24SR().disableAllPassword()==NFC_SUCCESS);
    tag.closeSession();
    
	// now we are in read/write -> switch on the writable led    
    getReadOnlyLed(nfcNucleo)=false;
    getWriteableLed(nfcNucleo)=true;
    bool tagIsLock=false;

    while(true){

        if(buttonPress){
        	if(tag.openSession()){
        		bool changeStatusResult;
				//switch the status
				if(tagIsLock)
					changeStatusResult=enableReadOnly(nfcNucleo);
				else
					changeStatusResult=disableReadOnly(nfcNucleo);

				if(changeStatusResult)
					tagIsLock=!tagIsLock;
				else
					console.printf("Error change the tag status");
				tag.closeSession();
        	}else{
        		console.printf("Error opening the session, tag is in use");
        	}
        	buttonPress=false;
        }//if

        //wait next event
        __WFE();
    }//while

}//sample_countClick





