/**
  ******************************************************************************
  * @file       Sample_writeAndChangeAll.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo write an ndef message different records, when the user press the buttun
  *             read the tag, change some data and write it again
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "NDefLib/NDefNfcTag.h"

#include "NDefLib/RecordType/RecordAAR.h"
#include "NDefLib/RecordType/RecordSMS.h"
#include "NDefLib/RecordType/RecordGeo.h"
#include "NDefLib/RecordType/RecordURI.h"
#include "NDefLib/RecordType/RecordMail.h"
#include "NDefLib/RecordType/RecordText.h"
#include "NDefLib/RecordType/RecordMimeType.h"
#include "NDefLib/RecordType/RecordVCard.h"

#include "X_NUCLEO_NFC01A1.h"

static Serial pc(SERIAL_TX, SERIAL_RX); ///serial console


/**
 * shift the led status between the 3 leds
 */
static void shiftLed(DigitalOut &led1,DigitalOut &led2,DigitalOut &led3){
    const uint8_t prevLed1=led1;
    const uint8_t prevLed2=led2;
    const uint8_t prevLed3=led3;
    led1=prevLed3;
    led2=prevLed1;
    led3=prevLed2;
}

/**
 * Create a message that contains all the possible records, and write it in the tag
 * @param tag nfc tag where write the message
 */
static void setNFCTag(NDefLib::NDefNfcTag &tag){

    bool writeStatus,closeStatus;
    if(tag.openSession()){
        NDefLib::Message msg;

        NDefLib::RecordAAR rAAR("com.st.BlueMS");
        msg.addRecord(&rAAR);

        NDefLib::RecordSMS rSMS("123456789","st.com.BlueMS");
        msg.addRecord(&rSMS);

        NDefLib::RecordGeo rGeo(123.123,-456.789);
        msg.addRecord(&rGeo);

        NDefLib::RecordURI rUri(NDefLib::RecordURI::HTTP_WWW,"http://www.st.com");
        msg.addRecord(&rUri);

        NDefLib::RecordMail rMail("mail@st.com","ciao","da nfc tag");
        msg.addRecord(&rMail);

        NDefLib::RecordMimeType rText1("text/plain",(const uint8_t*)"ciao",4);
        msg.addRecord(&rText1);

        NDefLib::RecordText rText3(NDefLib::RecordText::UTF8,"it","ciao");
        msg.addRecord(&rText3);

        NDefLib::RecordVCard::VCardInfo_t cardInfo;
        cardInfo[NDefLib::RecordVCard::FORMATTED_NAME]="prova prova1";
        cardInfo[NDefLib::RecordVCard::ADDRESS_HOME]=";;1 Main St.;Springfield;IL;12345;USA";
        cardInfo[NDefLib::RecordVCard::ADDRESS_WORK]=";;2 Main St.;Springfield;IL;12345;USA";
        cardInfo[NDefLib::RecordVCard::EMAIL_WORK]="workmail@st.com";
        cardInfo[NDefLib::RecordVCard::EMAIL_HOME]="homemail@st.com";
        cardInfo[NDefLib::RecordVCard::GEO]="39.95;-75.1667";
        cardInfo[NDefLib::RecordVCard::IMPP]="aim:johndoe@aol.com";
        cardInfo[NDefLib::RecordVCard::NAME]="prova2;prova3";
        cardInfo[NDefLib::RecordVCard::NICKNAME]="test";
        cardInfo[NDefLib::RecordVCard::NOTE]="A good test";
        cardInfo[NDefLib::RecordVCard::ORGANIZATION]="STM";
        cardInfo[NDefLib::RecordVCard::TEL_HOME]="123";
        cardInfo[NDefLib::RecordVCard::TEL_MOBILE]="456";
        cardInfo[NDefLib::RecordVCard::TEL_WORK]="789";
        cardInfo[NDefLib::RecordVCard::TITLE]="King";
        cardInfo[NDefLib::RecordVCard::URL]="www.st.com";
        cardInfo[NDefLib::RecordVCard::PHOTO_URI]="http://www.st.com/st-web-ui/static/active/en/fragment/multimedia/image/picture/customer_focus.jpg";
        NDefLib::RecordVCard rVCard(cardInfo);
        msg.addRecord(&rVCard);

        writeStatus = tag.write(msg);

        closeStatus = tag.closeSession();

    }else
        pc.printf("Error open Session\n\r");

    if(writeStatus)
        pc.printf("writeOk\n\r");
    else
        pc.printf("writeFail\n\r");

    if(closeStatus)
        pc.printf("closeOk\n\r");
    else
        pc.printf("CloseFail\n\r");
}

/**
 * Print the record content
 * @param r record to print
 */
static void printRecord(NDefLib::Record *const r){
    using namespace NDefLib;
    switch(r->getType()){
        case Record::TYPE_TEXT: {
            const RecordText *const temp = ( RecordText* )r;
            pc.printf("Read Text: %s\r\n",temp->getText().c_str());
            break; }
        case Record::TYPE_AAR:{
            const RecordAAR *const temp = ( RecordAAR* )r;
            pc.printf("Read ARR: %s\r\n",temp->getPackage().c_str());
            break; }
        case Record::TYPE_MIME:{
            const RecordMimeType *const temp = ( RecordMimeType* )r;
            pc.printf("Read mimeType: %s\r\n",temp->getMimeType().c_str());
            pc.printf("Read mimeData: %s\r\n",
                    std::string((const char*)temp->getMimeData(),
                            temp->getMimeDataLenght()).c_str());
            break;}
        case Record::TYPE_URI:{
            RecordURI *const temp = (RecordURI*)r;
            pc.printf("Read uriId: %d\r\n",temp->getUriId());
            pc.printf("Read uriType: %s\r\n",temp->getUriType().c_str());
            pc.printf("Read uriContent: %s\r\n",temp->getContent().c_str());
            break;}
        case Record::TYPE_URI_MAIL:{
            const RecordMail*const temp = (RecordMail*)r;
            pc.printf("Read Dest: %s\r\n",temp->getToAddress().c_str());
            pc.printf("Read Subject: %s\r\n",temp->getSubject().c_str());
            pc.printf("Read Body: %s\r\n",temp->getBody().c_str());
            break;}
        case Record::TYPE_URI_SMS:{
            const RecordSMS*const temp = (RecordSMS*)r;
            pc.printf("Read number: %s\r\n",temp->getNumber().c_str());
            pc.printf("Read message: %s\r\n",temp->getMessagge().c_str());
            break;}
        case Record::TYPE_URI_GEOLOCATION:{
            const RecordGeo*const temp = (RecordGeo*)r;
            pc.printf("Read lat: %f\r\n",temp->getLatitude());
            pc.printf("Read long: %f\r\n",temp->getLongitude());
            break;}
        case Record::TYPE_MIME_VCARD:{
            const RecordVCard *const temp = (RecordVCard*)r;
            pc.printf("Read Name: %s\r\n",(*temp)[RecordVCard::NAME].c_str());
            pc.printf("Read Mail: %s\r\n",(*temp)[RecordVCard::EMAIL_WORK].c_str());
            pc.printf("Read ORG: %s\r\n",(*temp)[RecordVCard::ORGANIZATION].c_str());
            break;}
        case Record::TYPE_UNKNOWN:{
            pc.printf("Unknown record\r\n");
            break;}
    }//switch
}

/**
 * change the record content
 * @param r record to change
 */
static void changeRecord(NDefLib::Record const* r){
    using namespace NDefLib;
    switch(r->getType()){
        case Record::TYPE_TEXT: {
            RecordText *temp = (RecordText*)r;
            temp->setText("CIAOCiao");
            break; }
        case Record::TYPE_AAR:{
            RecordAAR *temp = (RecordAAR*)r;
            temp->setPackage("set Package Ok");
            break; }
        case Record::TYPE_MIME:{
            RecordMimeType *temp = (RecordMimeType*)r;
            temp->copyMimeData((const uint8_t *)"String2",sizeof("String2"));
            break;}
        case Record::TYPE_URI:{
            RecordURI *temp = (RecordURI*)r;
            temp->setContent("google.it");
            break;}
        case Record::TYPE_URI_MAIL:{
            RecordMail *temp = (RecordMail*)r;
            temp->setToAddress("newMail@st.com");
            temp->setSubject("tag change");
            temp->setBody("read/change Works!");
            break;}
        case Record::TYPE_URI_SMS:{
            RecordSMS *temp = (RecordSMS*)r;
            temp->setMessage("Message Change");
            temp->setNumber("0987654321");
            break;}
        case Record::TYPE_URI_GEOLOCATION:{
            RecordGeo *temp = (RecordGeo*)r;
            temp->setLatitude(-temp->getLatitude());
            temp->setLongitude(-temp->getLongitude());
            break;}
        case Record::TYPE_MIME_VCARD:{
            RecordVCard *temp = (RecordVCard*)r;
            (*temp)[RecordVCard::NAME]="name change";
            (*temp)[RecordVCard::NICKNAME]="nic change";
            break;}
        case Record::TYPE_UNKNOWN:{
            pc.printf("Unknown record\r\n");
            break;}
    }//switch
}

/**
 * read the nfc message and print the content on the serial console
 * @param tag nfc tag where read the content
 */
static void readNfcTag(NDefLib::NDefNfcTag &tag){
    using namespace NDefLib;

    if(tag.openSession()){
        NDefLib::Message readMsg;

        tag.read(&readMsg);

        if(readMsg.getNRecords()==0){
            pc.printf("Error Read\r\n");
        }else{
            for(uint32_t i=0;i<readMsg.getNRecords();i++){
                Record *r = readMsg[i];
                printRecord(r);
                delete r;
            }//for
        }//if-else

        tag.closeSession();
    }else{
        pc.printf("Error open read Session\n\r");
    }
}

/**
 * read a nfc message, change the content of each record and write the new message
 * @param tag tag where read and write the nfc message
 */
static void changeNfcTag(NDefLib::NDefNfcTag &tag){
    using NDefLib::Record;
    using NDefLib::Message;

    if(tag.openSession()){
        Message readMsg;

        tag.read(&readMsg);

        if(readMsg.getNRecords()==0){
            pc.printf("Error Read\r\n");
        }else{
            for(uint32_t i=0;i<readMsg.getNRecords();i++){
                Record *r = readMsg[i];
                changeRecord(r);
            }//for
            tag.write(readMsg);
        }//if-else

        tag.closeSession();
    }else{
        pc.printf("Error open SessionChange\n\r");
    }
}

static volatile bool buttonPress=false; /// true when the user press the message

/**
 * call back called when the user press the button
 */
static void setButtonPress(){
    buttonPress=true;
}//if buttonPress

/**
 * write a message and when the user press the button it read the message, change it and update it
 */
void sample_writeAndChangeAll() {
    DigitalOut nucleoLed(LED1);
    
    //instance the board with the default paramiters
    I2C i2cChannel(X_NUCLEO_NFC01A1::DEFAULT_SDA_PIN,X_NUCLEO_NFC01A1::DEFAULT_SDL_PIN);
    X_NUCLEO_NFC01A1 *nfcNucleo = X_NUCLEO_NFC01A1::Instance(i2cChannel);
    
    //get the wrapper for use the NdefLib
    NDefLib::NDefNfcTag& tag = nfcNucleo->getM24SR().getNDefTag();
    
    //switch on the first led    
    nfcNucleo->getLed1()=1;
    nfcNucleo->getLed2()=0;
    nfcNucleo->getLed3()=0;

    //write the message
    setNFCTag(tag);
    //read the message and write it on console
    readNfcTag(tag);

    //enable the button
    InterruptIn mybutton(USER_BUTTON);
    mybutton.fall(setButtonPress);

    //each second change the led status and see if the user press the button
    while(true) {
        wait(1);
        //update the status
        nucleoLed = !nucleoLed;
        shiftLed(nfcNucleo->getLed1(),nfcNucleo->getLed2(),nfcNucleo->getLed3());
        
        if(buttonPress){
            //update the message content
            changeNfcTag(tag);
            //write the new message on console
            readNfcTag(tag);
            buttonPress=false;
        }
    }

}