/**
  ******************************************************************************
  * @file       Sample_countClick.cpp
  * @author     ST / Central Labs
  * @date       03 Dic 2015
  * @brief      This demo define a custom record that contains the number of time the user
  *             press the user button
  ******************************************************************************
  *
  * COPYRIGHT(c) 2015 STMicroelectronics
  *
  * Redistribution and use in source and binary forms, with or without modification,
  * are permitted provided that the following conditions are met:
  *   1. Redistributions of source code must retain the above copyright notice,
  *      this list of conditions and the following disclaimer.
  *   2. Redistributions in binary form must reproduce the above copyright notice,
  *      this list of conditions and the following disclaimer in the documentation
  *      and/or other materials provided with the distribution.
  *   3. Neither the name of STMicroelectronics nor the names of its contributors
  *      may be used to endorse or promote products derived from this software
  *      without specific prior written permission.
  *
  * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
  * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
  * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
  * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
  * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
  * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
  * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
  * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
  * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
  * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
  *
  ******************************************************************************
  */
  
#include "mbed.h"

#include "X_NUCLEO_NFC01A1.h"
#include "NDefLib/NDefNfcTag.h"
#include "NDefLib/RecordType/RecordText.h"

/**
 * castom record that map an integer in a text record
 */
class MyRecord : public NDefLib::RecordText{
    
private:
    char nClickStringBuffer[12]; ///buffer used for print the number
    uint32_t nClick; /// number to show
    
    /**
     * convert the number in char and update the tag content
     */
    void syncTextValue(){
        sprintf(nClickStringBuffer,"%d",nClick);
        setText(nClickStringBuffer);
    }
        
public: 
    
    MyRecord():nClick(0){
        syncTextValue();
    }
    
    /**
     * increment the stored number and update the racord content
     */
    void incrementClick(){
        nClick++;
        syncTextValue();
    }

};

/* write a nfc message into a tag
 * @param nfcNucleo board where write the data
 * @param msg message to write
 */
static void writeMessage(X_NUCLEO_NFC01A1 *nfcNucleo,NDefLib::Message &msg){
    NDefLib::NDefNfcTag& tag = nfcNucleo->getM24SR().getNDefTag();
    //open the i2c session with the nfc chip
    if(tag.openSession()){
        nfcNucleo->getLed1()=! nfcNucleo->getLed1();
        
        //write the tag
        if(tag.write(msg)){
            nfcNucleo->getLed2()=!nfcNucleo->getLed2();
        }//if

        //close the i2c session
        if(tag.closeSession())
            nfcNucleo->getLed3()=!nfcNucleo->getLed3();
    }//if open session
}//writeMessage


static volatile bool buttonPress=false; /// true when the user press the message

/**
 * call back called when the user press the button
 */
static void setButtonPress(){
    buttonPress=true;
}//if buttonPress

/**
 * Write a castum record that count how many times the user press the button
 */
void sample_countClick() {
    
    //instance the board with the default paramiters
    I2C i2cChannel(X_NUCLEO_NFC01A1::DEFAULT_SDA_PIN,X_NUCLEO_NFC01A1::DEFAULT_SDL_PIN);
    X_NUCLEO_NFC01A1 *nfcNucleo = X_NUCLEO_NFC01A1::Instance(i2cChannel);
    
    //get the wrapper for use the NdefLib
    NDefLib::NDefNfcTag& tag = nfcNucleo->getM24SR().getNDefTag();
    
    //set the button interrupt
    InterruptIn userButton(USER_BUTTON);    
    userButton.fall(setButtonPress);
    
    //create the NDef message and record
    NDefLib::Message msg;
    MyRecord rClickCount;
    msg.addRecord(&rClickCount);
    
    writeMessage(nfcNucleo,msg);
    
    while(true){
        
        if(buttonPress){
            //change the record content
            rClickCount.incrementClick();
            //write the new record content
            writeMessage(nfcNucleo,msg);  
            //wait a new button press          
            buttonPress=false;
        }//if
        //wait next event
        __WFE();
    }//while

}//sample_countClick