/**
 ******************************************************************************
 * @file    main.cpp
 * @author  AST / EST
 * @version V0.0.1
 * @date    9-August-2016
 * @brief   Simple Example application for using the X_NUCLEO_IKS01A2 
 *          MEMS Inertial & Environmental Sensor Nucleo expansion board.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2016 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 *  SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
*/ 

/* Includes */
#include "mbed.h"
#include "x_nucleo_iks01a2.h"

/* Instantiate the expansion board */
static X_NUCLEO_IKS01A2 *mems_expansion_board = X_NUCLEO_IKS01A2::Instance(D14, D15);

/* Retrieve the composing elements of the expansion board */
static LSM6DSLSensor *acc_gyro = mems_expansion_board->acc_gyro;

InterruptIn mybutton(USER_BUTTON);

volatile int mems_event = 0;
volatile int toggle_hw_event_enable = 0;
static int hw_event_is_enabled = 1;
uint16_t step_count = 0;

void pressed_cb();
void INT1_cb();
void INT2_cb();
void sendOrientation();

/* Simple main function */
int main() {
  /* Attach callback to User button press */
  mybutton.fall(&pressed_cb);
  /* Attach callback to LSM6DSL INT1 */
  acc_gyro->AttachINT1IRQ(&INT1_cb);
  /* Attach callback to LSM6DSL INT2 */
  acc_gyro->AttachINT2IRQ(&INT2_cb);
  
  /* Enable LSM6DSL accelerometer */
  acc_gyro->Enable_X();
  /* Enable HW events. */
  acc_gyro->Enable_Pedometer();
  acc_gyro->Enable_Tilt_Detection();
  acc_gyro->Enable_Free_Fall_Detection();
  acc_gyro->Enable_Single_Tap_Detection();
  acc_gyro->Enable_Double_Tap_Detection();
  acc_gyro->Enable_6D_Orientation();
  acc_gyro->Enable_Wake_Up_Detection();
  
  printf("\r\n--- Starting new run ---\r\n");
 
  while(1) {
    if (mems_event) {
      mems_event = 0;
      LSM6DSL_Event_Status_t status;
      acc_gyro->Get_Event_Status(&status);
      if (status.StepStatus) {
        /* New step detected, so print the step counter */
        acc_gyro->Get_Step_Counter(&step_count);
        printf("Step counter: %d\r\n", step_count);
      }

      if (status.FreeFallStatus) {
        /* Output data. */
        printf("Free Fall Detected!\r\n");
      }

      if (status.TapStatus) {
        /* Output data. */
        printf("Single Tap Detected!\r\n");
      }

      if (status.DoubleTapStatus) {
        /* Output data. */
        printf("Double Tap Detected!\r\n");
      }

      if (status.D6DOrientationStatus) {
        /* Send 6D Orientation */
        sendOrientation();
      }

      if (status.TiltStatus) {
        /* Output data. */
        printf("Tilt Detected!\r\n");
      }

      if (status.WakeUpStatus) {
        /* Output data. */
        printf("Wake Up Detected!\r\n");
      }
    }

    if(toggle_hw_event_enable) {
      toggle_hw_event_enable = 0;
      if(hw_event_is_enabled == 0) {
        /* Enable HW events. */
        acc_gyro->Enable_Pedometer();
        acc_gyro->Enable_Tilt_Detection();
        acc_gyro->Enable_Free_Fall_Detection();
        acc_gyro->Enable_Single_Tap_Detection();
        acc_gyro->Enable_Double_Tap_Detection();
        acc_gyro->Enable_6D_Orientation();
        acc_gyro->Enable_Wake_Up_Detection();
        hw_event_is_enabled = 1;
      } else {
        acc_gyro->Disable_Pedometer();
        acc_gyro->Disable_Tilt_Detection();
        acc_gyro->Disable_Free_Fall_Detection();
        acc_gyro->Disable_Single_Tap_Detection();
        acc_gyro->Disable_Double_Tap_Detection();
        acc_gyro->Disable_6D_Orientation();
        acc_gyro->Disable_Wake_Up_Detection();
        hw_event_is_enabled = 0;
      }
    }
  }
}

void pressed_cb() {
  toggle_hw_event_enable = 1;
}

void INT1_cb() {
  mems_event = 1;
}

void INT2_cb() {
  mems_event = 1;
}

void sendOrientation() {
  uint8_t xl = 0;
  uint8_t xh = 0;
  uint8_t yl = 0;
  uint8_t yh = 0;
  uint8_t zl = 0;
  uint8_t zh = 0;
  
  acc_gyro->Get_6D_Orientation_XL(&xl);
  acc_gyro->Get_6D_Orientation_XH(&xh);
  acc_gyro->Get_6D_Orientation_YL(&yl);
  acc_gyro->Get_6D_Orientation_YH(&yh);
  acc_gyro->Get_6D_Orientation_ZL(&zl);
  acc_gyro->Get_6D_Orientation_ZH(&zh);
  
  if ( xl == 0 && yl == 0 && zl == 0 && xh == 0 && yh == 1 && zh == 0 )
  {
    printf( "\r\n  ________________  " \
            "\r\n |                | " \
            "\r\n |  *             | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |________________| \r\n" );
  }
  
  else if ( xl == 1 && yl == 0 && zl == 0 && xh == 0 && yh == 0 && zh == 0 )
  {
    printf( "\r\n  ________________  " \
            "\r\n |                | " \
            "\r\n |             *  | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |________________| \r\n" );
  }
  
  else if ( xl == 0 && yl == 0 && zl == 0 && xh == 1 && yh == 0 && zh == 0 )
  {
    printf( "\r\n  ________________  " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |  *             | " \
            "\r\n |________________| \r\n" );
  }
  
  else if ( xl == 0 && yl == 1 && zl == 0 && xh == 0 && yh == 0 && zh == 0 )
  {
    printf( "\r\n  ________________  " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |                | " \
            "\r\n |             *  | " \
            "\r\n |________________| \r\n" );
  }
  
  else if ( xl == 0 && yl == 0 && zl == 0 && xh == 0 && yh == 0 && zh == 1 )
  {
    printf( "\r\n  __*_____________  " \
            "\r\n |________________| \r\n" );
  }
  
  else if ( xl == 0 && yl == 0 && zl == 1 && xh == 0 && yh == 0 && zh == 0 )
  {
    printf( "\r\n  ________________  " \
            "\r\n |________________| " \
            "\r\n    *               \r\n" );
  }
  
  else
  {
    printf( "None of the 6D orientation axes is set in LSM6DSL - accelerometer.\r\n" );
  }
}
