/**
 ******************************************************************************
 * @file    main.cpp
 * @author  Davide Aliprandi / AST
 * @version V1.0.0
 * @date    October 14th, 2015
 * @brief   mbed test application for the STMicrolectronics X-NUCLEO-IHM01A1
 *          Motor Control Expansion Board: control of 1 motor.
 *          This application makes use of a C++ component architecture obtained
 *          from the C component architecture through the Stm32CubeTOO tool.
 ******************************************************************************
 * @attention
 *
 * <h2><center>&copy; COPYRIGHT(c) 2015 STMicroelectronics</center></h2>
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *   1. Redistributions of source code must retain the above copyright notice,
 *      this list of conditions and the following disclaimer.
 *   2. Redistributions in binary form must reproduce the above copyright notice,
 *      this list of conditions and the following disclaimer in the documentation
 *      and/or other materials provided with the distribution.
 *   3. Neither the name of STMicroelectronics nor the names of its contributors
 *      may be used to endorse or promote products derived from this software
 *      without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
 * SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
 * CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
 * OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 ******************************************************************************
 */


/* Includes ------------------------------------------------------------------*/

/* mbed specific header files. */
#include "mbed.h"

/* Helper header files. */
#include "DevSPI.h"

/* Component specific header files. */
#include "l6474_class.h"


/* Variables -----------------------------------------------------------------*/

/* Motor Control Component. */
L6474 *l6474;

/* Flag to identify whenever a PWM pulse has finished. */
volatile int pwm_pulse_finished_flag;

/* Flag to identify whenever the desired delay has expired. */
volatile int delay_expired_flag;


/* Functions -----------------------------------------------------------------*/

/*
 * @brief  PWM callback.
 * @param  None
 * @retval None
 */
void PWMCallback(void)
{
    pwm_pulse_finished_flag = 1;
}

/*
 * @brief  Delay callback.
 * @param  None
 * @retval None
 */
void DelayCallback()
{
    delay_expired_flag = 1;
}

/*
 * @brief  Waiting until PWM pulse has finished.
 * @param  None
 * @retval None
 */
void WaitForPWMPulse(void)
{
    /* Waiting until PWM flag is set. */
    while (pwm_pulse_finished_flag == 0);
    
    /* Resetting PWM flag. */
    pwm_pulse_finished_flag = 0;

    /* Setting the device state machine. */
    if (l6474->GetDeviceState() != INACTIVE)
        l6474->StepClockHandler();
}

/*
 * @brief  Waiting while the motor is active.
 * @param  None
 * @retval None
 */
void WaitWhileActive(void)
{
    while (l6474->GetDeviceState() != INACTIVE)
        WaitForPWMPulse();
}

/*
 * @brief  Waiting until delay has expired.
 * @param  delay delay in milliseconds.
 * @retval None
 */
void WaitForDelay(int delay)
{
    Timeout timeout;
    timeout.attach(&DelayCallback, delay / 1E3);

    delay_expired_flag = 0;
    while (delay_expired_flag == 0)
        WaitForPWMPulse();
}


/* Main ----------------------------------------------------------------------*/

int main()
{
    /* Initializing SPI bus. */
    DevSPI dev_spi(D11, D12, D13);

    /* Resetting Timer PWM flag. */
    pwm_pulse_finished_flag = 0;

    /* Initializing Motor Control Component. */
    l6474 = new L6474(D8, D7, D9, D10, dev_spi);
    if (l6474->Init(NULL) != COMPONENT_OK)
        return false;

    /* Printing to the console. */
    printf("Motor Control Application Example for 1 Motor\r\n\n");

    /* Main Loop. */
    while(true)
    {
        /*----- Moving forward 16000 steps. -----*/

        /* Printing to the console. */
        printf("--> Moving forward 16000 steps.\r\n");

        /* Moving 16000 steps in the forward direction. */
        l6474->Move(FORWARD, 16000);
        
        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Waiting 2 seconds. */
        wait_ms(2000);

        
        /*----- Moving backward 16000 steps. -----*/
        
        /* Printing to the console. */
        printf("--> Moving backward 16000 steps.\r\n");
        
        /* Moving 16000 steps in the backward direction. */
        l6474->Move(BACKWARD, 16000);
        
        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Setting the current position to be the home position. */
        l6474->SetHome();
  
        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Going to a specified position. -----*/

        /* Printing to the console. */
        printf("--> Going to position -6400.\r\n");
        
        /* Requesting to go to a specified position. */
        l6474->GoTo(-6400);
        
        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Getting current position. */
        int position = l6474->GetPosition();
        
        /* Printing to the console. */
        printf("    Position: %d.\r\n", position);
        
        /* Setting the current position to be the mark position. */
        l6474->SetMark();

        /* Waiting 2 seconds. */
        wait_ms(2000);

        
        /*----- Going Home. -----*/

        /* Printing to the console. */
        printf("--> Going Home.\r\n");
        
        /* Requesting to go to home */
        l6474->GoHome();  
        
        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Getting current position. */
        position = l6474->GetPosition();

        /* Printing to the console. */
        printf("    Position: %d.\r\n", position);

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Going to a specified position. -----*/

        /* Printing to the console. */
        printf("--> Going to position 6400.\r\n");
        
        /* Requesting to go to a specified position. */
        l6474->GoTo(6400);
        
        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Getting current position. */
        position = l6474->GetPosition();

        /* Printing to the console. */
        printf("    Position: %d.\r\n", position);

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Going to mark which was set previously after going to -6400. -----*/

        /* Printing to the console. */
        printf("--> Going to mark which was set previously after going to -6400.\r\n");
        
        /* Requesting to go to mark position. */
        l6474->GoMark();  
        
        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Getting current position. */
        position = l6474->GetPosition();

        /* Printing to the console. */
        printf("    Position: %d.\r\n", position);

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Moving backward. -----*/

        /* Printing to the console. */
        printf("--> Moving backward.\r\n");

        /* Requesting to run backward. */
        l6474->Run(BACKWARD);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Getting current speed. */
        int speed = l6474->GetCurrentSpeed();

        /* Printing to the console. */
        printf("    Speed: %d.\r\n", speed);


        /*----- Increasing the speed while running. -----*/

        /* Printing to the console. */
        printf("--> Increasing the speed while running.\r\n");

        /* Increasing speed to 2400 step/s. */
        l6474->SetMaxSpeed(2400);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Getting current speed. */
        speed = l6474->GetCurrentSpeed();

        /* Printing to the console. */
        printf("    Speed: %d.\r\n", speed);


        /*----- Decreasing the speed while running. -----*/

        /* Printing to the console. */
        printf("--> Decreasing the speed while running.\r\n");

        /* Decreasing speed to 1200 step/s. */
        l6474->SetMaxSpeed(1200);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Getting current speed. */
        speed = l6474->GetCurrentSpeed();

        /* Printing to the console. */
        printf("    Speed: %d.\r\n", speed);


        /*----- Increasing acceleration while running. -----*/

        /* Printing to the console. */
        printf("--> Increasing acceleration while running.\r\n");

        /* Increasing acceleration to 480 step/s^2. */
        l6474->SetAcceleration(480);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Increasing speed to 2400 step/s. */
        l6474->SetMaxSpeed(2400);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Getting current speed. */
        speed = l6474->GetCurrentSpeed();

        /* Printing to the console. */
        printf("    Speed: %d.\r\n", speed);


        /*----- Increasing deceleration while running. -----*/

        /* Printing to the console. */
        printf("--> Increasing deceleration while running.\r\n");

        /* Increasing deceleration to 480 step/s^2. */
        l6474->SetDeceleration(480);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Decreasing speed to 1200 step/s. */
        l6474->SetMaxSpeed(1200);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);

        /* Getting current speed. */
        speed = l6474->GetCurrentSpeed();

        /* Printing to the console. */
        printf("    Speed: %d.\r\n", speed);


        /*----- Requiring soft-stop while running. -----*/

        /* Printing to the console. */
        printf("--> Requiring soft-stop while running.\r\n");

        /* Requesting soft stop. */
        l6474->SoftStop();

        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- Requiring hard-stop while running. -----*/

        /* Printing to the console. */
        printf("--> Requiring hard-stop while running.\r\n");

        /* Requesting to run in forward direction. */
        l6474->Run(FORWARD);

        /* Waiting until delay has expired. */
        WaitForDelay(5000);
        
        /* Requesting to immediatly stop. */
        l6474->HardStop();

        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Waiting 2 seconds. */
        wait_ms(2000);


        /*----- GOTO stopped by soft-stop. -----*/

        /* Printing to the console. */
        printf("--> GOTO stopped by soft-stop.\r\n");

        /* Requesting to go to a specified position. */
        l6474->GoTo(20000);  
        
        /* Waiting while the motor is active. */
        WaitForDelay(5000);

        /* Requesting to perform a soft stop */
        l6474->SoftStop();

        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Waiting 2 seconds. */
        wait_ms(2000);  


        /*----- Reading inexistent register to test "MyFlagInterruptHandler". -----*/

        /* Printing to the console. */
        printf("--> Reading inexistent register to test \"MyFlagInterruptHandler\".\r\n");

        /*
         * Trying to read an inexistent register.
         * The flag interrupt should be raised and the "MyFlagInterruptHandler"
         * function called.
         */
        l6474->CmdGetParam(0x1F);

        /* Waiting 0.5 seconds. */
        wait_ms(500);


        /*----- Changing step mode to full step mode. -----*/

        /* Printing to the console. */
        printf("--> Changing step mode to full step mode.\r\n");

        /* Selecting full step mode. */
        l6474->SelectStepMode(STEP_MODE_FULL);

        /* Setting speed and acceleration to be consistent with full step mode. */
        l6474->SetMaxSpeed(100);
        l6474->SetMinSpeed(50);
        l6474->SetAcceleration(10);
        l6474->SetDeceleration(10);

        /* Requesting to go to a specified position. */
        l6474->GoTo(200);

        /* Waiting while the motor is active. */
        WaitWhileActive();

        /* Getting current position */
        position = l6474->GetPosition();

        /* Printing to the console. */
        printf("    Position: %d.\r\n", position);

        /* Waiting 2 seconds. */
        wait_ms(2000);
        

        /*----- Restoring 1/16 microstepping mode. -----*/

        /* Printing to the console. */
        printf("--> Restoring 1/16 microstepping mode.\r\n");

        /* Resetting to 1/16 microstepping mode */
        l6474->SelectStepMode(STEP_MODE_1_16);    

        /* Update speed, acceleration, deceleration for 1/16 microstepping mode*/
        l6474->SetMaxSpeed(1600);
        l6474->SetMinSpeed(800);
        l6474->SetAcceleration(160);
        l6474->SetDeceleration(160);  


        /*----- Infinite Loop. -----*/

        /* Printing to the console. */
        printf("--> Infinite Loop...\r\n");

        /* Infinite Loop. */
        while(1)
        {
            /* Requesting to go to a specified position. */
            l6474->GoTo(-6400);

            /* Waiting while the motor is active. */
            WaitWhileActive();

            /* Requesting to go to a specified position. */
            l6474->GoTo(6400);

            /* Waiting while the motor is active. */
            WaitWhileActive();
        }
    }
}
